import { expect } from 'vitest';
import type { CoreMessage } from '../llm';
import type { MastraMessageV1 } from '../memory';
import { MockMemory } from '../memory/mock';
import type { MastraDBMessage } from './message-list';
import { MessageList } from './message-list';

export { MockMemory };

const toolArgs = {
  weather: { location: 'New York' },
  calculator: { expression: '2+2' },
  search: { query: 'latest AI developments' },
};

const toolResults = {
  weather: 'Pretty hot',
  calculator: '4',
  search: 'Anthropic blah blah blah',
};

/**
 * Creates a simulated conversation history with alternating messages and occasional tool calls
 * @param threadId Thread ID for the messages
 * @param messageCount Number of turn pairs (user + assistant) to generate
 * @param toolFrequency How often to include tool calls (e.g., 3 means every 3rd assistant message)
 * @returns Array of messages representing the conversation
 */
export function generateConversationHistory({
  threadId,
  resourceId = 'test-resource',
  messageCount = 5,
  toolFrequency = 3,
  toolNames = ['weather', 'calculator', 'search'],
}: {
  threadId: string;
  resourceId?: string;
  messageCount?: number;
  toolFrequency?: number;
  toolNames?: (keyof typeof toolArgs)[];
}): {
  messages: MastraMessageV1[];
  messagesV2: MastraDBMessage[];
  fakeCore: CoreMessage[];
  counts: { messages: number; toolCalls: number; toolResults: number };
} {
  const counts = { messages: 0, toolCalls: 0, toolResults: 0 };
  // Create some words that will each be about one token
  const words = ['apple', 'banana', 'orange', 'grape'];
  // Arguments for different tools

  const messages: MastraDBMessage[] = [];
  const startTime = Date.now();

  // Generate message pairs (user message followed by assistant response)
  for (let i = 0; i < messageCount; i++) {
    // Create user message content
    const userContent = Array(25).fill(words).flat().join(' '); // ~100 tokens

    // Add user message
    messages.push({
      role: 'user',
      content: { format: 2, parts: [{ type: 'text', text: userContent }] },
      id: `message-${i * 2}`,
      threadId,
      resourceId,
      createdAt: new Date(startTime + i * 2000), // Each pair 2 seconds apart
    });
    counts.messages++;

    // Determine if this assistant message should include a tool call
    const includeTool = i > 0 && i % toolFrequency === 0;
    const toolIndex = includeTool ? (i / toolFrequency) % toolNames.length : -1;
    const toolName = includeTool ? toolNames[toolIndex] : '';

    // Create assistant message
    if (includeTool && toolName) {
      // Assistant message with tool result only (matching old behavior)
      messages.push({
        role: 'assistant',
        content: {
          format: 2,
          parts: [
            {
              type: 'tool-invocation',
              toolInvocation: {
                state: 'result',
                toolCallId: `tool-${i}`,
                toolName,
                args: toolArgs[toolName as keyof typeof toolArgs] || {},
                result: toolResults[toolName as keyof typeof toolResults] || {},
              },
            },
          ],
        },
        id: `tool-call-${i * 2 + 1}`,
        threadId,
        resourceId,
        createdAt: new Date(startTime + i * 2000 + 1000), // 1 second after user message
      });
      counts.messages++;
      counts.toolCalls++;
      counts.toolResults++;
    } else {
      // Regular assistant text message
      messages.push({
        role: 'assistant',
        content: { format: 2, parts: [{ type: 'text', text: Array(15).fill(words).flat().join(' ') }] }, // ~60 tokens
        id: `message-${i * 2 + 1}`,
        threadId,
        resourceId,
        createdAt: new Date(startTime + i * 2000 + 1000), // 1 second after user message
      });
      counts.messages++;
    }
  }

  const latestMessage = messages.at(-1)!;
  if (latestMessage.role === `assistant` && latestMessage.content.parts.at(-1)?.type === `tool-invocation`) {
    const userContent = Array(25).fill(words).flat().join(' '); // ~100 tokens
    messages.push({
      role: 'user',
      content: { format: 2, parts: [{ type: 'text', text: userContent }] },
      id: `message-${messages.length + 1 * 2}`,
      threadId,
      resourceId,
      createdAt: new Date(startTime + messages.length + 1 * 2000), // Each pair 2 seconds apart
    });
    counts.messages++;
  }

  const list = new MessageList().add(messages, 'memory');
  return {
    fakeCore: list.get.all.v1() as CoreMessage[],
    messages: list.get.all.v1(),
    messagesV2: list.get.all.db(),
    counts,
  };
}

export function assertNoDuplicateParts(parts: any[]) {
  // Check for duplicate tool-invocation results by toolCallId
  const seenToolResults = new Set();
  for (const part of parts) {
    if (part.type === 'tool-invocation' && part.toolInvocation.state === 'result') {
      const key = `${part.toolInvocation.toolCallId}|${JSON.stringify(part.toolInvocation.result)}`;
      expect(seenToolResults.has(key)).toBe(false);
      seenToolResults.add(key);
    }
  }

  // Check for duplicate text parts
  const seenTexts = new Set();
  for (const part of parts) {
    if (part.type === 'text') {
      expect(seenTexts.has(part.text)).toBe(false);
      seenTexts.add(part.text);
    }
  }
}
