import { readFile } from 'node:fs/promises';
import { parse } from 'dotenv';

import { MastraBase } from '../base';

export interface IBundler {
  loadEnvVars(): Promise<Map<string, string>>;
  getEnvFiles(): Promise<string[]>;
  getAllToolPaths(mastraDir: string, toolsPaths: (string | string[])[]): (string | string[])[];
  bundle(
    entryFile: string,
    outputDirectory: string,
    options: { toolsPaths: (string | string[])[]; projectRoot: string },
  ): Promise<void>;
  prepare(outputDirectory: string): Promise<void>;
  writePackageJson(outputDirectory: string, dependencies: Map<string, string>): Promise<void>;
  lint(entryFile: string, outputDirectory: string, toolsPaths: (string | string[])[]): Promise<void>;
}

export abstract class MastraBundler extends MastraBase implements IBundler {
  constructor({ name, component = 'BUNDLER' }: { name: string; component?: 'BUNDLER' | 'DEPLOYER' }) {
    super({ component, name });
  }

  async loadEnvVars(): Promise<Map<string, string>> {
    const envVars = new Map();

    for (const file of await this.getEnvFiles()) {
      const content = await readFile(file, 'utf-8');
      const config = parse(content);

      Object.entries(config).forEach(([key, value]) => {
        envVars.set(key, value);
      });
    }

    return envVars;
  }

  abstract getAllToolPaths(mastraDir: string, toolsPaths: (string | string[])[]): (string | string[])[];
  abstract prepare(outputDirectory: string): Promise<void>;
  abstract writePackageJson(outputDirectory: string, dependencies: Map<string, string>): Promise<void>;
  abstract getEnvFiles(): Promise<string[]>;
  abstract bundle(
    entryFile: string,
    outputDirectory: string,
    { toolsPaths, projectRoot }: { toolsPaths: (string | string[])[]; projectRoot: string },
  ): Promise<void>;
  abstract lint(entryFile: string, outputDirectory: string, toolsPaths: (string | string[])[]): Promise<void>;
}
