import type { Handler, MiddlewareHandler, HonoRequest, Context } from 'hono';
import type { cors } from 'hono/cors';
import type { DescribeRouteOptions } from 'hono-openapi';
import type { Mastra } from '../mastra';
import type { RequestContext } from '../request-context';
import type { MastraAuthProvider } from './auth';

export type Methods = 'GET' | 'POST' | 'PUT' | 'DELETE' | 'PATCH' | 'ALL';

export type ApiRoute =
  | {
      path: string;
      method: Methods;
      handler: Handler;
      middleware?: MiddlewareHandler | MiddlewareHandler[];
      openapi?: DescribeRouteOptions;
      requiresAuth?: boolean;
    }
  | {
      path: string;
      method: Methods;
      createHandler: ({ mastra }: { mastra: Mastra }) => Promise<Handler>;
      middleware?: MiddlewareHandler | MiddlewareHandler[];
      openapi?: DescribeRouteOptions;
      requiresAuth?: boolean;
    };

export type Middleware = MiddlewareHandler | { path: string; handler: MiddlewareHandler };

export type ContextWithMastra = Context<{
  Variables: {
    mastra: Mastra;
    requestContext: RequestContext;
    customRouteAuthConfig?: Map<string, boolean>;
  };
}>;

export type MastraAuthConfig<TUser = unknown> = {
  /**
   * Protected paths for the server
   */
  protected?: (RegExp | string | [string, Methods | Methods[]])[];

  /**
   * Public paths for the server
   */
  public?: (RegExp | string | [string, Methods | Methods[]])[];

  /**
   * Public paths for the server
   */
  authenticateToken?: (token: string, request: HonoRequest) => Promise<TUser>;

  /**
   * Authorization function for the server
   */
  authorize?: (path: string, method: string, user: TUser, context: ContextWithMastra) => Promise<boolean>;

  /**
   * Rules for the server
   */
  rules?: {
    /**
     * Path for the rule
     */
    path?: RegExp | string | string[];
    /**
     * Method for the rule
     */
    methods?: Methods | Methods[];
    /**
     * Condition for the rule
     */
    condition?: (user: TUser) => Promise<boolean> | boolean;
    /**
     * Allow the rule
     */
    allow?: boolean;
  }[];
};

export type ServerConfig = {
  /**
   * Port for the server
   * @default 4111
   */
  port?: number;
  /**
   * Host for the server
   * @default 'localhost'
   */
  host?: string;
  /**
   * Base path for Mastra Studio
   * @default '/'
   * @example '/my-mastra-studio'
   */
  studioBase?: string;
  /**
   * Timeout for the server
   */
  timeout?: number;
  /**
   * Custom API routes for the server
   */
  apiRoutes?: ApiRoute[];
  /**
   * Middleware for the server
   */
  middleware?: Middleware | Middleware[];
  /**
   * CORS configuration for the server
   * @default { origin: '*', allowMethods: ['GET', 'POST', 'PUT', 'PATCH', 'DELETE', 'OPTIONS'], allowHeaders: ['Content-Type', 'Authorization', 'x-mastra-client-type'], exposeHeaders: ['Content-Length', 'X-Requested-With'], credentials: false }
   */
  cors?: Parameters<typeof cors>[0] | false;
  /**
   * Build configuration for the server
   */
  build?: {
    /**
     * Enable Swagger UI
     * @default false
     */
    swaggerUI?: boolean;
    /**
     * Enable API request logging
     * @default false
     */
    apiReqLogs?: boolean;
    /**
     * Enable OpenAPI documentation
     * @default false
     */
    openAPIDocs?: boolean;
  };
  /**
   * Body size limit for the server
   * @default 4_718_592 bytes (4.5 MB)
   */
  bodySizeLimit?: number;

  /**
   * Authentication configuration for the server
   */
  auth?: MastraAuthConfig<any> | MastraAuthProvider<any>;

  /**
   * If you want to run `mastra dev` with HTTPS, you can run it with the `--https` flag and provide the key and cert files here.
   */
  https?: {
    key: Buffer;
    cert: Buffer;
  };

  /**
   * Custom error handler for the server. This hook is called when an unhandled error occurs.
   * Use this to customize error responses, log errors to external services (e.g., Sentry),
   * or implement custom error formatting.
   *
   * @param err - The error that was thrown
   * @param c - The Hono context object, providing access to request details and response methods
   * @returns A Response object or a Promise that resolves to a Response
   *
   * @example
   * ```ts
   * const mastra = new Mastra({
   *   server: {
   *     onError: (err, c) => {
   *       // Log to Sentry
   *       Sentry.captureException(err);
   *
   *       // Return custom formatted response
   *       return c.json({
   *         error: err.message,
   *         timestamp: new Date().toISOString(),
   *       }, 500);
   *     },
   *   },
   * });
   * ```
   */
  onError?: (err: Error, c: Context) => Response | Promise<Response>;
};
