import { resolveSerializedZodOutput } from '@/lib/form/utils';
import type { JsonSchema } from '@mastra/schema-compat/json-to-zod';
import { jsonSchemaToZod } from '@mastra/schema-compat/json-to-zod';
import { z } from 'zod';
import { Txt } from '@/ds/components/Txt';
import { Skeleton } from '@/ds/components/Skeleton';
import ToolExecutor from '@/domains/tools/components/ToolExecutor';
import { useExecuteMCPTool, useMCPServerTool } from '@/domains/mcps/hooks/use-mcp-server-tool';
import { toast } from '@/lib/toast';
import { useEffect } from 'react';

export interface MCPToolPanelProps {
  toolId: string;
  serverId: string;
}

export const MCPToolPanel = ({ toolId, serverId }: MCPToolPanelProps) => {
  const { data: tool, isLoading, error } = useMCPServerTool(serverId, toolId);
  const { mutateAsync: executeTool, isPending: isExecuting, data: result } = useExecuteMCPTool(serverId, toolId);

  useEffect(() => {
    if (error) {
      const errorMessage = error instanceof Error ? error.message : 'Failed to load tool';
      toast.error(`Error loading tool: ${errorMessage}`);
    }
  }, [error]);

  const handleExecuteTool = async (data: any) => {
    if (!tool) return;

    return await executeTool(data);
  };

  if (isLoading) {
    return (
      <div className="p-6">
        <Skeleton className="h-8 w-48 mb-4" />
        <Skeleton className="h-32 w-full" />
      </div>
    );
  }

  if (error) return null;

  if (!tool)
    return (
      <div className="py-12 text-center px-6">
        <Txt variant="header-md" className="text-icon3">
          Tool not found
        </Txt>
      </div>
    );

  let zodInputSchema;
  try {
    zodInputSchema = resolveSerializedZodOutput(jsonSchemaToZod(tool.inputSchema as unknown as JsonSchema));
  } catch (e) {
    console.error('Error processing input schema:', e);
    toast.error('Failed to process tool input schema.');
    zodInputSchema = z.object({});
  }

  return (
    <ToolExecutor
      executionResult={result}
      isExecutingTool={isExecuting}
      zodInputSchema={zodInputSchema}
      handleExecuteTool={handleExecuteTool}
      toolDescription={tool.description || ''}
      toolId={tool.id}
    />
  );
};
