import { RequestContext } from '@mastra/core/request-context';
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { rerank } from '../rerank';
import { vectorQuerySearch } from '../utils';
import { createVectorQueryTool } from './vector-query';

vi.mock('../utils', async importOriginal => {
  const actual: any = await importOriginal();
  return {
    ...actual,
    vectorQuerySearch: vi.fn().mockResolvedValue({ results: [{ metadata: { text: 'foo' }, vector: [1, 2, 3] }] }),
  };
});

vi.mock('../rerank', async importOriginal => {
  const actual: any = await importOriginal();
  return {
    ...actual,
    rerank: vi
      .fn()
      .mockResolvedValue([
        { result: { id: '1', metadata: { text: 'bar' }, score: 1, details: { semantic: 1, vector: 1, position: 1 } } },
      ]),
  };
});

describe('createVectorQueryTool', () => {
  const mockModel = { name: 'test-model' } as any;
  const mockMastra = {
    vectors: {
      testStore: {
        // Mock vector store methods
      },
      anotherStore: {
        // Mock vector store methods
      },
    },
    getVector: vi.fn(storeName => ({
      [storeName]: {
        // Mock vector store methods
      },
    })),
    logger: {
      debug: vi.fn(),
      warn: vi.fn(),
      info: vi.fn(),
      error: vi.fn(),
    },
    getLogger: vi.fn(() => ({
      debug: vi.fn(),
      warn: vi.fn(),
      info: vi.fn(),
      error: vi.fn(),
    })),
  };

  beforeEach(() => {
    vi.clearAllMocks();
  });

  describe('input schema validation', () => {
    it('should handle filter permissively when enableFilter is false', () => {
      // Create tool with enableFilter set to false
      const tool = createVectorQueryTool({
        vectorStoreName: 'testStore',
        indexName: 'testIndex',
        model: mockModel,
        enableFilter: false,
      });

      // Get the Zod schema
      const schema = tool.inputSchema;

      // Test with no filter (should be valid)
      const validInput = {
        queryText: 'test query',
        topK: 5,
      };
      expect(() => schema?.parse(validInput)).not.toThrow();

      // Test with filter (should throw - unexpected property)
      const inputWithFilter = {
        ...validInput,
        filter: '{"field": "value"}',
      };
      expect(() => schema?.parse(inputWithFilter)).not.toThrow();
    });

    it('should handle filter when enableFilter is true', () => {
      const tool = createVectorQueryTool({
        vectorStoreName: 'testStore',
        indexName: 'testIndex',
        model: mockModel,
        enableFilter: true,
      });

      // Get the Zod schema
      const schema = tool.inputSchema;

      // Test various filter inputs that should coerce to string
      const testCases = [
        // String inputs
        { filter: '{"field": "value"}' },
        { filter: '{}' },
        { filter: 'simple-string' },
        // Empty
        { filter: '' },
        { filter: { field: 'value' } },
        { filter: {} },
        { filter: 123 },
        { filter: null },
        { filter: undefined },
      ];

      testCases.forEach(({ filter }) => {
        expect(() =>
          schema?.parse({
            queryText: 'test query',
            topK: 5,
            filter,
          }),
        ).not.toThrow();
      });

      // Verify that all parsed values are strings
      testCases.forEach(({ filter }) => {
        const result = schema?.parse({
          queryText: 'test query',
          topK: 5,
          filter,
        });
        expect(typeof result?.filter).toBe('string');
      });
    });

    it('should not reject unexpected properties in both modes', () => {
      // Test with enableFilter false
      const toolWithoutFilter = createVectorQueryTool({
        vectorStoreName: 'testStore',
        indexName: 'testIndex',
        model: mockModel,
        enableFilter: false,
      });

      // Should reject unexpected property
      expect(() =>
        toolWithoutFilter.inputSchema?.parse({
          queryText: 'test query',
          topK: 5,
          unexpectedProp: 'value',
        }),
      ).not.toThrow();

      // Test with enableFilter true
      const toolWithFilter = createVectorQueryTool({
        vectorStoreName: 'testStore',
        indexName: 'testIndex',
        model: mockModel,
        enableFilter: true,
      });

      // Should reject unexpected property even with valid filter
      expect(() =>
        toolWithFilter.inputSchema?.parse({
          queryText: 'test query',
          topK: 5,
          filter: '{}',
          unexpectedProp: 'value',
        }),
      ).not.toThrow();
    });
  });

  describe('execute function', () => {
    it('should not process filter when enableFilter is false', async () => {
      const requestContext = new RequestContext();

      // Create tool with enableFilter set to false
      const tool = createVectorQueryTool({
        vectorStoreName: 'testStore',
        indexName: 'testIndex',
        model: mockModel,
        enableFilter: false,
      });

      // Execute with no filter
      await tool.execute?.(
        {
          queryText: 'test query',
          topK: 5,
        },
        {
          mastra: mockMastra as any,
          requestContext,
        },
      );

      // Check that vectorQuerySearch was called with undefined queryFilter
      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          queryFilter: undefined,
        }),
      );
    });

    it('should process filter when enableFilter is true and filter is provided', async () => {
      const requestContext = new RequestContext();
      // Create tool with enableFilter set to true
      const tool = createVectorQueryTool({
        vectorStoreName: 'testStore',
        indexName: 'testIndex',
        model: mockModel,
        enableFilter: true,
      });

      const filterJson = '{"field": "value"}';

      // Execute with filter
      await tool.execute?.(
        {
          queryText: 'test query',
          topK: 5,
          filter: filterJson,
        },
        {
          mastra: mockMastra as any,
          requestContext,
        },
      );

      // Check that vectorQuerySearch was called with the parsed filter
      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          queryFilter: { field: 'value' },
        }),
      );
    });

    it('should return empty results for invalid JSON string filters', async () => {
      const requestContext = new RequestContext();
      // Create tool with enableFilter set to true
      const tool = createVectorQueryTool({
        vectorStoreName: 'testStore',
        indexName: 'testIndex',
        model: mockModel,
        enableFilter: true,
      });

      const stringFilter = 'string-filter';

      // Execute with string filter - invalid JSON is logged and returns empty results
      const result = await tool.execute?.(
        {
          queryText: 'test query',
          topK: 5,
          filter: stringFilter,
        },
        {
          mastra: mockMastra as any,
          requestContext,
        },
      );

      expect(result).toEqual({ relevantContext: [], sources: [] });
      expect(vectorQuerySearch).not.toHaveBeenCalled();
    });

    it('returns empty results when no Mastra server or vector store is provided', async () => {
      const tool = createVectorQueryTool({
        id: 'test',
        model: mockModel,
        indexName: 'testIndex',
        vectorStoreName: 'testStore',
      });

      const requestContext = new RequestContext();
      const result = await tool.execute(
        {
          queryText: 'foo',
          topK: 1,
        },
        {
          requestContext,
        },
      );

      // Returns empty results for graceful degradation
      expect(result).toEqual({ relevantContext: [], sources: [] });
      expect(vectorQuerySearch).not.toHaveBeenCalled();
    });

    it('works without a mastra server if a vector store is passed as an argument', async () => {
      const testStore = {
        testStore: {},
      };
      const tool = createVectorQueryTool({
        id: 'test',
        model: mockModel,
        indexName: 'testIndex',
        vectorStoreName: 'testStore',
        vectorStore: testStore as any,
      });

      const requestContext = new RequestContext();
      const result = await tool.execute(
        {
          queryText: 'foo',
          topK: 1,
        },
        {
          requestContext,
        },
      );

      expect(result.relevantContext[0]).toEqual({ text: 'foo' });
      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          databaseConfig: undefined,
          indexName: 'testIndex',
          vectorStore: {
            testStore: {},
          },
          queryText: 'foo',
          model: mockModel,
          queryFilter: undefined,
          topK: 1,
        }),
      );
    });

    it('prefers the passed vector store over one from a passed Mastra server', async () => {
      const thirdStore = {
        thirdStore: {},
      };
      const tool = createVectorQueryTool({
        id: 'test',
        model: mockModel,
        indexName: 'testIndex',
        vectorStoreName: 'thirdStore',
        vectorStore: thirdStore as any,
      });

      const requestContext = new RequestContext();
      const result = await tool.execute(
        {
          queryText: 'foo',
          topK: 1,
        },
        {
          mastra: mockMastra as any,
          requestContext,
        },
      );

      expect(result.relevantContext[0]).toEqual({ text: 'foo' });
      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          databaseConfig: undefined,
          indexName: 'testIndex',
          vectorStore: {
            thirdStore: {},
          },
          queryText: 'foo',
          model: mockModel,
          queryFilter: undefined,
          topK: 1,
        }),
      );
    });
  });

  describe('requestContext', () => {
    it('calls vectorQuerySearch with requestContext params', async () => {
      const tool = createVectorQueryTool({
        id: 'test',
        model: mockModel,
        indexName: 'testIndex',
        vectorStoreName: 'testStore',
      });
      const requestContext = new RequestContext();
      requestContext.set('indexName', 'anotherIndex');
      requestContext.set('vectorStoreName', 'anotherStore');
      requestContext.set('topK', 3);
      requestContext.set('filter', { foo: 'bar' });
      requestContext.set('includeVectors', true);
      requestContext.set('includeSources', false);
      const result = await tool.execute(
        {
          queryText: 'foo',
          topK: 6,
        },
        {
          mastra: mockMastra as any,
          requestContext,
        },
      );
      expect(result.relevantContext.length).toBeGreaterThan(0);
      expect(result.sources).toEqual([]); // includeSources false
      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          indexName: 'anotherIndex',
          vectorStore: {
            anotherStore: {},
          },
          queryText: 'foo',
          model: mockModel,
          queryFilter: { foo: 'bar' },
          topK: 3,
          includeVectors: true,
        }),
      );
    });

    it('handles reranker from requestContext', async () => {
      const tool = createVectorQueryTool({
        id: 'test',
        model: mockModel,
        indexName: 'testIndex',
        vectorStoreName: 'testStore',
      });
      const requestContext = new RequestContext();
      requestContext.set('indexName', 'testIndex');
      requestContext.set('vectorStoreName', 'testStore');
      requestContext.set('reranker', { model: 'reranker-model', options: { topK: 1 } });
      // Mock rerank
      vi.mocked(rerank).mockResolvedValue([
        {
          result: { id: '1', metadata: { text: 'bar' }, score: 1 },
          score: 1,
          details: { semantic: 1, vector: 1, position: 1 },
        },
      ]);
      const result = await tool.execute(
        {
          queryText: 'foo',
          topK: 1,
        },
        {
          mastra: mockMastra as any,
          requestContext,
        },
      );
      expect(result.relevantContext[0]).toEqual({ text: 'bar' });
    });
  });

  describe('dynamic vectorStore (multi-tenant schema support)', () => {
    it('should support vectorStore as a function that receives requestContext', async () => {
      // Simulate multi-tenant setup where each tenant has a different schema
      const tenantAVectorStore = { id: 'tenant-a-store' } as any;
      const tenantBVectorStore = { id: 'tenant-b-store' } as any;

      const vectorStoreResolver = vi.fn(({ requestContext }: { requestContext?: RequestContext }) => {
        const schemaId = requestContext?.get('schemaId');
        return schemaId === 'tenant-a' ? tenantAVectorStore : tenantBVectorStore;
      });

      const tool = createVectorQueryTool({
        indexName: 'tenant_embeddings',
        model: mockModel,
        vectorStore: vectorStoreResolver,
      });

      // Test with tenant A context
      const tenantAContext = new RequestContext();
      tenantAContext.set('schemaId', 'tenant-a');

      await tool.execute({ queryText: 'test query', topK: 5 }, { requestContext: tenantAContext });

      expect(vectorStoreResolver).toHaveBeenCalledWith(expect.objectContaining({ requestContext: tenantAContext }));
      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          vectorStore: tenantAVectorStore,
        }),
      );

      vi.clearAllMocks();

      // Test with tenant B context
      const tenantBContext = new RequestContext();
      tenantBContext.set('schemaId', 'tenant-b');

      await tool.execute({ queryText: 'test query', topK: 5 }, { requestContext: tenantBContext });

      expect(vectorStoreResolver).toHaveBeenCalledWith(expect.objectContaining({ requestContext: tenantBContext }));
      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          vectorStore: tenantBVectorStore,
        }),
      );
    });

    it('should support async vectorStore resolver function', async () => {
      const asyncVectorStore = { id: 'async-resolved-store' } as any;

      const asyncVectorStoreResolver = vi.fn(
        async ({ requestContext: _requestContext }: { requestContext?: RequestContext }) => {
          // Simulate async operation (e.g., fetching from DB or initializing per-tenant store)
          await new Promise(resolve => setTimeout(resolve, 10));
          return asyncVectorStore;
        },
      );

      const tool = createVectorQueryTool({
        indexName: 'testIndex',
        model: mockModel,
        vectorStore: asyncVectorStoreResolver,
      });

      const requestContext = new RequestContext();

      await tool.execute({ queryText: 'test query', topK: 5 }, { requestContext });

      expect(asyncVectorStoreResolver).toHaveBeenCalled();
      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          vectorStore: asyncVectorStore,
        }),
      );
    });

    it('should return empty results when async vectorStore resolver throws', async () => {
      const asyncVectorStoreResolver = vi.fn(async () => {
        await new Promise(resolve => setTimeout(resolve, 10));
        throw new Error('Failed to initialize vector store for tenant');
      });

      const tool = createVectorQueryTool({
        indexName: 'testIndex',
        model: mockModel,
        vectorStore: asyncVectorStoreResolver,
      });

      const requestContext = new RequestContext();

      // Error is logged and returns empty results for graceful degradation
      const result = await tool.execute({ queryText: 'test query', topK: 5 }, { requestContext });

      expect(result).toEqual({ relevantContext: [], sources: [] });
      expect(asyncVectorStoreResolver).toHaveBeenCalled();
      expect(vectorQuerySearch).not.toHaveBeenCalled();
    });

    it('should pass mastra instance to vectorStore resolver function', async () => {
      const vectorStoreResolver = vi.fn(({ mastra: _mastra }: { mastra?: any }) => {
        // Use mastra to get a custom vector store
        return { id: 'mastra-resolved-store' } as any;
      });

      const tool = createVectorQueryTool({
        indexName: 'testIndex',
        model: mockModel,
        vectorStore: vectorStoreResolver,
      });

      const requestContext = new RequestContext();

      await tool.execute({ queryText: 'test query', topK: 5 }, { mastra: mockMastra as any, requestContext });

      expect(vectorStoreResolver).toHaveBeenCalledWith(expect.objectContaining({ mastra: mockMastra }));
    });

    it('should still support static vectorStore (existing behavior)', async () => {
      const staticVectorStore = { id: 'static-store' } as any;

      const tool = createVectorQueryTool({
        indexName: 'testIndex',
        model: mockModel,
        vectorStore: staticVectorStore,
      });

      await tool.execute({ queryText: 'test query', topK: 5 }, { requestContext: new RequestContext() });

      expect(vectorQuerySearch).toHaveBeenCalledWith(
        expect.objectContaining({
          vectorStore: staticVectorStore,
        }),
      );
    });

    it('should return empty results when vectorStore resolver returns undefined', async () => {
      const vectorStoreResolver = vi.fn(({ requestContext: _requestContext }: { requestContext?: RequestContext }) => {
        // Simulate a resolver that can't find a store for the given context
        return undefined;
      });

      const tool = createVectorQueryTool({
        indexName: 'testIndex',
        model: mockModel,
        vectorStore: vectorStoreResolver,
      });

      const requestContext = new RequestContext();
      requestContext.set('schemaId', 'unknown-tenant');

      const result = await tool.execute({ queryText: 'test query', topK: 5 }, { requestContext });

      // Returns empty results for graceful degradation
      expect(result).toEqual({ relevantContext: [], sources: [] });
      expect(vectorStoreResolver).toHaveBeenCalled();
      expect(vectorQuerySearch).not.toHaveBeenCalled();
    });

    it('should return empty results when async vectorStore resolver returns undefined', async () => {
      const asyncVectorStoreResolver = vi.fn(
        async ({ requestContext: _requestContext }: { requestContext?: RequestContext }) => {
          await new Promise(resolve => setTimeout(resolve, 10));
          return undefined;
        },
      );

      const tool = createVectorQueryTool({
        indexName: 'testIndex',
        model: mockModel,
        vectorStore: asyncVectorStoreResolver,
      });

      const requestContext = new RequestContext();

      const result = await tool.execute({ queryText: 'test query', topK: 5 }, { requestContext });

      // Returns empty results for graceful degradation
      expect(result).toEqual({ relevantContext: [], sources: [] });
      expect(asyncVectorStoreResolver).toHaveBeenCalled();
      expect(vectorQuerySearch).not.toHaveBeenCalled();
    });
  });

  describe('providerOptions', () => {
    it('should pass providerOptions to vectorQuerySearch', async () => {
      const tool = createVectorQueryTool({
        indexName: 'testIndex',
        model: mockModel,
        vectorStoreName: 'testStore',
        providerOptions: { google: { outputDimensionality: 1536 } },
      });

      await tool.execute(
        {
          queryText: 'foo',
          topK: 10,
        },
        {
          mastra: mockMastra as any,
          requestContext: new RequestContext(),
        },
      );

      expect(vectorQuerySearch).toHaveBeenCalledWith({
        indexName: 'testIndex',
        vectorStore: { testStore: {} },
        queryText: 'foo',
        model: mockModel,
        queryFilter: undefined,
        topK: 10,
        includeVectors: false,
        databaseConfig: undefined,
        providerOptions: { google: { outputDimensionality: 1536 } },
      });
    });

    it('should allow providerOptions override via requestContext', async () => {
      const tool = createVectorQueryTool({
        indexName: 'testIndex',
        model: mockModel,
        vectorStoreName: 'testStore',
        providerOptions: { google: { outputDimensionality: 1536 } },
      });

      const requestContext = new RequestContext();
      requestContext.set('providerOptions', { google: { outputDimensionality: 768 } });

      await tool.execute(
        {
          queryText: 'foo',
          topK: 10,
        },
        {
          mastra: mockMastra as any,
          requestContext,
        },
      );

      expect(vectorQuerySearch).toHaveBeenCalledWith({
        indexName: 'testIndex',
        vectorStore: { testStore: {} },
        queryText: 'foo',
        model: mockModel,
        queryFilter: undefined,
        topK: 10,
        includeVectors: false,
        databaseConfig: undefined,
        providerOptions: { google: { outputDimensionality: 768 } },
      });
    });
  });
});
