import { describe, it, expect, beforeAll } from 'vitest';
import type { MastraStorage, MemoryStorage } from '@mastra/core/storage';
import { createSampleThread, createSampleMessageV2 } from './data';

export function createMessagesBulkDeleteTest({ storage }: { storage: MastraStorage }) {
  describe('Messages Bulk Delete', () => {
    let memoryStorage: MemoryStorage;

    beforeAll(async () => {
      const store = await storage.getStore('memory');
      if (!store) {
        throw new Error('Memory storage not found');
      }
      memoryStorage = store;
    });

    it('should delete multiple messages successfully', async () => {
      // Create a thread first
      const thread = createSampleThread();
      await memoryStorage.saveThread({ thread });

      // Save multiple messages
      const messages = Array.from({ length: 5 }, (_, index) => {
        const msg = createSampleMessageV2({
          threadId: thread.id,
          content: { content: `Message ${index}` },
        });
        msg.id = `msg-${index}`;
        return msg;
      });

      const { messages: savedMessages } = await memoryStorage.saveMessages({ messages });
      expect(savedMessages).toHaveLength(5);

      // Delete messages 1, 2, and 4
      await memoryStorage.deleteMessages(['msg-1', 'msg-2', 'msg-4']);

      // Verify only messages 0 and 3 remain
      const { messages: remainingMessages } = await memoryStorage.listMessages({ threadId: thread.id });
      expect(remainingMessages).toHaveLength(2);
      expect(remainingMessages.map(m => m.id).sort()).toEqual(['msg-0', 'msg-3']);
    });

    it('should handle empty array gracefully', async () => {
      // Should not throw when deleting empty array
      await expect(memoryStorage.deleteMessages([])).resolves.not.toThrow();
    });

    it('should handle deleting non-existent messages', async () => {
      // Should not throw when deleting messages that don't exist
      await expect(memoryStorage.deleteMessages(['non-existent-1', 'non-existent-2'])).resolves.not.toThrow();
    });

    it('should update thread timestamp when messages are deleted', async () => {
      // Create a thread
      const thread = createSampleThread();
      const savedThread = await memoryStorage.saveThread({ thread });
      const originalUpdatedAt = new Date(savedThread.updatedAt).getTime();

      // Wait a bit to ensure timestamp difference
      await new Promise(resolve => setTimeout(resolve, 10));

      // Save multiple messages
      const messages = Array.from({ length: 3 }, (_, index) => {
        const msg = createSampleMessageV2({
          threadId: thread.id,
          content: { content: `Message ${index}` },
        });
        msg.id = `bulk-msg-${index}`;
        return msg;
      });
      await memoryStorage.saveMessages({ messages });

      // Wait a bit more
      await new Promise(resolve => setTimeout(resolve, 10));

      // Delete all messages
      await memoryStorage.deleteMessages(['bulk-msg-0', 'bulk-msg-1', 'bulk-msg-2']);

      // Check thread timestamp was updated
      const updatedThread = await memoryStorage.getThreadById({ threadId: thread.id });
      const newUpdatedAt = new Date(updatedThread!.updatedAt).getTime();
      expect(newUpdatedAt).toBeGreaterThan(originalUpdatedAt);
    });

    it('should handle messages from different threads', async () => {
      // Create two threads
      const thread1 = createSampleThread({ id: 'bulk-thread-1' });
      const thread2 = createSampleThread({ id: 'bulk-thread-2' });
      await memoryStorage.saveThread({ thread: thread1 });
      await memoryStorage.saveThread({ thread: thread2 });

      // Save messages to both threads
      const messages1 = Array.from({ length: 2 }, (_, index) => {
        const msg = createSampleMessageV2({
          threadId: 'bulk-thread-1',
          content: { content: `Thread 1 Message ${index}` },
        });
        msg.id = `bulk-thread1-msg-${index}`;
        return msg;
      });
      const messages2 = Array.from({ length: 2 }, (_, index) => {
        const msg = createSampleMessageV2({
          threadId: 'bulk-thread-2',
          content: { content: `Thread 2 Message ${index}` },
        });
        msg.id = `bulk-thread2-msg-${index}`;
        return msg;
      });

      await memoryStorage.saveMessages({ messages: messages1 });
      await memoryStorage.saveMessages({ messages: messages2 });

      // Delete one message from each thread
      await memoryStorage.deleteMessages(['bulk-thread1-msg-0', 'bulk-thread2-msg-1']);

      // Verify thread 1 has one message remaining
      const { messages: thread1Messages } = await memoryStorage.listMessages({ threadId: 'bulk-thread-1' });
      expect(thread1Messages).toHaveLength(1);
      expect(thread1Messages[0]!.id).toBe('bulk-thread1-msg-1');

      // Verify thread 2 has one message remaining
      const { messages: thread2Messages } = await memoryStorage.listMessages({ threadId: 'bulk-thread-2' });
      expect(thread2Messages).toHaveLength(1);
      expect(thread2Messages[0]!.id).toBe('bulk-thread2-msg-0');
    });

    it('should handle large batches of message deletions', async () => {
      // Create a thread with a unique ID for this test
      const thread = createSampleThread({ id: `bulk-delete-test-thread-${Date.now()}` });
      await memoryStorage.saveThread({ thread });

      // Save 100 messages with alternating roles
      const messages = Array.from({ length: 100 }, (_, index) => {
        const msg = createSampleMessageV2({
          threadId: thread.id,
          content: { content: `Message ${index}` },
        });
        msg.id = `large-batch-msg-${index}`;
        // Alternate between user and assistant roles
        msg.role = index % 2 === 0 ? 'user' : 'assistant';
        return msg;
      });

      await memoryStorage.saveMessages({ messages });

      // Verify all 100 messages were saved
      const { messages: allMessages } = await memoryStorage.listMessages({
        threadId: thread.id,
        perPage: 100,
        orderBy: { field: 'createdAt', direction: 'DESC' },
      });
      expect(allMessages).toHaveLength(100);

      // Delete the most recent 50 messages (indices 50-99)
      const messagesToDelete = messages.slice(50).map(msg => msg.id);

      await memoryStorage.deleteMessages(messagesToDelete);

      // Verify 50 messages remain - need to specify limit to get all remaining messages
      const { messages: remainingMessages } = await memoryStorage.listMessages({
        threadId: thread.id,
        perPage: 100,
        orderBy: { field: 'createdAt', direction: 'DESC' },
      });
      expect(remainingMessages).toHaveLength(50);

      // Verify the correct messages remain (first 50 messages, indices 0-49)
      const remainingIds = remainingMessages.map(m => m.id);
      for (let i = 0; i < 50; i++) {
        expect(remainingIds).toContain(`large-batch-msg-${i}`);
      }

      // Verify the deleted messages are not present (indices 50-99)
      for (let i = 50; i < 100; i++) {
        expect(remainingIds).not.toContain(`large-batch-msg-${i}`);
      }
    });

    it('should handle mixed valid and invalid message IDs', async () => {
      // Create a thread
      const thread = createSampleThread();
      await memoryStorage.saveThread({ thread });

      // Save some messages
      const messages = Array.from({ length: 3 }, (_, index) => {
        const msg = createSampleMessageV2({
          threadId: thread.id,
          content: { content: `Message ${index}` },
        });
        msg.id = `mixed-msg-${index}`;
        return msg;
      });

      await memoryStorage.saveMessages({ messages });

      // Delete mix of valid and invalid IDs
      await memoryStorage.deleteMessages(['mixed-msg-0', 'invalid-id-1', 'mixed-msg-2', 'invalid-id-2']);

      // Verify only the valid messages were deleted
      const { messages: remainingMessages } = await memoryStorage.listMessages({ threadId: thread.id });
      expect(remainingMessages).toHaveLength(1);
      expect(remainingMessages[0]!.id).toBe('mixed-msg-1');
    });
  });
}
