import { randomUUID } from 'node:crypto';
import { beforeAll, describe, it, expect, beforeEach } from 'vitest';
import { createSampleScore } from './data';
import type { ScoreRowData } from '@mastra/core/evals';
import type { MastraStorage, ScoresStorage } from '@mastra/core/storage';
import type { TestCapabilities } from '../../factory';

export function createScoresTest({
  storage,
  capabilities = {},
}: {
  storage: MastraStorage;
  capabilities?: TestCapabilities;
}) {
  let scoresStorage: ScoresStorage;

  // Helper function for creating test scores
  async function createScores(
    configs: Array<{
      count: number;
      scorerId: string;
      traceId: string;
      spanId: string;
    }>,
  ): Promise<ScoreRowData[]> {
    const allScores: ScoreRowData[] = [];

    for (const config of configs) {
      for (let i = 0; i < config.count; i++) {
        const score = createSampleScore({
          scorerId: config.scorerId,
          traceId: config.traceId,
          spanId: config.spanId,
        });
        allScores.push(score);
        await scoresStorage.saveScore(score);
      }
    }

    return allScores;
  }

  beforeAll(async () => {
    const store = await storage.getStore('scores');
    if (!store) {
      throw new Error('Scores storage not found');
    }
    scoresStorage = store;
  });

  describe('Score Operations', () => {
    beforeEach(async () => {
      await scoresStorage.dangerouslyClearAll();
    });

    it('should retrieve scores by scorer id', async () => {
      const scorerId = `scorer-${randomUUID()}`;

      // Create sample scores
      const score1 = createSampleScore({ scorerId });
      const score2 = createSampleScore({ scorerId });
      const score3 = createSampleScore({ scorerId });

      // Insert evals

      await scoresStorage.saveScore(score1);
      await scoresStorage.saveScore(score2);
      await scoresStorage.saveScore(score3);

      // Test getting all evals for the agent
      const allScoresByScorerId = await scoresStorage.listScoresByScorerId({
        scorerId,
        pagination: { page: 0, perPage: 10 },
      });
      expect(allScoresByScorerId?.scores).toHaveLength(3);
      expect(allScoresByScorerId?.scores.map(e => e.runId)).toEqual(
        expect.arrayContaining([score1.runId, score2.runId, score3.runId]),
      );
      expect(allScoresByScorerId?.scores.map(s => s.scorer.id)).toEqual(
        expect.arrayContaining([score1.scorer.id, score2.scorer.id, score3.scorer.id]),
      );

      // Test getting scores for non-existent scorer
      const nonExistentScores = await scoresStorage.listScoresByScorerId({
        scorerId: 'non-existent-scorer',
        pagination: { page: 0, perPage: 10 },
      });
      expect(nonExistentScores?.scores).toHaveLength(0);
    });

    it('should return score payload matching the saved score', async () => {
      const scorerId = `scorer-${randomUUID()}`;
      const score = createSampleScore({ scorerId });

      await scoresStorage.saveScore(score);

      const result = await scoresStorage.listScoresByScorerId({
        scorerId,
        pagination: { page: 0, perPage: 10 },
      });

      expect(result?.scores).toHaveLength(1);
      const returnedScore = result?.scores[0];

      // Normalize for comparison:
      // - Dates: omit from object comparison, check separately with tolerance
      // - Nullish values: omit from comparison (storage may omit null/undefined)
      const normalizeScore = (s: typeof score) => {
        const normalized: Record<string, unknown> = {};
        for (const [key, value] of Object.entries(s)) {
          if (value === undefined || value === null) {
            continue;
          } else if (key === 'createdAt' || key === 'updatedAt') {
            continue; // Dates are automatically created by the database
          } else if (key === 'id') {
            continue; // Skip id since it's generated by the database
          } else {
            normalized[key] = value;
          }
        }
        return normalized;
      };

      const normalizedReturned = normalizeScore(returnedScore!);
      const normalizedExpected = normalizeScore(score);

      // Check each property individually for clearer error messages
      for (const key of Object.keys(normalizedExpected)) {
        if (key === 'score') {
          expect(normalizedReturned[key], `Property "${key}" should match`).toBeCloseTo(
            normalizedExpected[key] as number,
          );
        } else {
          expect(normalizedReturned[key], `Property "${key}" should match`).toEqual(normalizedExpected[key]);
        }
      }

      // Ensure no extra properties in returned score
      expect(Object.keys(normalizedReturned).sort()).toEqual(Object.keys(normalizedExpected).sort());
    });

    it('should retrieve scores by source', async () => {
      const scorerId = `scorer-${randomUUID()}`;
      const score1 = createSampleScore({ scorerId, source: 'TEST' });
      const score2 = createSampleScore({ scorerId, source: 'LIVE' });
      await scoresStorage.saveScore(score1);
      await scoresStorage.saveScore(score2);
      const scoresBySource = await scoresStorage.listScoresByScorerId({
        scorerId,
        pagination: { page: 0, perPage: 10 },
        source: 'TEST',
      });
      expect(scoresBySource?.scores).toHaveLength(1);
      expect(scoresBySource?.scores.map(s => s.source)).toEqual(['TEST']);
    });

    it('should save scorer', async () => {
      const scorerId = `scorer-${randomUUID()}`;
      const scorer = createSampleScore({ scorerId });
      await scoresStorage.saveScore(scorer);
      const result = await scoresStorage.listScoresByRunId({
        runId: scorer.runId,
        pagination: { page: 0, perPage: 10 },
      });
      expect(result.scores).toHaveLength(1);
      expect(result.pagination.total).toBe(1);
      expect(result.pagination.page).toBe(0);
      expect(result.pagination.perPage).toBe(10);
      expect(result.pagination.hasMore).toBe(false);
    });

    it('should retrieve saved score by its returned id', async () => {
      const scorerId = `scorer-${randomUUID()}`;
      const score = createSampleScore({ scorerId });

      // Save the score and get the returned score with its id
      const { score: savedScore } = await scoresStorage.saveScore(score);
      expect(savedScore.id).toBeDefined();

      // Retrieve the score by its id - this should find the saved score
      const retrievedScore = await scoresStorage.getScoreById({ id: savedScore.id });

      expect(retrievedScore).not.toBeNull();
      expect(retrievedScore?.id).toBe(savedScore.id);
      expect(retrievedScore?.scorerId).toBe(scorerId);
      expect(retrievedScore?.runId).toBe(score.runId);
    });

    it('listScoresByEntityId should return paginated scores with total count when returnPaginationResults is true', async () => {
      const scorer = createSampleScore({ scorerId: `scorer-${randomUUID()}` });
      await scoresStorage.saveScore(scorer);

      const result = await scoresStorage.listScoresByEntityId({
        entityId: scorer.entity!.id as string,
        entityType: scorer.entityType!,
        pagination: { page: 0, perPage: 10 },
      });
      expect(result.scores).toHaveLength(1);
      expect(result.pagination.total).toBe(1);
      expect(result.pagination.page).toBe(0);
      expect(result.pagination.perPage).toBe(10);
      expect(result.pagination.hasMore).toBe(false);
    });

    // listScoresBySpan defaults to true since most stores support it
    if (capabilities.listScoresBySpan !== false) {
      it('should retrieve scores by trace and span id', async () => {
        const scorerId = `scorer-${randomUUID()}`;
        const traceId = randomUUID();
        const spanId = randomUUID();

        const score = createSampleScore({ scorerId, traceId, spanId });
        await scoresStorage.saveScore(score);

        const result = await scoresStorage.listScoresBySpan({
          traceId,
          spanId,
          pagination: { page: 0, perPage: 10 },
        });

        expect(result.scores.length).toBe(1);
        expect(result.scores[0]?.traceId).toBe(traceId);
        expect(result.scores[0]?.spanId).toBe(spanId);
        expect(result.pagination.total).toBe(1);
        expect(result.pagination.hasMore).toBe(false);
      });

      it('should retrieve multiple scores by trace and span id', async () => {
        const scorerId = `scorer-${randomUUID()}`;
        const traceId = randomUUID();
        const spanId = randomUUID();

        // Create multiple scores for the same trace/span
        const score1 = createSampleScore({ scorerId, traceId, spanId });
        const score2 = createSampleScore({ scorerId, traceId, spanId });
        const score3 = createSampleScore({ scorerId, traceId, spanId });

        await scoresStorage.saveScore(score1);
        await scoresStorage.saveScore(score2);
        await scoresStorage.saveScore(score3);

        const result = await scoresStorage.listScoresBySpan({
          traceId,
          spanId,
          pagination: { page: 0, perPage: 10 },
        });

        expect(result.scores.every(s => s.traceId === traceId)).toBe(true);
        expect(result.scores.every(s => s.spanId === spanId)).toBe(true);
        expect(result.scores.length).toBe(3);
        expect(result.pagination.total).toBe(3);
        expect(result.pagination.hasMore).toBe(false);
      });

      it('should handle first page pagination correctly', async () => {
        const scorerId = `scorer-${randomUUID()}`;
        const traceId = randomUUID();
        const spanId = randomUUID();

        await createScores([
          { count: 5, scorerId, traceId, spanId }, // target scores
          { count: 1, scorerId, traceId: randomUUID(), spanId }, // different trace
          { count: 1, scorerId, traceId, spanId: randomUUID() }, // different span
          { count: 1, scorerId, traceId: randomUUID(), spanId: randomUUID() }, // both different
        ]);

        const firstPage = await scoresStorage.listScoresBySpan({
          traceId,
          spanId,
          pagination: { page: 0, perPage: 2 },
        });

        expect(firstPage.scores.length).toBe(2);
        expect(firstPage.pagination.total).toBe(5);
        expect(firstPage.pagination.page).toBe(0);
        expect(firstPage.pagination.perPage).toBe(2);
        expect(firstPage.pagination.hasMore).toBe(true);

        expect(firstPage.scores.every(s => s.traceId === traceId && s.spanId === spanId)).toBe(true);
      });

      it('should handle middle page pagination correctly', async () => {
        const scorerId = `scorer-${randomUUID()}`;
        const traceId = randomUUID();
        const spanId = randomUUID();

        const allScores = await createScores([
          { count: 5, scorerId, traceId, spanId }, // target scores
          { count: 1, scorerId, traceId: randomUUID(), spanId }, // different trace
          { count: 1, scorerId, traceId, spanId: randomUUID() }, // different span
          { count: 1, scorerId, traceId: randomUUID(), spanId: randomUUID() }, // both different
        ]);

        const secondPage = await scoresStorage.listScoresBySpan({
          traceId,
          spanId,
          pagination: { page: 1, perPage: 2 },
        });

        expect(secondPage.scores.length).toBe(2);
        expect(secondPage.pagination.total).toBe(5);
        expect(secondPage.pagination.page).toBe(1);
        expect(secondPage.pagination.perPage).toBe(2);
        expect(secondPage.pagination.hasMore).toBe(true);

        expect(secondPage.scores.every(s => s.traceId === traceId && s.spanId === spanId)).toBe(true);
      });

      it('should handle last page pagination', async () => {
        const scorerId = `scorer-${randomUUID()}`;
        const traceId = randomUUID();
        const spanId = randomUUID();

        const otherTraceId1 = randomUUID();
        const otherTraceId2 = randomUUID();
        const otherSpanId1 = randomUUID();
        const otherSpanId2 = randomUUID();

        await createScores([
          { count: 5, scorerId, traceId, spanId }, // target scores
          { count: 1, scorerId, traceId: otherTraceId1, spanId }, // different trace, same span
          { count: 1, scorerId, traceId, spanId: otherSpanId1 }, // same trace, different span
          { count: 1, scorerId, traceId: otherTraceId2, spanId: otherSpanId2 }, // both different
          { count: 1, scorerId, traceId: otherTraceId1, spanId }, // different trace, same span (again)
          { count: 1, scorerId, traceId, spanId: otherSpanId2 }, // same trace, different span (again)
        ]);

        const firstPage = await scoresStorage.listScoresBySpan({
          traceId,
          spanId,
          pagination: { page: 0, perPage: 2 },
        });

        const secondPage = await scoresStorage.listScoresBySpan({
          traceId,
          spanId,
          pagination: { page: 1, perPage: 2 },
        });

        const lastPage = await scoresStorage.listScoresBySpan({
          traceId,
          spanId,
          pagination: { page: 2, perPage: 2 },
        });

        expect(lastPage.scores.length).toBe(1);
        expect(lastPage.pagination.total).toBe(5);
        expect(lastPage.pagination.page).toBe(2);
        expect(lastPage.pagination.perPage).toBe(2);
        expect(lastPage.pagination.hasMore).toBe(false);

        const allPages = [firstPage, secondPage, lastPage];
        for (const page of allPages) {
          expect(page.scores.every(s => s.traceId === traceId && s.spanId === spanId)).toBe(true);
        }
      });
    }
  });
}
