import type { TABLE_NAMES, TABLE_SCHEMAS, StorageColumn } from '@mastra/core/storage';
import {
  TABLE_MESSAGES,
  TABLE_RESOURCES,
  TABLE_SCORERS,
  TABLE_THREADS,
  TABLE_TRACES,
  TABLE_WORKFLOW_SNAPSHOT,
  safelyParseJSON,
  TABLE_SPANS,
} from '@mastra/core/storage';

export const TABLE_ENGINES: Record<TABLE_NAMES, string> = {
  [TABLE_MESSAGES]: `MergeTree()`,
  [TABLE_WORKFLOW_SNAPSHOT]: `ReplacingMergeTree()`,
  [TABLE_TRACES]: `MergeTree()`,
  [TABLE_THREADS]: `ReplacingMergeTree()`,
  [TABLE_SCORERS]: `MergeTree()`,
  [TABLE_RESOURCES]: `ReplacingMergeTree()`,
  // TODO: verify this is the correct engine for Spans when implementing clickhouse storage
  [TABLE_SPANS]: `ReplacingMergeTree()`,
  mastra_agents: `ReplacingMergeTree()`,
};

export const COLUMN_TYPES: Record<StorageColumn['type'], string> = {
  text: 'String',
  timestamp: 'DateTime64(3)',
  uuid: 'String',
  jsonb: 'String',
  integer: 'Int64',
  float: 'Float64',
  bigint: 'Int64',
  boolean: 'Bool',
};

export type IntervalUnit =
  | 'NANOSECOND'
  | 'MICROSECOND'
  | 'MILLISECOND'
  | 'SECOND'
  | 'MINUTE'
  | 'HOUR'
  | 'DAY'
  | 'WEEK'
  | 'MONTH'
  | 'QUARTER'
  | 'YEAR';

export type ClickhouseConfig = {
  url: string;
  username: string;
  password: string;
  ttl?: {
    [TableKey in TABLE_NAMES]?: {
      row?: { interval: number; unit: IntervalUnit; ttlKey?: string };
      columns?: Partial<{
        [ColumnKey in keyof (typeof TABLE_SCHEMAS)[TableKey]]: {
          interval: number;
          unit: IntervalUnit;
          ttlKey?: string;
        };
      }>;
    };
  };
};

// List of fields that should be parsed as JSON
const JSON_FIELDS = ['content', 'attributes', 'metadata', 'input', 'output', 'error', 'scope', 'links'];

// Fields that should be null instead of empty string when empty
const NULLABLE_STRING_FIELDS = ['parentSpanId', 'error'];

export function transformRow<R>(row: any): R {
  if (!row) {
    return row;
  }

  if (row.createdAt) {
    row.createdAt = new Date(row.createdAt);
  }
  if (row.updatedAt) {
    row.updatedAt = new Date(row.updatedAt);
  }
  if (row.startedAt) {
    row.startedAt = new Date(row.startedAt);
  }
  if (row.endedAt) {
    row.endedAt = new Date(row.endedAt);
  }

  // Parse JSONB fields if they're JSON strings
  for (const field of JSON_FIELDS) {
    if (row[field] && typeof row[field] === 'string') {
      row[field] = safelyParseJSON(row[field]);
    }
  }

  // Convert empty strings to null for nullable fields
  for (const field of NULLABLE_STRING_FIELDS) {
    if (row[field] === '') {
      row[field] = null;
    }
  }

  return row;
}

export function transformRows<R>(rows: any[]): R[] {
  return rows.map((row: any) => transformRow<R>(row));
}
