import { createWorkflow, createStep } from '@mastra/core/workflows';
import { z } from 'zod';

// Simple step that adds a letter to a string
const addLetterStep = createStep({
  id: 'add-letter',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
  execute: async ({ inputData }) => {
    const { text } = inputData;
    await new Promise(resolve => setTimeout(resolve, 100));
    return { text: text + 'A' };
  },
});

// Step that adds a different letter
const addLetterBStep = createStep({
  id: 'add-letter-b',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
  execute: async ({ inputData }) => {
    const { text } = inputData;
    await new Promise(resolve => setTimeout(resolve, 100));
    return { text: text + 'B' };
  },
});

// Step that adds another letter
const addLetterCStep = createStep({
  id: 'add-letter-c',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
  execute: async ({ inputData }) => {
    const { text } = inputData;
    // Make sure it runs after the other branch
    await new Promise(resolve => setTimeout(resolve, 100));
    return { text: text + 'C' };
  },
});

// Step that adds a letter and tracks iteration count
const addLetterWithCountStep = createStep({
  id: 'add-letter-with-count',
  inputSchema: z.object({
    text: z.string(),
    iterationCount: z.number().optional(),
  }),
  outputSchema: z.object({
    text: z.string(),
    iterationCount: z.number(),
  }),
  execute: async ({ inputData }) => {
    const { text, iterationCount = 0 } = inputData;
    await new Promise(resolve => setTimeout(resolve, 100));
    return {
      text: text + 'D',
      iterationCount: iterationCount + 1,
    };
  },
});

// Step with suspend/resume functionality
const suspendResumeStep = createStep({
  id: 'suspend-resume',
  inputSchema: z.object({
    text: z.string(),
    iterationCount: z.number(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
  suspendSchema: z.object({
    reason: z.string(),
  }),
  resumeSchema: z.object({
    userInput: z.string(),
  }),
  execute: async ({ inputData, resumeData, suspend }) => {
    const { text } = inputData;

    if (!resumeData?.userInput) {
      return await suspend({
        reason: 'Please provide user input to continue',
      });
    }

    await new Promise(resolve => setTimeout(resolve, 100));

    return { text: text + resumeData.userInput };
  },
});

// Step for short text (used in conditional branching)
const shortTextStep = createStep({
  id: 'short-text',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
  execute: async ({ inputData }) => {
    const { text } = inputData;
    await new Promise(resolve => setTimeout(resolve, 100));
    return { text: text + 'S' };
  },
});

// Step for long text (used in conditional branching)
const longTextStep = createStep({
  id: 'long-text',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
  execute: async ({ inputData }) => {
    const { text } = inputData;
    await new Promise(resolve => setTimeout(resolve, 100));
    return { text: text + 'L' };
  },
});

const finalStep = createStep({
  id: 'final-step',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
  execute: async ({ inputData }) => {
    const { text } = inputData;
    await new Promise(resolve => setTimeout(resolve, 100));
    return { text: text + '-ENDED' };
  },
});

// Nested workflow that processes text
export const nestedTextProcessor = createWorkflow({
  id: 'nested-text-processor',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
})
  .then(addLetterStep)
  .then(addLetterBStep)
  .commit();

// Main comprehensive workflow showcasing all features
export const complexWorkflow = createWorkflow({
  id: 'complex-workflow',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
})
  // Start with initial step
  .then(addLetterStep)

  // Parallel execution - both steps run simultaneously
  .parallel([addLetterBStep, addLetterCStep])

  // Map the parallel results back to a single text field
  .map(async ({ inputData }) => {
    const { 'add-letter-b': stepB, 'add-letter-c': stepC } = inputData;
    return { text: stepB.text + stepC.text };
  })

  // Conditional branching based on text length
  .branch([
    [async ({ inputData: { text } }) => text.length <= 10, shortTextStep],
    [async ({ inputData: { text } }) => text.length > 10, longTextStep],
  ])

  // Map the branch result back to a single text field
  .map(async ({ inputData }) => {
    // The branch step returns either short-text or long-text result
    const result = inputData['short-text'] || inputData['long-text'];
    return { text: result.text };
  })

  // Nested workflow
  .then(nestedTextProcessor)

  // doUntil loop - continues until text has 20+ characters
  .dountil(addLetterWithCountStep, async ({ inputData: { text } }) => text.length >= 20)

  // Suspend/resume step - requires user input
  .then(suspendResumeStep)

  // Final step
  .then(finalStep)
  .commit();

export const lessComplexWorkflow = createWorkflow({
  id: 'lessComplexWorkflow',
  inputSchema: z.object({
    text: z.string(),
  }),
  outputSchema: z.object({
    text: z.string(),
  }),
})
  // Start with initial step
  .then(addLetterStep)

  // Parallel execution - both steps run simultaneously
  .parallel([addLetterBStep, addLetterCStep])

  // Map the parallel results back to a single text field
  .map(async ({ inputData }) => {
    const { 'add-letter-b': stepB, 'add-letter-c': stepC } = inputData;
    return { text: stepB.text + stepC.text };
  })

  // Conditional branching based on text length
  .branch([
    [async ({ inputData: { text } }) => text.length <= 10, shortTextStep],
    [async ({ inputData: { text } }) => text.length > 10, longTextStep],
  ])

  // Map the branch result back to a single text field
  .map(async ({ inputData }) => {
    // The branch step returns either short-text or long-text result
    const result = inputData['short-text'] || inputData['long-text'];
    return { text: result.text };
  })

  // Nested workflow
  .then(nestedTextProcessor)

  // doUntil loop - continues until text has 20+ characters
  .dountil(addLetterWithCountStep, async ({ inputData: { text } }) => text.length >= 20)

  // Final step
  .then(finalStep)
  .commit();
