/**
 * Network Completion Scorers
 *
 * Completion checks are just MastraScorers that return 0 (failed) or 1 (passed).
 * This unifies completion checking with the evaluation system.
 *
 * @example
 * ```typescript
 * import { createScorer } from '@mastra/core/evals';
 *
 * // Simple completion scorer
 * const testsScorer = createScorer({
 *   id: 'tests',
 *   description: 'Run unit tests',
 * }).generateScore(async ({ run }) => {
 *   const result = await exec('npm test');
 *   return result.exitCode === 0 ? 1 : 0;
 * });
 *
 * // Use in network
 * await agent.network(messages, {
 *   completion: {
 *     scorers: [testsScorer],
 *   },
 * });
 * ```
 */

import { z } from 'zod';

import type { MastraDBMessage, Agent } from '../../agent';
import type { StructuredOutputOptions } from '../../agent/types';
import type { MastraScorer } from '../../evals/base';
import { ChunkFrom } from '../../stream';
import type { InferSchemaOutput, OutputSchema } from '../../stream/base/schema';
import type { NetworkChunkType } from '../../stream/types';

// ============================================================================
// Core Types
// ============================================================================

/**
 * Runtime context passed to completion scoring.
 * Available via run.input when using a completion scorer.
 */
export interface CompletionContext {
  /** Current iteration number (1-based) */
  iteration: number;
  /** Maximum iterations allowed */
  maxIterations?: number;
  /** All messages in the conversation thread */
  messages: MastraDBMessage[];
  /** The original task/prompt that started this network run */
  originalTask: string;
  /** Which primitive was selected this iteration */
  selectedPrimitive: {
    id: string;
    type: 'agent' | 'workflow' | 'tool' | 'none';
  };
  /** The prompt/input sent to the selected primitive */
  primitivePrompt: string;
  /** Result from the primitive execution */
  primitiveResult: string;
  /** Name of the network/routing agent */
  networkName: string;
  /** ID of the current run */
  runId: string;
  /** Current thread ID (if using memory) */
  threadId?: string;
  /** Resource ID (if using memory) */
  resourceId?: string;
  /** Custom context from the request */
  customContext?: Record<string, unknown>;
}

/**
 * Result of running a single scorer.
 * Scorers just evaluate pass/fail - they don't generate the final result.
 */
export interface ScorerResult {
  /** The score (0 = failed, 1 = passed) */
  score: number;
  /** Whether this scorer passed (score === 1) */
  passed: boolean;
  /** Reason from the scorer (why it passed/failed) */
  reason?: string;
  /** Scorer ID */
  scorerId: string;
  /** Scorer name */
  scorerName: string;
  /** Duration in ms */
  duration: number;
  /** Final result generated by the scorer (if any) */
  finalResult?: string;
}

/**
 * Configuration for network completion.
 */
export interface CompletionConfig {
  /**
   * Scorers to run to determine if the task is complete.
   * Each scorer should return 0 (not complete) or 1 (complete).
   *
   * @example
   * ```typescript
   * completion: {
   *   scorers: [testsScorer, buildScorer],
   * }
   * ```
   */
  scorers?: MastraScorer<any, any, any, any>[];

  /**
   * How to combine scorer results:
   * - 'all': All scorers must pass (score = 1) (default)
   * - 'any': At least one scorer must pass
   */
  strategy?: 'all' | 'any';

  /**
   * Maximum time for all scorers (ms)
   * Default: 600000 (10 minutes)
   */
  timeout?: number;

  /**
   * Run scorers in parallel (default: true)
   */
  parallel?: boolean;

  /**
   * Called after scorers run with results
   */
  onComplete?: (results: CompletionRunResult) => void | Promise<void>;
}

/**
 * Result of running completion checks.
 *
 * Completion checks just evaluate "is this done?" - they don't generate the final result.
 * The final result comes from the agent network's primitives.
 */
export interface CompletionRunResult {
  /** Whether the task is complete (based on strategy) */
  complete: boolean;
  /** Reason for completion/failure */
  completionReason?: string;
  /** Individual scorer results */
  scorers: ScorerResult[];
  /** Total duration of all checks */
  totalDuration: number;
  /** Whether checks timed out */
  timedOut: boolean;
}

// Legacy type aliases for backwards compatibility
/** @deprecated Use CompletionContext instead */
export type CheckContext = CompletionContext;
/** @deprecated Use CompletionConfig instead */
export type NetworkValidationConfig = CompletionConfig;
/** @deprecated Use CompletionRunResult instead */
export type CheckRunResult = CompletionRunResult;
/** @deprecated Use CompletionRunResult instead */
export type ValidationRunResult = CompletionRunResult;

// ============================================================================
// Scorer Runner
// ============================================================================

/**
 * Run a single scorer and return the result.
 *
 * Scorers receive:
 * - `run.input` - CompletionContext with all network state
 * - `run.output` - The primitive's result (what we're evaluating)
 * - `run.runId` - The network run ID
 * - `run.requestContext` - Custom context from the request
 */
async function runSingleScorer(
  scorer: MastraScorer<any, any, any, any>,
  context: CompletionContext,
): Promise<ScorerResult> {
  const start = Date.now();

  try {
    const result = await scorer.run({
      runId: context.runId,
      input: context,
      output: context.primitiveResult,
      requestContext: context.customContext,
    });

    const score = typeof result.score === 'number' ? result.score : 0;
    const reason = typeof result.reason === 'string' ? result.reason : undefined;

    return {
      score,
      passed: score === 1,
      reason,
      scorerId: scorer.id,
      scorerName: scorer.name ?? scorer.id,
      duration: Date.now() - start,
    };
  } catch (error: any) {
    return {
      score: 0,
      passed: false,
      reason: `Scorer threw an error: ${error.message}`,
      scorerId: scorer.id,
      scorerName: scorer.name ?? scorer.id,
      duration: Date.now() - start,
    };
  }
}

/**
 * Runs all completion scorers according to the configuration
 */
export async function runCompletionScorers(
  scorers: MastraScorer<any, any, any, any>[],
  context: CompletionContext,
  options?: {
    strategy?: 'all' | 'any';
    parallel?: boolean;
    timeout?: number;
  },
): Promise<CompletionRunResult> {
  const strategy = options?.strategy ?? 'all';
  const parallel = options?.parallel ?? true;
  const timeout = options?.timeout ?? 600000;

  const startTime = Date.now();
  const results: ScorerResult[] = [];
  let timedOut = false;

  const timeoutPromise = new Promise<'timeout'>(resolve => {
    setTimeout(() => resolve('timeout'), timeout);
  });

  if (parallel) {
    const scorerPromises = scorers.map(scorer => runSingleScorer(scorer, context));
    const raceResult = await Promise.race([Promise.all(scorerPromises), timeoutPromise]);

    if (raceResult === 'timeout') {
      timedOut = true;
      const settledResults = await Promise.allSettled(scorerPromises);
      for (const settled of settledResults) {
        if (settled.status === 'fulfilled') {
          results.push(settled.value);
        }
      }
    } else {
      results.push(...raceResult);
    }
  } else {
    for (const scorer of scorers) {
      if (Date.now() - startTime > timeout) {
        timedOut = true;
        break;
      }

      const result = await runSingleScorer(scorer, context);
      results.push(result);

      // Short-circuit
      if (strategy === 'all' && !result.passed) break;
      if (strategy === 'any' && result.passed) break;
    }
  }

  const complete =
    strategy === 'all'
      ? results.length === scorers.length && results.every(r => r.passed)
      : results.some(r => r.passed);

  // Get reason from first passing scorer (or first failing if none passed)
  const relevantScorer = results.find(r => r.passed) || results[0];
  const completionReason = relevantScorer?.reason;

  return {
    complete,
    completionReason,
    scorers: results,
    totalDuration: Date.now() - startTime,
    timedOut,
  };
}

// Legacy function aliases
/** @deprecated Use runCompletionScorers instead */
export async function runChecks(
  scorers: MastraScorer<any, any, any, any>[],
  context: CompletionContext,
  options?: { strategy?: 'all' | 'any'; parallel?: boolean; timeout?: number },
): Promise<CompletionRunResult> {
  return runCompletionScorers(scorers, context, options);
}

/** @deprecated Use runCompletionScorers instead */
export async function runValidation(
  config: CompletionConfig,
  context: CompletionContext,
): Promise<CompletionRunResult> {
  const result = await runCompletionScorers(config.scorers || [], context, {
    strategy: config.strategy,
    parallel: config.parallel,
    timeout: config.timeout,
  });
  await config.onComplete?.(result);
  return result;
}

/**
 * Formats scorer results into a message for the LLM
 */
export function formatCompletionFeedback(result: CompletionRunResult, maxIterationReached: boolean): string {
  const lines: string[] = [];

  lines.push('#### Completion Check Results');
  lines.push('');
  lines.push(`Overall: ${result.complete ? '✅ COMPLETE' : '❌ NOT COMPLETE'}`);
  lines.push(`Duration: ${result.totalDuration}ms`);
  if (result.timedOut) {
    lines.push('⚠️ Scoring timed out');
  }
  lines.push('');

  for (const scorer of result.scorers) {
    lines.push(`###### ${scorer.scorerName} (${scorer.scorerId})`);
    lines.push(`Score: ${scorer.score} ${scorer.passed ? '✅' : '❌'}`);
    if (scorer.reason) {
      lines.push(`Reason: ${scorer.reason}`);
    }
    lines.push('');
  }

  if (result.complete) {
    lines.push('\n\n✅ The task is complete.');
  } else if (maxIterationReached) {
    lines.push('\n\n⚠️ Max iterations reached.');
  } else {
    lines.push('\n\n🔄 Will continue working on the task.');
  }

  return lines.join('\n');
}

// Legacy alias
/** @deprecated Use formatCompletionFeedback instead */
export const formatCheckFeedback = formatCompletionFeedback;
/** @deprecated Use formatCompletionFeedback instead */
export const formatValidationFeedback = formatCompletionFeedback;

// ============================================================================
// Default LLM Completion Scorer
// ============================================================================

/**
 * Schema for the default LLM completion response
 */
const defaultCompletionSchema = z.object({
  isComplete: z.boolean().describe('Whether the task is complete'),
  completionReason: z.string().describe('Explanation of why the task is or is not complete'),
  finalResult: z
    .string()
    .optional()
    .describe('The final result text to return to the user. omit if primitive result is sufficient'),
});

/**
 * Runs the default LLM completion check.
 * Just evaluates "is this done?" - does NOT generate the final result.
 *
 * @internal Used by the network loop when no scorers are configured
 */
export async function runDefaultCompletionCheck(
  agent: Agent,
  context: CompletionContext,
  streamContext?: {
    writer?: { write: (chunk: NetworkChunkType) => Promise<void> };
    stepId?: string;
    runId?: string;
  },
): Promise<ScorerResult> {
  const start = Date.now();

  // Build compact list of completed primitives from network messages
  const completedPrimitives = context.messages
    .map(m => {
      try {
        if (typeof m.content === 'string') return null;

        const text = m.content.parts?.[0]?.type === 'text' ? m.content.parts?.[0]?.text : null;

        if (text?.includes('"isNetwork":true')) {
          const parsed = JSON.parse(text);
          if (parsed.isNetwork) {
            return `${parsed.primitiveType} "${parsed.primitiveId}"`;
          }
        }
      } catch {
        // Ignore parse errors
      }
      return null;
    })
    .filter(Boolean);

  const completedSection =
    completedPrimitives.length > 0 ? `\n\nPrimitives already executed: ${completedPrimitives.join(', ')}` : '';

  const completionPrompt = `
    The ${context.selectedPrimitive.type} ${context.selectedPrimitive.id} has contributed to the task.
    This is the result: ${JSON.stringify(context.primitiveResult)}
    
    ${completedSection}

    You need to evaluate if the task is complete. Pay very close attention to the SYSTEM INSTRUCTIONS for when the task is considered complete. 
    Only return true if the task is complete according to the system instructions.
    Original task: ${context.originalTask}

    If no primitive (type = 'none'), the task is complete because we can't run any primitive to further task completion.

    Also, if the ${context.selectedPrimitive.type} ${context.selectedPrimitive.id} has declined the tool call in its response, then the task is complete as the primitive tool-call was declined by the user.

    IMPORTANT: If the above result is from an AGENT PRIMITIVE and it is a suitable final result itself considering the original task, then finalResult should be an empty string or undefined.
    
    If the task is complete and the result is not from an AGENT PRIMITIVE, always generate a finalResult.
    IF the task is complete and the result is from an AGENT PRIMITIVE, but the AGENT PRIMITIVE response is not comprehensive enough to accomplish the user's original task, then generate a finalResult.

    IMPORTANT: The generated finalResult should not be the exact primitive result. You should craft a comprehensive response based on the message history.
    The finalResult field should be written in natural language.

    You must return this JSON shape:
    {
      "isComplete": boolean,
      "completionReason": string,
      "finalResult": string,
    }
  `;

  try {
    const stream = await agent.stream(completionPrompt, {
      maxSteps: 1,
      structuredOutput: {
        schema: defaultCompletionSchema,
      },
    });

    let currentText = '';
    let currentTextIdx = 0;

    const { writer, stepId, runId: streamRunId } = streamContext ?? {};
    const canStream = writer && stepId && streamRunId;

    if (canStream) {
      await writer.write({
        type: 'routing-agent-text-start',
        payload: { runId: stepId },
        from: ChunkFrom.NETWORK,
        runId: streamRunId,
      });
    }

    for await (const chunk of stream.objectStream) {
      if (chunk?.finalResult) {
        currentText = chunk.finalResult;
      }

      if (canStream) {
        const currentSlice = currentText.slice(currentTextIdx);
        if (chunk?.isComplete && currentSlice.length) {
          await writer.write({
            type: 'routing-agent-text-delta',
            payload: { text: currentSlice },
            from: ChunkFrom.NETWORK,
            runId: streamRunId,
          });
          currentTextIdx = currentText.length;
        }
      }
    }

    const result = await stream.getFullOutput();

    const output = result.object;

    return {
      score: output?.isComplete ? 1 : 0,
      passed: output?.isComplete ?? false,
      reason: output?.completionReason,
      finalResult: output?.finalResult,
      scorerId: 'default-completion',
      scorerName: 'Default LLM Completion',
      duration: Date.now() - start,
    };
  } catch (error: any) {
    return {
      score: 0,
      passed: false,
      reason: `LLM completion check failed: ${error.message}`,
      scorerId: 'default-completion',
      scorerName: 'Default LLM Completion',
      duration: Date.now() - start,
    };
  }
}

// ============================================================================
// Final Result Generation (for use after custom scorers pass)
// ============================================================================

/**
 * Schema for generating only the final result
 */
const finalResultSchema = z.object({
  finalResult: z
    .string()
    .optional()
    .describe('The final result text to return to the user, omit if primitive result is sufficient'),
});

/**
 * Generates and streams the final result after custom scorers have passed.
 * Unlike runDefaultCompletionCheck, this doesn't evaluate completion - it only generates the result.
 *
 * @internal Used by the network loop after custom scorers pass
 */
export async function generateFinalResult(
  agent: Agent,
  context: CompletionContext,
  streamContext?: {
    writer?: { write: (chunk: NetworkChunkType) => Promise<void> };
    stepId?: string;
    runId?: string;
  },
): Promise<string | undefined> {
  const prompt = `
    The task has been completed successfully.
    Original task: ${context.originalTask}

    The ${context.selectedPrimitive.type} ${context.selectedPrimitive.id} produced this result:
    ${JSON.stringify(context.primitiveResult)}

    IMPORTANT: If the above result is from an AGENT PRIMITIVE and it is a suitable final result itself considering the original task, then finalResult should be an empty string or undefined.
    You should evaluate if the above result is comprehensive enough to accomplish the user's original task.
    Otherwise, generate the finalResult object. If the result is not from an AGENT PRIMITIVE, always generate a finalResult.

    The generated finalResult should not be the exact primitive result. You should craft a comprehensive response based on the message history.
    The response should be written in natural language.

    Return JSON:
    {
      "finalResult": string,
    }
  `;

  const stream = await agent.stream(prompt, {
    maxSteps: 1,
    structuredOutput: { schema: finalResultSchema },
  });

  let currentText = '';
  let currentTextIdx = 0;

  const { writer, stepId, runId: streamRunId } = streamContext ?? {};
  const canStream = writer && stepId && streamRunId;

  if (canStream) {
    await writer.write({
      type: 'routing-agent-text-start',
      payload: { runId: stepId },
      from: ChunkFrom.NETWORK,
      runId: streamRunId,
    });
  }

  for await (const chunk of stream.objectStream) {
    if (chunk?.finalResult) {
      currentText = chunk.finalResult;
    }

    if (canStream) {
      const currentSlice = currentText.slice(currentTextIdx);
      if (currentSlice.length) {
        await writer.write({
          type: 'routing-agent-text-delta',
          payload: { text: currentSlice },
          from: ChunkFrom.NETWORK,
          runId: streamRunId,
        });
        currentTextIdx = currentText.length;
      }
    }
  }

  const result = await stream.getFullOutput();
  return result.object?.finalResult;
}

/**
 * Result type for structured final result generation
 */
export interface StructuredFinalResult<OUTPUT extends OutputSchema = undefined> {
  /** Text result (for backward compatibility) */
  text?: string;
  /** Structured object result when user schema is provided */
  object?: InferSchemaOutput<OUTPUT>;
}

/**
 * Generates a structured final result using the user-provided schema.
 * This is called when the network has structuredOutput option configured.
 *
 * @internal Used by the network loop when structuredOutput is provided
 */
export async function generateStructuredFinalResult<OUTPUT extends OutputSchema = undefined>(
  agent: Agent,
  context: CompletionContext,
  structuredOutputOptions: StructuredOutputOptions<OUTPUT>,
  streamContext?: {
    writer?: { write: (chunk: NetworkChunkType) => Promise<void> };
    stepId?: string;
    runId?: string;
  },
): Promise<StructuredFinalResult<OUTPUT>> {
  const prompt = `
    The task has been completed successfully.
    Original task: ${context.originalTask}

    The ${context.selectedPrimitive.type} ${context.selectedPrimitive.id} produced this result:
    ${JSON.stringify(context.primitiveResult)}

    Based on the task and result above, generate a structured response according to the provided schema.
    Use the conversation history and primitive results to craft the response.
  `;

  // Cast structuredOutputOptions to the expected type - OUTPUT already extends OutputSchema
  // so the conditional OUTPUT extends OutputSchema ? OUTPUT : never evaluates to OUTPUT
  const stream = await agent.stream(prompt, {
    maxSteps: 1,
    structuredOutput: structuredOutputOptions as StructuredOutputOptions<OUTPUT extends OutputSchema ? OUTPUT : never>,
  });

  const { writer, stepId, runId: streamRunId } = streamContext ?? {};
  const canStream = writer && stepId && streamRunId;

  // Stream partial objects via network-object chunks
  for await (const partialObject of stream.objectStream) {
    if (canStream && partialObject) {
      // Cast via unknown because the generic OUTPUT is opaque at this point
      await writer.write({
        type: 'network-object',
        payload: { object: partialObject },
        from: ChunkFrom.NETWORK,
        runId: streamRunId,
      } as unknown as NetworkChunkType);
    }
  }

  const result = await stream.getFullOutput();
  const finalObject = result.object as InferSchemaOutput<OUTPUT> | undefined;

  // Emit final object-result chunk
  if (canStream && finalObject) {
    // Cast via unknown because the generic OUTPUT is opaque at this point
    await writer.write({
      type: 'network-object-result',
      payload: { object: finalObject },
      from: ChunkFrom.NETWORK,
      runId: streamRunId,
    } as unknown as NetworkChunkType);
  }

  return {
    text: finalObject ? JSON.stringify(finalObject) : undefined,
    object: finalObject,
  };
}

// Re-export for users who want to create custom scorers
export { createScorer } from '../../evals/base';
