type RecordToTuple<T> = {
  [K in keyof T]: [K, T[K]];
}[keyof T][];

/**
 * Reserved key for setting resourceId from middleware.
 * When set in RequestContext, this takes precedence over client-provided values
 * for security (prevents attackers from hijacking another user's memory).
 *
 * @example
 * ```typescript
 * // In your auth middleware:
 * const requestContext = c.get('requestContext');
 * requestContext.set(MASTRA_RESOURCE_ID_KEY, authenticatedUser.id);
 * ```
 */
export const MASTRA_RESOURCE_ID_KEY = 'mastra__resourceId';

/**
 * Reserved key for setting threadId from middleware.
 * When set in RequestContext, this takes precedence over client-provided values
 * for security (prevents attackers from hijacking another user's memory).
 *
 * @example
 * ```typescript
 * // In your auth middleware:
 * const requestContext = c.get('requestContext');
 * requestContext.set(MASTRA_THREAD_ID_KEY, threadId);
 * ```
 */
export const MASTRA_THREAD_ID_KEY = 'mastra__threadId';

export class RequestContext<Values extends Record<string, any> | unknown = unknown> {
  private registry = new Map<string, unknown>();

  constructor(
    iterable?: Values extends Record<string, any>
      ? RecordToTuple<Partial<Values>>
      : Iterable<readonly [string, unknown]>,
  ) {
    this.registry = new Map(iterable);
  }

  /**
   * set a value with strict typing if `Values` is a Record and the key exists in it.
   */
  public set<K extends Values extends Record<string, any> ? keyof Values : string>(
    key: K,
    value: Values extends Record<string, any> ? (K extends keyof Values ? Values[K] : never) : unknown,
  ): void {
    // The type assertion `key as string` is safe because K always extends string ultimately.
    this.registry.set(key as string, value);
  }

  /**
   * Get a value with its type
   */
  public get<
    K extends Values extends Record<string, any> ? keyof Values : string,
    R = Values extends Record<string, any> ? (K extends keyof Values ? Values[K] : never) : unknown,
  >(key: K): R {
    return this.registry.get(key as string) as R;
  }

  /**
   * Check if a key exists in the container
   */
  public has<K extends Values extends Record<string, any> ? keyof Values : string>(key: K): boolean {
    return this.registry.has(key);
  }

  /**
   * Delete a value by key
   */
  public delete<K extends Values extends Record<string, any> ? keyof Values : string>(key: K): boolean {
    return this.registry.delete(key);
  }

  /**
   * Clear all values from the container
   */
  public clear(): void {
    this.registry.clear();
  }

  /**
   * Get all keys in the container
   */
  public keys(): IterableIterator<Values extends Record<string, any> ? keyof Values : string> {
    return this.registry.keys() as IterableIterator<Values extends Record<string, any> ? keyof Values : string>;
  }

  /**
   * Get all values in the container
   */
  public values(): IterableIterator<Values extends Record<string, any> ? Values[keyof Values] : unknown> {
    return this.registry.values() as IterableIterator<
      Values extends Record<string, any> ? Values[keyof Values] : unknown
    >;
  }

  /**
   * Get all entries in the container.
   * Returns a discriminated union of tuples for proper type narrowing when iterating.
   */
  public entries(): IterableIterator<
    Values extends Record<string, any> ? { [K in keyof Values]: [K, Values[K]] }[keyof Values] : [string, unknown]
  > {
    return this.registry.entries() as IterableIterator<
      Values extends Record<string, any> ? { [K in keyof Values]: [K, Values[K]] }[keyof Values] : [string, unknown]
    >;
  }

  /**
   * Get the size of the container
   */
  public size(): number {
    return this.registry.size;
  }

  /**
   * Execute a function for each entry in the container.
   * The callback receives properly typed key-value pairs.
   */
  public forEach<K extends Values extends Record<string, any> ? keyof Values : string>(
    callbackfn: (
      value: Values extends Record<string, any> ? (K extends keyof Values ? Values[K] : unknown) : unknown,
      key: K,
      map: Map<string, unknown>,
    ) => void,
  ): void {
    this.registry.forEach(callbackfn as (value: unknown, key: string, map: Map<string, unknown>) => void);
  }

  /**
   * Custom JSON serialization method
   * Converts the internal Map to a plain object for proper JSON serialization
   */
  public toJSON(): Record<string, any> {
    return Object.fromEntries(this.registry);
  }
}
