import { describe, it, expect, vi, beforeEach, afterEach } from 'vitest';
import { createVirtualDependencies, bundleExternals } from './bundleExternals';
import type { DependencyMetadata } from '../types';
import type { WorkspacePackageInfo } from '../../bundler/workspaceDependencies';
import { tmpdir } from 'node:os';
import { join } from 'node:path';
import { ensureDir, remove, pathExists, writeFile } from 'fs-extra';

// Mock the utilities that bundleExternals depends on
vi.mock('../utils', async importOriginal => {
  const actual = await importOriginal<typeof import('../utils')>();
  return {
    ...actual,
    getCompiledDepCachePath: vi.fn((rootPath: string, fileName: string) =>
      join(rootPath, 'node_modules', '.cache', fileName),
    ),
  };
});

vi.mock('../package-info', () => ({
  getPackageRootPath: vi.fn((pkg: string) => {
    if (pkg.startsWith('@workspace/')) return '/workspace/packages/' + pkg.split('/')[1];
    if (pkg === 'lodash') return '/node_modules/lodash';
    if (pkg === 'react') return '/node_modules/react';
    return null;
  }),
}));

vi.mock('../plugins/esbuild', () => ({
  esbuild: vi.fn(() => ({ name: 'esbuild-mock' })),
}));

vi.mock('../plugins/hono-alias', () => ({
  aliasHono: vi.fn(() => ({ name: 'hono-alias-mock' })),
}));

describe('createVirtualDependencies', () => {
  it('should handle named exports only', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'lodash',
        {
          exports: ['map', 'filter', 'reduce'],
          rootPath: '/node_modules/lodash',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.fileNameToDependencyMap.get('.mastra/.build/lodash')).toBe('lodash');
    expect(result.optimizedDependencyEntries.get('lodash')).toEqual({
      name: '.mastra/.build/lodash',
      virtual: "export { map, filter, reduce } from 'lodash';",
    });
  });

  it('should handle default export only', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'react',
        {
          exports: ['default'],
          rootPath: '/node_modules/react',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.fileNameToDependencyMap.get('.mastra/.build/react')).toBe('react');
    expect(result.optimizedDependencyEntries.get('react')).toEqual({
      name: '.mastra/.build/react',
      virtual: "export { default } from 'react';",
    });
  });

  it('should handle star export only', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        '@types/node',
        {
          exports: ['*'],
          rootPath: '/node_modules/@types/node',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.fileNameToDependencyMap.get('.mastra/.build/@types-node')).toBe('@types/node');
    expect(result.optimizedDependencyEntries.get('@types/node')).toEqual({
      name: '.mastra/.build/@types-node',
      virtual: "export * from '@types/node';",
    });
  });

  it('should handle mixed exports (named + default)', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'axios',
        {
          exports: ['default', 'AxiosError', 'AxiosResponse'],
          rootPath: '/node_modules/axios',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.optimizedDependencyEntries.get('axios')).toEqual({
      name: '.mastra/.build/axios',
      virtual: "export { default, AxiosError, AxiosResponse } from 'axios';",
    });
  });

  it('should handle mixed exports (named + star)', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'utils-lib',
        {
          exports: ['*', 'specificUtil', 'anotherUtil'],
          rootPath: '/node_modules/utils-lib',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.optimizedDependencyEntries.get('utils-lib')).toEqual({
      name: '.mastra/.build/utils-lib',
      virtual: `export * from 'utils-lib';
export { specificUtil, anotherUtil } from 'utils-lib';`,
    });
  });

  it('should handle mixed exports (default + star)', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'full-lib',
        {
          exports: ['*', 'default'],
          rootPath: '/node_modules/full-lib',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.optimizedDependencyEntries.get('full-lib')).toEqual({
      name: '.mastra/.build/full-lib',
      virtual: `export * from 'full-lib';
export { default } from 'full-lib';`,
    });
  });

  it('should handle all export types together', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'complete-lib',
        {
          exports: ['*', 'default', 'namedExport1', 'namedExport2'],
          rootPath: '/node_modules/complete-lib',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.optimizedDependencyEntries.get('complete-lib')).toEqual({
      name: '.mastra/.build/complete-lib',
      virtual: "export * from 'complete-lib';\nexport { default, namedExport1, namedExport2 } from 'complete-lib';",
    });
  });

  it('should handle scoped package names by replacing slashes with dashes', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        '@scope/package',
        {
          exports: ['someExport'],
          rootPath: '/node_modules/@scope/package',
          isWorkspace: false,
        },
      ],
      [
        '@another/deeply/nested/package',
        {
          exports: ['anotherExport'],
          rootPath: '/node_modules/@another/deeply/nested/package',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.fileNameToDependencyMap.get('.mastra/.build/@scope-package')).toBe('@scope/package');
    expect(result.fileNameToDependencyMap.get('.mastra/.build/@another-deeply-nested-package')).toBe(
      '@another/deeply/nested/package',
    );

    expect(result.optimizedDependencyEntries.get('@scope/package')).toEqual({
      name: '.mastra/.build/@scope-package',
      virtual: "export { someExport } from '@scope/package';",
    });
  });

  it('should handle multiple dependencies', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'lodash',
        {
          exports: ['map', 'filter'],
          rootPath: '/node_modules/lodash',
          isWorkspace: false,
        },
      ],
      [
        'react',
        {
          exports: ['default'],
          rootPath: '/node_modules/react',
          isWorkspace: false,
        },
      ],
      [
        '@types/node',
        {
          exports: ['*'],
          rootPath: '/node_modules/@types/node',
          isWorkspace: false,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.fileNameToDependencyMap.size).toBe(3);
    expect(result.optimizedDependencyEntries.size).toBe(3);

    expect(result.optimizedDependencyEntries.get('lodash')?.virtual).toBe("export { map, filter } from 'lodash';");
    expect(result.optimizedDependencyEntries.get('react')?.virtual).toBe("export { default } from 'react';");
    expect(result.optimizedDependencyEntries.get('@types/node')?.virtual).toBe("export * from '@types/node';");
  });

  it('should handle empty exports array', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: null,
      projectRoot: '/',
      outputDir: '/.mastra/.build',
    });

    expect(result.fileNameToDependencyMap.get('empty-lib')).toBeUndefined();
    expect(result.optimizedDependencyEntries.get('empty-lib')).toBeUndefined();
  });

  it('should handle workspace packages (dev)', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        '@workspace/internal-lib',
        {
          exports: ['internalUtil', 'default'],
          rootPath: '/workspace/packages/internal-lib',
          isWorkspace: true,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: '/workspace',
      projectRoot: '/workspace/app',
      outputDir: '/workspace/app/.mastra/.build',
      bundlerOptions: {
        isDev: true,
      },
    });

    const compiledDepCachePath = `packages/internal-lib/node_modules/.cache/@workspace-internal-lib`;
    expect(result.fileNameToDependencyMap.get(compiledDepCachePath)).toBe('@workspace/internal-lib');
    expect(result.optimizedDependencyEntries.get('@workspace/internal-lib')).toEqual({
      name: compiledDepCachePath,
      virtual: "export { internalUtil, default } from '@workspace/internal-lib';",
    });
  });

  it('should handle workspace packages (build)', () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        '@workspace/internal-lib',
        {
          exports: ['internalUtil', 'default'],
          rootPath: '/workspace/packages/internal-lib',
          isWorkspace: true,
        },
      ],
    ]);

    const result = createVirtualDependencies(depsToOptimize, {
      workspaceRoot: '/workspace',
      projectRoot: '/workspace/app',
      outputDir: '/workspace/app/.mastra/.build',
      bundlerOptions: {
        isDev: false,
      },
    });

    const entryName = result.optimizedDependencyEntries.get('@workspace/internal-lib')?.name;
    expect(entryName).not.toContain('node_modules/.cache');
    expect(entryName).toBe('app/.mastra/.build/@workspace-internal-lib');

    expect(result.fileNameToDependencyMap.get('app/.mastra/.build/@workspace-internal-lib')).toBe(
      '@workspace/internal-lib',
    );

    expect(result.optimizedDependencyEntries.get('@workspace/internal-lib')).toEqual({
      name: 'app/.mastra/.build/@workspace-internal-lib',
      virtual: "export { internalUtil, default } from '@workspace/internal-lib';",
    });
  });
});

describe('bundleExternals', () => {
  let testDir: string;

  beforeEach(async () => {
    testDir = join(tmpdir(), 'bundleExternals-test-' + Date.now());
    await ensureDir(testDir);
  });

  afterEach(async () => {
    if (await pathExists(testDir)) {
      await remove(testDir);
    }
  });

  async function createWorkspacePackageJson(packagePath: string, packageName: string) {
    await ensureDir(packagePath);
    await writeFile(
      join(packagePath, 'package.json'),
      JSON.stringify({
        name: packageName,
        version: '1.0.0',
        main: 'index.js',
      }),
    );
  }

  it('should bundle dependencies and return correct structure', async () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'lodash',
        {
          exports: ['map', 'filter'],
          rootPath: '/node_modules/lodash',
          isWorkspace: false,
        },
      ],
    ]);

    const result = await bundleExternals(depsToOptimize, testDir, {
      projectRoot: testDir,
    });

    // Verify return structure
    expect(result).toHaveProperty('output');
    expect(result).toHaveProperty('fileNameToDependencyMap');
    expect(result).toHaveProperty('usedExternals');

    // Verify output is an array of Rollup output chunks
    expect(Array.isArray(result.output)).toBe(true);
    // TODO fix why it's not always 4
    expect(result.output.length).greaterThan(1);

    // Verify file mapping - the key format depends on the internal logic
    expect(result.fileNameToDependencyMap).toBeInstanceOf(Map);
    expect(result.fileNameToDependencyMap.size).toBe(1);
    const mappingEntries = Array.from(result.fileNameToDependencyMap.entries());
    expect(mappingEntries[0][1]).toBe('lodash');

    // Verify usedExternals is a plain object
    expect(typeof result.usedExternals).toBe('object');
    expect(result.usedExternals).not.toBeInstanceOf(Map);
  });

  it('should handle different bundler options configurations', async () => {
    await writeFile(
      join(testDir, 'package.json'),
      JSON.stringify({
        name: 'react',
        version: '18.0.0',
        main: 'index.js',
      }),
    );

    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'react',
        {
          exports: ['default', 'useState'],
          rootPath: '/node_modules/react',
          isWorkspace: false,
        },
      ],
    ]);

    // Test with custom externals and transpilePackages
    const result = await bundleExternals(depsToOptimize, testDir, {
      projectRoot: testDir,
      bundlerOptions: {
        externals: ['custom-external'],
        transpilePackages: ['some-package'],
        isDev: true,
      },
    });

    expect(result.output).toBeDefined();
    expect(result.fileNameToDependencyMap.size).toBe(1);
    expect(Array.from(result.fileNameToDependencyMap.values())[0]).toBe('react');

    // Test with minimal options
    const result2 = await bundleExternals(depsToOptimize, testDir, {});

    expect(result2.output).toBeDefined();
    expect(result2.fileNameToDependencyMap).toBeInstanceOf(Map);
    expect(result2.fileNameToDependencyMap.size).toBe(1);
  });

  it('should handle isDev: false explicitly and use standard bundling behavior', async () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        'react',
        {
          exports: ['default', 'useState'],
          rootPath: '/node_modules/react',
          isWorkspace: false,
        },
      ],
    ]);

    const result = await bundleExternals(depsToOptimize, testDir, {
      projectRoot: testDir,
      bundlerOptions: {
        isDev: false,
        externals: ['some-external'],
        transpilePackages: ['some-package'],
      },
    });

    expect(result.output).toBeDefined();
    expect(result.fileNameToDependencyMap.size).toBe(1);
    expect(Array.from(result.fileNameToDependencyMap.values())[0]).toBe('react');

    const chunks = result.output.filter(o => o.type === 'chunk');
    chunks.forEach(chunk => {
      expect(chunk.fileName).toMatch(/\.mjs$/);
    });

    expect(typeof result.usedExternals).toBe('object');
    expect(result.usedExternals).not.toBeInstanceOf(Map);
  });

  it('should handle workspace packages with isDev: false', async () => {
    const workspaceMap = new Map<string, WorkspacePackageInfo>([
      [
        '@workspace/utils',
        {
          location: join(testDir, 'packages', 'utils'),
          dependencies: {},
          version: '1.0.0',
        },
      ],
    ]);

    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        '@workspace/utils',
        {
          exports: ['helper', 'default'],
          rootPath: join(testDir, 'packages', 'utils'),
          isWorkspace: true,
        },
      ],
    ]);

    const result = await bundleExternals(depsToOptimize, testDir, {
      workspaceRoot: testDir,
      projectRoot: join(testDir, 'app'),
      workspaceMap,
      bundlerOptions: {
        isDev: false,
      },
    });

    expect(result.output).toBeDefined();
    expect(result.fileNameToDependencyMap).toBeInstanceOf(Map);

    const fileNames = Array.from(result.fileNameToDependencyMap.keys());
    fileNames.forEach(fileName => {
      expect(fileName).not.toContain('node_modules/.cache');
    });

    const dependencyValues = Array.from(result.fileNameToDependencyMap.values());
    expect(dependencyValues).toContain('@workspace/utils');
  });

  it('should handle workspace packages correctly', async () => {
    await createWorkspacePackageJson(join(testDir, 'packages', 'utils'), '@workspace/utils');
    await writeFile(
      join(testDir, 'package.json'),
      JSON.stringify({
        name: 'lodash',
        version: '4.17.21',
        main: 'index.js',
      }),
    );

    const workspaceMap = new Map<string, WorkspacePackageInfo>([
      [
        '@workspace/utils',
        {
          location: join(testDir, 'packages', 'utils'),
          dependencies: {},
          version: '1.0.0',
        },
      ],
    ]);

    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        '@workspace/utils',
        {
          exports: ['helper', 'default'],
          rootPath: join(testDir, 'packages', 'utils'),
          isWorkspace: true,
        },
      ],
      [
        'lodash',
        {
          exports: ['map'],
          rootPath: '/node_modules/lodash',
          isWorkspace: false,
        },
      ],
    ]);

    const result = await bundleExternals(depsToOptimize, testDir, {
      workspaceRoot: testDir,
      projectRoot: join(testDir, 'app'),
      workspaceMap,
      bundlerOptions: {
        isDev: true,
      },
    });

    expect(result.output).toBeDefined();
    expect(result.fileNameToDependencyMap).toBeInstanceOf(Map);
    expect(result.fileNameToDependencyMap.size).toBe(3);

    // Check that both workspace and external packages are handled
    const dependencyValues = Array.from(result.fileNameToDependencyMap.values());
    expect(dependencyValues).toContain('@workspace/utils');
    expect(dependencyValues).toContain('lodash');
  });

  it('should validate output structure and file patterns', async () => {
    const depsToOptimize = new Map<string, DependencyMetadata>([
      [
        '@scoped/package',
        {
          exports: ['namedExport'],
          rootPath: '/node_modules/@scoped/package',
          isWorkspace: false,
        },
      ],
      [
        'regular-package',
        {
          exports: ['*'],
          rootPath: '/node_modules/regular-package',
          isWorkspace: false,
        },
      ],
    ]);

    const result = await bundleExternals(depsToOptimize, testDir, {
      projectRoot: testDir,
    });

    // Validate all output chunks have .mjs extension
    const chunks = result.output.filter(o => o.type === 'chunk');
    chunks.forEach(chunk => {
      expect(chunk.fileName).toMatch(/\.mjs$/);
    });

    // Validate file mapping structure - check values instead of specific keys
    const mappingValues = Array.from(result.fileNameToDependencyMap.values());
    expect(mappingValues).toContain('@scoped/package');
    expect(mappingValues).toContain('regular-package');
    expect(result.fileNameToDependencyMap.size).toBe(2);
  });

  it('should handle edge cases gracefully', async () => {
    // Test with dependency that has no root path
    const depsWithNullPath = new Map<string, DependencyMetadata>([
      [
        'unknown-package',
        {
          exports: ['something'],
          rootPath: null,
          isWorkspace: false,
        },
      ],
    ]);

    const nullPathResult = await bundleExternals(depsWithNullPath, testDir, {
      projectRoot: testDir,
    });

    expect(nullPathResult.output).toBeDefined();
    expect(nullPathResult.fileNameToDependencyMap.size).toBe(1);
    expect(Array.from(nullPathResult.fileNameToDependencyMap.values())[0]).toBe('unknown-package');

    // Test with mixed workspace and non-workspace dependencies using testDir
    const mixedDeps = new Map<string, DependencyMetadata>([
      [
        'external-lib',
        {
          exports: ['default'],
          rootPath: '/node_modules/external-lib',
          isWorkspace: false,
        },
      ],
    ]);

    const mixedResult = await bundleExternals(mixedDeps, testDir, {
      workspaceRoot: testDir,
      projectRoot: join(testDir, 'app'),
      workspaceMap: new Map([
        [
          '@workspace/internal',
          {
            location: join(testDir, 'packages', 'internal'),
            dependencies: {},
            version: '1.0.0',
          },
        ],
      ]),
    });

    expect(mixedResult.output).toBeDefined();
    expect(mixedResult.fileNameToDependencyMap.size).toBe(1);

    // Test bundler options structure validation
    const optionsTestResult = await bundleExternals(
      new Map([['test-pkg', { exports: ['test'], rootPath: null, isWorkspace: false }]]),
      testDir,
      {
        bundlerOptions: {
          externals: ['test'],
          transpilePackages: ['pkg'],
          isDev: false,
        },
      },
    );

    expect(optionsTestResult.output).toBeDefined();
  });

  describe('externals: true behavior', () => {
    it('should remove non-workspace deps from bundling and add to usedExternals', async () => {
      const depsToOptimize = new Map<string, DependencyMetadata>([
        [
          'lodash',
          {
            exports: ['map', 'filter'],
            rootPath: '/node_modules/lodash',
            isWorkspace: false,
          },
        ],
        [
          'axios',
          {
            exports: ['default'],
            rootPath: '/node_modules/axios',
            isWorkspace: false,
          },
        ],
      ]);

      const result = await bundleExternals(depsToOptimize, testDir, {
        projectRoot: testDir,
        bundlerOptions: {
          externals: true,
        },
      });

      // Non-workspace deps should be removed from depsToOptimize (no bundling)
      expect(depsToOptimize.size).toBe(0);

      // Should have no bundled output chunks (everything is external)
      const chunks = result.output.filter(o => o.type === 'chunk');
      expect(chunks.length).toBe(0);

      // usedExternals should contain the non-workspace deps
      const externalEntries = Object.entries(result.usedExternals);
      expect(externalEntries.length).toBeGreaterThan(0);

      // Find the synthetic entry that contains our externals
      const syntheticEntry = externalEntries.find(([_key, value]) => {
        return typeof value === 'object' && ('lodash' in value || 'axios' in value);
      });

      expect(syntheticEntry).toBeDefined();
      const [_path, externalsMap] = syntheticEntry!;
      expect(externalsMap['lodash']).toBe('/node_modules/lodash');
      expect(externalsMap['axios']).toBe('/node_modules/axios');
    });

    it('should still bundle workspace packages when externals: true', async () => {
      await createWorkspacePackageJson(join(testDir, 'packages', 'utils'), '@workspace/utils');

      const workspaceMap = new Map<string, WorkspacePackageInfo>([
        [
          '@workspace/utils',
          {
            location: join(testDir, 'packages', 'utils'),
            dependencies: {},
            version: '1.0.0',
          },
        ],
      ]);

      const depsToOptimize = new Map<string, DependencyMetadata>([
        [
          '@workspace/utils',
          {
            exports: ['helper', 'default'],
            rootPath: join(testDir, 'packages', 'utils'),
            isWorkspace: true,
          },
        ],
        [
          'lodash',
          {
            exports: ['map'],
            rootPath: '/node_modules/lodash',
            isWorkspace: false,
          },
        ],
      ]);

      const result = await bundleExternals(depsToOptimize, testDir, {
        workspaceRoot: testDir,
        projectRoot: join(testDir, 'app'),
        workspaceMap,
        bundlerOptions: {
          externals: true,
          isDev: true,
        },
      });

      // Only workspace package should remain in depsToOptimize
      expect(depsToOptimize.size).toBe(1);
      expect(depsToOptimize.has('@workspace/utils')).toBe(true);

      // Should have bundled output for workspace package
      const chunks = result.output.filter(o => o.type === 'chunk');
      expect(chunks.length).toBeGreaterThan(0);

      // Workspace package should be in fileNameToDependencyMap
      const dependencyValues = Array.from(result.fileNameToDependencyMap.values());
      expect(dependencyValues).toContain('@workspace/utils');

      // Non-workspace dep should be in usedExternals
      const externalEntries = Object.entries(result.usedExternals);
      const syntheticEntry = externalEntries.find(([_key, value]) => {
        return typeof value === 'object' && 'lodash' in value;
      });
      expect(syntheticEntry).toBeDefined();
    });

    it('should handle externals: true with only workspace packages', async () => {
      await createWorkspacePackageJson(join(testDir, 'packages', 'utils'), '@workspace/utils');

      const workspaceMap = new Map<string, WorkspacePackageInfo>([
        [
          '@workspace/utils',
          {
            location: join(testDir, 'packages', 'utils'),
            dependencies: {},
            version: '1.0.0',
          },
        ],
      ]);

      const depsToOptimize = new Map<string, DependencyMetadata>([
        [
          '@workspace/utils',
          {
            exports: ['helper'],
            rootPath: join(testDir, 'packages', 'utils'),
            isWorkspace: true,
          },
        ],
      ]);

      const result = await bundleExternals(depsToOptimize, testDir, {
        workspaceRoot: testDir,
        projectRoot: join(testDir, 'app'),
        workspaceMap,
        bundlerOptions: {
          externals: true,
          isDev: false,
        },
      });

      // Workspace package should still be bundled
      expect(depsToOptimize.size).toBe(1);
      expect(depsToOptimize.has('@workspace/utils')).toBe(true);

      // Should have output chunks
      const chunks = result.output.filter(o => o.type === 'chunk');
      expect(chunks.length).toBeGreaterThan(0);

      // Workspace package should be in mapping
      const dependencyValues = Array.from(result.fileNameToDependencyMap.values());
      expect(dependencyValues).toContain('@workspace/utils');
    });

    it('should handle externals: true with only non-workspace packages', async () => {
      const depsToOptimize = new Map<string, DependencyMetadata>([
        [
          'react',
          {
            exports: ['default', 'useState'],
            rootPath: '/node_modules/react',
            isWorkspace: false,
          },
        ],
        [
          'lodash',
          {
            exports: ['map'],
            rootPath: '/node_modules/lodash',
            isWorkspace: false,
          },
        ],
      ]);

      const result = await bundleExternals(depsToOptimize, testDir, {
        projectRoot: testDir,
        bundlerOptions: {
          externals: true,
        },
      });

      // All deps should be removed (nothing to bundle)
      expect(depsToOptimize.size).toBe(0);

      // No bundled chunks
      const chunks = result.output.filter(o => o.type === 'chunk');
      expect(chunks.length).toBe(0);

      // All deps should be in usedExternals
      const externalEntries = Object.entries(result.usedExternals);
      const syntheticEntry = externalEntries.find(([_key, value]) => {
        return typeof value === 'object' && ('react' in value || 'lodash' in value);
      });

      expect(syntheticEntry).toBeDefined();
      const [_path, externalsMap] = syntheticEntry!;
      expect(externalsMap['react']).toBe('/node_modules/react');
      expect(externalsMap['lodash']).toBe('/node_modules/lodash');
    });

    it('should handle externals: true with deps that have null rootPath', async () => {
      const depsToOptimize = new Map<string, DependencyMetadata>([
        [
          'unknown-package',
          {
            exports: ['something'],
            rootPath: null,
            isWorkspace: false,
          },
        ],
        [
          'another-unknown',
          {
            exports: ['other'],
            rootPath: null,
            isWorkspace: false,
          },
        ],
      ]);

      const result = await bundleExternals(depsToOptimize, testDir, {
        projectRoot: testDir,
        bundlerOptions: {
          externals: true,
        },
      });

      // Deps should be removed
      expect(depsToOptimize.size).toBe(0);

      // Should be in usedExternals with package name as fallback
      const externalEntries = Object.entries(result.usedExternals);
      const syntheticEntry = externalEntries.find(([_key, value]) => {
        return typeof value === 'object' && ('unknown-package' in value || 'another-unknown' in value);
      });

      expect(syntheticEntry).toBeDefined();
      const [_path, externalsMap] = syntheticEntry!;
      // When rootPath is null, it falls back to the package name
      expect(externalsMap['unknown-package']).toBe('unknown-package');
      expect(externalsMap['another-unknown']).toBe('another-unknown');
    });

    it('should not affect behavior when externals is an array (existing behavior)', async () => {
      const depsToOptimize = new Map<string, DependencyMetadata>([
        [
          'lodash',
          {
            exports: ['map'],
            rootPath: '/node_modules/lodash',
            isWorkspace: false,
          },
        ],
        [
          'react',
          {
            exports: ['default'],
            rootPath: '/node_modules/react',
            isWorkspace: false,
          },
        ],
      ]);

      const result = await bundleExternals(depsToOptimize, testDir, {
        projectRoot: testDir,
        bundlerOptions: {
          externals: ['custom-external'],
        },
      });

      // When externals is an array, deps should still be bundled
      // (they're not removed from depsToOptimize)
      expect(result.output).toBeDefined();
      expect(result.fileNameToDependencyMap.size).toBe(2);

      const dependencyValues = Array.from(result.fileNameToDependencyMap.values());
      expect(dependencyValues).toContain('lodash');
      expect(dependencyValues).toContain('react');
    });
  });
});
