import { createTool } from '@mastra/core/tools';
import { z } from 'zod';

import { GraphRAG } from '../graph-rag';
import {
  vectorQuerySearch,
  defaultGraphRagDescription,
  filterSchema,
  outputSchema,
  baseSchema,
  coerceTopK,
  parseFilterValue,
  resolveVectorStore,
} from '../utils';
import type { RagTool } from '../utils';
import { convertToSources } from '../utils/convert-sources';
import type { GraphRagToolOptions, ProviderOptions } from './types';
import { defaultGraphOptions } from './types';

export const createGraphRAGTool = (options: GraphRagToolOptions) => {
  const { model, id, description } = options;
  const storeName = options['vectorStoreName'] ? options.vectorStoreName : 'DirectVectorStore';

  const toolId = id || `GraphRAG ${storeName} ${options.indexName} Tool`;
  const toolDescription = description || defaultGraphRagDescription();
  const graphOptions = {
    ...defaultGraphOptions,
    ...(options.graphOptions || {}),
  };
  // Initialize GraphRAG
  const graphRag = new GraphRAG(graphOptions.dimension, graphOptions.threshold);
  let isInitialized = false;

  const inputSchema = options.enableFilter ? filterSchema : z.object(baseSchema).passthrough();

  return createTool({
    id: toolId,
    inputSchema,
    outputSchema,
    description: toolDescription,
    execute: async (inputData, context) => {
      const { requestContext, mastra } = context || {};
      const indexName: string = requestContext?.get('indexName') ?? options.indexName;
      const vectorStoreName: string =
        'vectorStore' in options ? storeName : (requestContext?.get('vectorStoreName') ?? storeName);
      if (!indexName) throw new Error(`indexName is required, got: ${indexName}`);
      if (!vectorStoreName) throw new Error(`vectorStoreName is required, got: ${vectorStoreName}`);
      const includeSources: boolean = requestContext?.get('includeSources') ?? options.includeSources ?? true;
      const randomWalkSteps: number | undefined =
        requestContext?.get('randomWalkSteps') ?? graphOptions.randomWalkSteps;
      const restartProb: number | undefined = requestContext?.get('restartProb') ?? graphOptions.restartProb;
      const topK: number = requestContext?.get('topK') ?? inputData.topK ?? 10;
      const filter: unknown = requestContext?.get('filter') ?? inputData.filter;
      const queryText = inputData.queryText;
      const providerOptions: ProviderOptions['providerOptions'] =
        requestContext?.get('providerOptions') ?? options.providerOptions;

      const enableFilter = !!requestContext?.get('filter') || (options.enableFilter ?? false);

      const logger = mastra?.getLogger();
      if (logger) {
        logger.debug('[GraphRAGTool] execute called with:', { queryText, topK, filter });
      }
      try {
        const topKValue = coerceTopK(topK);

        const vectorStore = await resolveVectorStore(options, { requestContext, mastra, vectorStoreName });
        if (!vectorStore) {
          if (logger) {
            logger.error(`Vector store '${vectorStoreName}' not found`);
          }
          // Return empty results for graceful degradation when store is not found
          return { relevantContext: [], sources: [] };
        }

        const queryFilter = enableFilter && filter ? parseFilterValue(filter, logger) : {};
        if (logger) {
          logger.debug('Prepared vector query parameters:', { queryFilter, topK: topKValue });
        }
        const { results, queryEmbedding } = await vectorQuerySearch({
          indexName,
          vectorStore,
          queryText,
          model,
          queryFilter: Object.keys(queryFilter || {}).length > 0 ? queryFilter : undefined,
          topK: topKValue,
          includeVectors: true,
          providerOptions,
        });
        if (logger) {
          logger.debug('vectorQuerySearch returned results', { count: results.length });
        }

        // Initialize graph if not done yet
        if (!isInitialized) {
          // Get all chunks and embeddings for graph construction
          const chunks = results.map(result => ({
            text: result?.metadata?.text,
            metadata: result.metadata ?? {},
          }));
          const embeddings = results.map(result => ({
            vector: result.vector || [],
          }));

          if (logger) {
            logger.debug('Initializing graph', { chunkCount: chunks.length, embeddingCount: embeddings.length });
          }
          graphRag.createGraph(chunks, embeddings);
          isInitialized = true;
        } else if (logger) {
          logger.debug('Graph already initialized, skipping graph construction');
        }

        // Get reranked results using GraphRAG
        const rerankedResults = graphRag.query({
          query: queryEmbedding,
          topK: topKValue,
          randomWalkSteps,
          restartProb,
        });
        if (logger) {
          logger.debug('GraphRAG query returned results', { count: rerankedResults.length });
        }
        // Extract and combine relevant chunks
        const relevantChunks = rerankedResults.map(result => result.content);
        if (logger) {
          logger.debug('Returning relevant context chunks', { count: relevantChunks.length });
        }
        // `sources` exposes the full retrieval objects
        const sources = includeSources ? convertToSources(rerankedResults) : [];
        return {
          relevantContext: relevantChunks,
          sources,
        };
      } catch (err) {
        if (logger) {
          logger.error('Unexpected error in GraphRAGTool execute', {
            error: err,
            errorMessage: err instanceof Error ? err.message : String(err),
            errorStack: err instanceof Error ? err.stack : undefined,
          });
        }
        return { relevantContext: [], sources: [] };
      }
    },
    // Use any for output schema as the structure of the output causes type inference issues
  }) as RagTool<z.infer<typeof inputSchema>, any>;
};
