import { createSampleScore } from '@internal/storage-test-utils';
import { Agent } from '@mastra/core/agent';
import { RequestContext } from '@mastra/core/di';
import { Mastra } from '@mastra/core/mastra';
import type { ScoresStorage, StoragePagination } from '@mastra/core/storage';
import { InMemoryStore } from '@mastra/core/storage';
import { createWorkflow } from '@mastra/core/workflows';
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { z } from 'zod';
import { HTTPException } from '../http-exception';
import {
  LIST_SCORERS_ROUTE,
  LIST_SCORES_BY_RUN_ID_ROUTE,
  LIST_SCORES_BY_ENTITY_ID_ROUTE,
  SAVE_SCORE_ROUTE,
} from './scores';
import { createTestServerContext } from './test-utils';

function createPagination(args: Partial<StoragePagination>): StoragePagination {
  return {
    page: 0,
    perPage: 10,
    ...args,
  };
}

describe('Scores Handlers', () => {
  let mockStorage: InMemoryStore;
  let scoresStore: ScoresStorage;
  let mastra: Mastra;

  beforeEach(async () => {
    vi.clearAllMocks();
    mockStorage = new InMemoryStore();
    await mockStorage.init();
    scoresStore = (await mockStorage.getStore('scores'))!;

    mastra = new Mastra({
      logger: false,
      storage: mockStorage,
      workflows: {
        'test-workflow': createWorkflow({
          id: 'test-workflow',
          inputSchema: z.object({}),
          outputSchema: z.object({}),
          description: 'test-workflow',
        }).commit(),
      },
      agents: {
        'test-agent': new Agent({
          id: 'test-agent',
          name: 'test-agent',
          instructions: 'test-agent',
          model: {} as any,
        }),
      },
    });
  });

  describe('listScorersHandler', () => {
    it('should return empty object', async () => {
      const result = await LIST_SCORERS_ROUTE.handler({
        ...createTestServerContext({ mastra }),
        requestContext: new RequestContext(),
      });
      expect(result).toEqual({});
    });
  });

  describe('listScoresByRunIdHandler', () => {
    it('should get scores by run ID successfully', async () => {
      const mockScores = [createSampleScore({ scorerId: 'test-1-scorer' })];

      await scoresStore.saveScore(mockScores[0]);

      const pagination = createPagination({ page: 0, perPage: 10 });

      const result = await LIST_SCORES_BY_RUN_ID_ROUTE.handler({
        ...createTestServerContext({ mastra }),
        runId: mockScores?.[0]?.runId,
        page: pagination.page,
        perPage: pagination.perPage as number,
      });

      expect(result.scores).toHaveLength(1);

      expect(result.pagination).toEqual({
        total: 1,
        page: 0,
        perPage: 10,
        hasMore: false,
      });
    });

    it('should return empty array when storage method is not available', async () => {
      const pagination = createPagination({ page: 0, perPage: 10 });

      // Create mastra instance without storage
      const mastraWithoutStorage = new Mastra({
        logger: false,
      });

      const result = await LIST_SCORES_BY_RUN_ID_ROUTE.handler({
        ...createTestServerContext({ mastra: mastraWithoutStorage }),
        runId: 'test-run-1',
        page: pagination.page,
        perPage: pagination.perPage as number,
      });

      expect(result).toEqual({
        pagination: {
          hasMore: false,
          page: 0,
          perPage: 0,
          total: 0,
        },
        scores: [],
      });
    });

    it('should handle storage errors gracefully', async () => {
      const pagination = createPagination({ page: 0, perPage: 10 });
      const error = new Error('Storage error');

      scoresStore.listScoresByRunId = vi.fn().mockRejectedValue(error);

      await expect(
        LIST_SCORES_BY_RUN_ID_ROUTE.handler({
          ...createTestServerContext({ mastra }),
          runId: 'test-run-1',
          page: pagination.page,
          perPage: pagination.perPage as number,
        }),
      ).rejects.toThrow(HTTPException);
    });

    it('should handle API errors with status codes', async () => {
      const pagination = createPagination({ page: 0, perPage: 10 });
      const apiError = {
        message: 'Not found',
        status: 404,
      };

      scoresStore.listScoresByRunId = vi.fn().mockRejectedValue(apiError);

      await expect(
        LIST_SCORES_BY_RUN_ID_ROUTE.handler({
          ...createTestServerContext({ mastra }),
          runId: 'test-run-1',
          page: pagination.page,
          perPage: pagination.perPage as number,
        }),
      ).rejects.toThrow(HTTPException);
    });
  });

  describe('listScoresByEntityIdHandler', () => {
    it('should get scores by entity ID successfully', async () => {
      const mockScores = [createSampleScore({ entityType: 'AGENT', entityId: 'test-agent', scorerId: 'foo-scorer' })];
      const pagination = createPagination({ page: 0, perPage: 10 });

      await scoresStore.saveScore(mockScores[0]);

      const result = await LIST_SCORES_BY_ENTITY_ID_ROUTE.handler({
        ...createTestServerContext({ mastra }),
        entityId: 'test-agent',
        entityType: 'AGENT',
        page: pagination.page,
        perPage: pagination.perPage as number,
      });

      expect(result.scores).toHaveLength(1);

      expect(result.pagination).toEqual({
        total: 1,
        page: 0,
        perPage: 10,
        hasMore: false,
      });
    });

    it('should return empty array when storage method is not available', async () => {
      const pagination = createPagination({ page: 0, perPage: 10 });

      // Create mastra instance without storage
      const mastraWithoutStorage = new Mastra({
        logger: false,
      });

      const result = await LIST_SCORES_BY_ENTITY_ID_ROUTE.handler({
        ...createTestServerContext({ mastra: mastraWithoutStorage }),
        entityId: 'test-agent',
        entityType: 'agent',
        page: pagination.page,
        perPage: pagination.perPage as number,
      });

      expect(result).toEqual({
        pagination: {
          hasMore: false,
          page: 0,
          perPage: 0,
          total: 0,
        },
        scores: [],
      });
    });

    it('should handle storage errors gracefully', async () => {
      const pagination = createPagination({ page: 0, perPage: 10 });
      const error = new Error('Storage error');

      scoresStore.listScoresByEntityId = vi.fn().mockRejectedValue(error);

      await expect(
        LIST_SCORES_BY_ENTITY_ID_ROUTE.handler({
          ...createTestServerContext({ mastra }),
          entityId: 'test-agent',
          entityType: 'agent',
          page: pagination.page,
          perPage: pagination.perPage as number,
        }),
      ).rejects.toThrow(HTTPException);
    });

    it('should handle API errors with status codes', async () => {
      const pagination = createPagination({ page: 0, perPage: 10 });
      const apiError = {
        message: 'Entity not found',
        status: 404,
      };

      scoresStore.listScoresByEntityId = vi.fn().mockRejectedValue(apiError);

      await expect(
        LIST_SCORES_BY_ENTITY_ID_ROUTE.handler({
          ...createTestServerContext({ mastra }),
          entityId: 'test-agent',
          entityType: 'agent',
          page: pagination.page,
          perPage: pagination.perPage as number,
        }),
      ).rejects.toThrow(HTTPException);
    });

    it('should work with different entity types', async () => {
      const mockScores = [
        createSampleScore({ entityType: 'WORKFLOW', entityId: 'test-workflow', scorerId: 'foo-scorer' }),
      ];
      const pagination = createPagination({ page: 0, perPage: 10 });

      await scoresStore.saveScore(mockScores[0]);

      const result = await LIST_SCORES_BY_ENTITY_ID_ROUTE.handler({
        ...createTestServerContext({ mastra }),
        entityId: 'test-workflow',
        entityType: 'WORKFLOW',
        page: pagination.page,
        perPage: pagination.perPage as number,
      });

      expect(result.scores).toHaveLength(1);
      expect(result.pagination).toEqual({
        total: 1,
        page: 0,
        perPage: 10,
        hasMore: false,
      });
    });
  });

  describe('saveScoreHandler', () => {
    it('should save score successfully', async () => {
      const score = createSampleScore({ scorerId: 'new-score-1' });
      const savedScore = { score };

      const result = await SAVE_SCORE_ROUTE.handler({
        ...createTestServerContext({ mastra }),
        score,
      });

      expect(result).toEqual(savedScore);
    });

    it('should throw error when storage method is not available', async () => {
      const score = createSampleScore({ scorerId: 'new-score-1' });

      // Create mastra instance without storage
      const mastraWithoutStorage = new Mastra({
        logger: false,
      });

      await expect(
        SAVE_SCORE_ROUTE.handler({
          ...createTestServerContext({ mastra: mastraWithoutStorage }),
          score,
        }),
      ).rejects.toThrow(HTTPException);
    });

    it('should handle storage errors gracefully', async () => {
      const score = createSampleScore({ scorerId: 'new-score-1' });
      const error = new Error('Storage error');

      scoresStore.saveScore = vi.fn().mockRejectedValue(error);

      await expect(
        SAVE_SCORE_ROUTE.handler({
          ...createTestServerContext({ mastra }),
          score,
        }),
      ).rejects.toThrow(HTTPException);
    });

    it('should handle API errors with status codes', async () => {
      const score = createSampleScore({ scorerId: 'new-score-1' });
      const apiError = {
        message: 'Validation error',
        status: 400,
      };

      scoresStore.saveScore = vi.fn().mockRejectedValue(apiError);

      await expect(
        SAVE_SCORE_ROUTE.handler({
          ...createTestServerContext({ mastra }),
          score,
        }),
      ).rejects.toThrow(HTTPException);
    });

    it('should handle score with all optional fields', async () => {
      const score = createSampleScore({ scorerId: 'test-1-scorer' });

      const savedScore = { score };

      const result = await SAVE_SCORE_ROUTE.handler({
        ...createTestServerContext({ mastra }),
        score,
      });

      expect(result).toEqual(savedScore);
    });
  });
});
