import { Observability, DefaultExporter, CloudExporter, SensitiveDataFilter } from '@mastra/observability';
import { Mastra } from '@mastra/core/mastra';
import { PinoLogger } from '@mastra/loggers';
import { financialModelingAgent } from './agents/financial-modeling-agent';
import { HTTPException } from 'hono/http-exception';
import { LibSQLStore, LibSQLVector } from '@mastra/libsql';
import { Composio } from '@composio/core';
import { MastraProvider } from '@composio/mastra';
import { RequestContext } from '@mastra/core/request-context';

export const mastra = new Mastra({
  agents: { financialModelingAgent },
  storage: new LibSQLStore({
    id: 'mastra-storage',
    url: 'file:../../mastra.db',
  }),
  vectors: {
    default: new LibSQLVector({
      id: 'mastra-vector',
      url: 'file:../../mastra.db',
    }),
  },
  logger: new PinoLogger({
    name: 'Mastra',
    level: process.env.NODE_ENV === 'production' ? 'info' : 'debug',
  }),
  server: {
    build: {
      openAPIDocs: true,
      swaggerUI: true,
      apiReqLogs: true,
    },
    middleware: [
      async (c, next) => {
        const composio = new Composio({
          provider: new MastraProvider(),
        });

        const requestContext = c.get('requestContext') as RequestContext<string | undefined>;

        if (!process.env.COMPOSIO_AUTH_CONFIG_ID)
          throw new HTTPException(500, {
            message: 'COMPOSIO_AUTH_CONFIG_ID missing',
          });

        // TODO: Retrieve unique user id and set it on the request context
        // Consider using Authentication headers for user identification
        // e.g const bearerToken = c.get('Authorization')
        // https://mastra.ai/en/docs/server-db/middleware#common-examples
        const userId = 'unique-user-id';
        requestContext.set('userId', userId);

        // check for active/intiated connection or initiate a new connection to composio
        const connectedAccounts = await composio.connectedAccounts.list({
          authConfigIds: [process.env.COMPOSIO_AUTH_CONFIG_ID],
          userIds: [userId],
        });

        // active connection
        const activeAccount = connectedAccounts.items.find(item => item.status === 'ACTIVE');
        if (activeAccount) {
          requestContext.set('activeAccount', activeAccount);
          return await next();
        }

        // initiated connection
        const initiatedAccount = connectedAccounts.items.find(item => item.status === 'INITIATED');
        if (initiatedAccount && initiatedAccount.data?.redirectUrl) {
          requestContext.set('redirectUrl', initiatedAccount.data.redirectUrl);
          return await next();
        }

        // initiate a new connection to composio
        const connectionRequest = await composio.connectedAccounts.initiate(
          userId,
          process.env.COMPOSIO_AUTH_CONFIG_ID,
        );
        if (connectionRequest.redirectUrl) {
          requestContext.set('redirectUrl', connectionRequest.redirectUrl);
          return await next();
        }

        throw new HTTPException(500, {
          message: 'Could not connect to composio',
        });
      },
    ],
  },
  observability: new Observability({
    configs: {
      default: {
        serviceName: 'mastra',
        exporters: [
          new DefaultExporter(), // Persists traces to storage for Mastra Studio
          new CloudExporter(), // Sends traces to Mastra Cloud (if MASTRA_CLOUD_ACCESS_TOKEN is set)
        ],
        spanOutputProcessors: [
          new SensitiveDataFilter(), // Redacts sensitive data like passwords, tokens, keys
        ],
      },
    },
  }),
});
