﻿// Copyright (c) Microsoft. All rights reserved.

using System.Text;
using Azure.AI.Agents.Persistent;
using Azure.Identity;
using Microsoft.Agents.AI;
using Microsoft.Extensions.AI;
using Microsoft.SemanticKernel;
using Microsoft.SemanticKernel.Agents;
using Microsoft.SemanticKernel.Agents.AzureAI;

#pragma warning disable SKEXP0110 // Type is for evaluation purposes only and is subject to change or removal in future updates. Suppress this diagnostic to proceed.

var azureEndpoint = Environment.GetEnvironmentVariable("AZURE_FOUNDRY_PROJECT_ENDPOINT") ?? throw new InvalidOperationException("AZURE_FOUNDRY_PROJECT_ENDPOINT is not set.");
var deploymentName = System.Environment.GetEnvironmentVariable("AZURE_FOUNDRY_PROJECT_DEPLOYMENT_NAME") ?? "gpt-4o";
var userInput = "Create a python code file using the code interpreter tool with a code ready to determine the values in the Fibonacci sequence that are less then the value of 101";

Console.WriteLine($"User Input: {userInput}");

await SKAgentAsync();
await SKAgent_As_AFAgentAsync();
await AFAgentAsync();

async Task SKAgentAsync()
{
    Console.WriteLine("\n=== SK Agent ===\n");

    var azureAgentClient = AzureAIAgent.CreateAgentsClient(azureEndpoint, new AzureCliCredential());

    PersistentAgent definition = await azureAgentClient.Administration.CreateAgentAsync(deploymentName, tools: [new CodeInterpreterToolDefinition()]);

    AzureAIAgent agent = new(definition, azureAgentClient);
    var thread = new AzureAIAgentThread(azureAgentClient);

    // SK Azure AI Agent provides the code interpreter content and the assistant message as different contents in the call iteration.
    await foreach (var content in agent.InvokeAsync(userInput, thread))
    {
        if (!string.IsNullOrWhiteSpace(content.Message.Content))
        {
            bool isCode = content.Message.Metadata?.ContainsKey(AzureAIAgent.CodeInterpreterMetadataKey) ?? false;
            Console.WriteLine($"\n# {content.Message.Role}{(isCode ? "\n# Generated Code:\n" : ":")}{content.Message.Content}");
        }

        // Check for the citations
        foreach (var item in content.Message.Items)
        {
            // Process each item in the message
            if (item is AnnotationContent annotation)
            {
                if (annotation.Kind != AnnotationKind.UrlCitation)
                {
                    Console.WriteLine($"  [{item.GetType().Name}] {annotation.Label}: File #{annotation.ReferenceId}");
                }
            }
            else if (item is FileReferenceContent fileReference)
            {
                Console.WriteLine($"  [{item.GetType().Name}] File #{fileReference.FileId}");
            }
        }
    }

    // Clean up
    await thread.DeleteAsync();
    await azureAgentClient.Administration.DeleteAgentAsync(agent.Id);
}

async Task SKAgent_As_AFAgentAsync()
{
    Console.WriteLine("\n=== SK Agent Converted as an AF Agent ===\n");

    var azureAgentClient = AzureAIAgent.CreateAgentsClient(azureEndpoint, new AzureCliCredential());

    PersistentAgent definition = await azureAgentClient.Administration.CreateAgentAsync(deploymentName, tools: [new CodeInterpreterToolDefinition()]);

    AzureAIAgent skAgent = new(definition, azureAgentClient);

    var agent = skAgent.AsAIAgent();

    var thread = agent.GetNewThread();

    var result = await agent.RunAsync(userInput, thread);
    Console.WriteLine(result);

    // Extracts via breaking glass the code generated by code interpreter tool
    var chatResponse = result.RawRepresentation as ChatResponse;
    StringBuilder generatedCode = new();
    foreach (object? updateRawRepresentation in chatResponse?.RawRepresentation as IEnumerable<object?> ?? [])
    {
        // To capture the code interpreter input we need to break glass all the updates raw representations, to check for the RunStepDetailsUpdate type and
        // get the CodeInterpreterInput property which contains the generated code.
        // Note: Similar logic would needed for each individual update if used in the agent.RunStreamingAsync streaming API to aggregate or yield the generated code.
        if (updateRawRepresentation is RunStepDetailsUpdate update && update.CodeInterpreterInput is not null)
        {
            generatedCode.Append(update.CodeInterpreterInput);
        }
    }

    if (!string.IsNullOrEmpty(generatedCode.ToString()))
    {
        Console.WriteLine($"\n# {chatResponse?.Messages[0].Role}:Generated Code:\n{generatedCode}");
    }

    // Update the citations
    foreach (var textContent in result.Messages[0].Contents.OfType<Microsoft.Extensions.AI.TextContent>())
    {
        foreach (var annotation in textContent.Annotations ?? [])
        {
            if (annotation is CitationAnnotation citation)
            {
                if (citation.Url is null)
                {
                    Console.WriteLine($"  [{citation.GetType().Name}] {citation.Snippet}: File #{citation.FileId}");
                }

                foreach (var region in citation.AnnotatedRegions ?? [])
                {
                    if (region is TextSpanAnnotatedRegion textSpanRegion)
                    {
                        Console.WriteLine($"\n[TextSpan Region] {textSpanRegion.StartIndex}-{textSpanRegion.EndIndex}");
                    }
                }
            }
        }
    }

    // Clean up
    if (thread is ChatClientAgentThread chatThread)
    {
        await azureAgentClient.Threads.DeleteThreadAsync(chatThread.ConversationId);
    }
    await azureAgentClient.Administration.DeleteAgentAsync(agent.Id);
}

async Task AFAgentAsync()
{
    Console.WriteLine("\n=== AF Agent ===\n");

    var azureAgentClient = new PersistentAgentsClient(azureEndpoint, new AzureCliCredential());
    var agent = await azureAgentClient.CreateAIAgentAsync(deploymentName, tools: [new CodeInterpreterToolDefinition()]);
    var thread = agent.GetNewThread();

    var result = await agent.RunAsync(userInput, thread);
    Console.WriteLine(result);

    // Extracts via breaking glass the code generated by code interpreter tool
    var chatResponse = result.RawRepresentation as ChatResponse;
    StringBuilder generatedCode = new();
    foreach (object? updateRawRepresentation in chatResponse?.RawRepresentation as IEnumerable<object?> ?? [])
    {
        // To capture the code interpreter input we need to break glass all the updates raw representations, to check for the RunStepDetailsUpdate type and
        // get the CodeInterpreterInput property which contains the generated code. 
        // Note: Similar logic would needed for each individual update if used in the agent.RunStreamingAsync streaming API to aggregate or yield the generated code.
        if (updateRawRepresentation is RunStepDetailsUpdate update && update.CodeInterpreterInput is not null)
        {
            generatedCode.Append(update.CodeInterpreterInput);
        }
    }

    if (!string.IsNullOrEmpty(generatedCode.ToString()))
    {
        Console.WriteLine($"\n# {chatResponse?.Messages[0].Role}:Generated Code:\n{generatedCode}");
    }

    // Update the citations
    foreach (var textContent in result.Messages[0].Contents.OfType<Microsoft.Extensions.AI.TextContent>())
    {
        foreach (var annotation in textContent.Annotations ?? [])
        {
            if (annotation is CitationAnnotation citation)
            {
                if (citation.Url is null)
                {
                    Console.WriteLine($"  [{citation.GetType().Name}] {citation.Snippet}: File #{citation.FileId}");
                }

                foreach (var region in citation.AnnotatedRegions ?? [])
                {
                    if (region is TextSpanAnnotatedRegion textSpanRegion)
                    {
                        Console.WriteLine($"\n[TextSpan Region] {textSpanRegion.StartIndex}-{textSpanRegion.EndIndex}");
                    }
                }
            }
        }
    }

    // Clean up
    if (thread is ChatClientAgentThread chatThread)
    {
        await azureAgentClient.Threads.DeleteThreadAsync(chatThread.ConversationId);
    }
    await azureAgentClient.Administration.DeleteAgentAsync(agent.Id);
}
