﻿// Copyright (c) Microsoft. All rights reserved.

using System.Text;
using Microsoft.Agents.AI;
using Microsoft.Extensions.AI;
using Microsoft.SemanticKernel;
using Microsoft.SemanticKernel.Agents;
using Microsoft.SemanticKernel.Agents.OpenAI;
using OpenAI;
using OpenAI.Assistants;

#pragma warning disable OPENAI001 // Type is for evaluation purposes only and is subject to change or removal in future updates. Suppress this diagnostic to proceed.
#pragma warning disable SKEXP0110 // Type is for evaluation purposes only and is subject to change or removal in future updates. Suppress this diagnostic to proceed.

var apiKey = Environment.GetEnvironmentVariable("OPENAI_API_KEY") ?? throw new InvalidOperationException("OPENAI_API_KEY is not set.");
var model = System.Environment.GetEnvironmentVariable("OPENAI_MODEL") ?? "gpt-4o";
var userInput = "Create a python code file using the code interpreter tool with a code ready to determine the values in the Fibonacci sequence that are less then the value of 101";

var assistantsClient = new AssistantClient(apiKey);

Console.WriteLine($"User Input: {userInput}");

await SKAgentAsync();
await SKAgent_As_AFAgentAsync();
await AFAgentAsync();

async Task SKAgentAsync()
{
    Console.WriteLine("\n=== SK Agent ===\n");

    // Define the assistant
    Assistant assistant = await assistantsClient.CreateAssistantAsync(model, enableCodeInterpreter: true);

    // Create the agent
    OpenAIAssistantAgent agent = new(assistant, assistantsClient);

    // Create a thread for the agent conversation.
    var thread = new OpenAIAssistantAgentThread(assistantsClient);

    // Respond to user input
    await foreach (var content in agent.InvokeAsync(userInput, thread))
    {
        if (!string.IsNullOrWhiteSpace(content.Message.Content))
        {
            bool isCode = content.Message.Metadata?.ContainsKey(OpenAIAssistantAgent.CodeInterpreterMetadataKey) ?? false;
            Console.WriteLine($"\n# {content.Message.Role}{(isCode ? "\n# Generated Code:\n" : ":")}{content.Message.Content}");
        }

        // Check for the citations
        foreach (var item in content.Message.Items)
        {
            // Process each item in the message
            if (item is AnnotationContent annotation)
            {
                if (annotation.Kind != AnnotationKind.UrlCitation)
                {
                    Console.WriteLine($"  [{item.GetType().Name}] {annotation.Label}: File #{annotation.ReferenceId}");
                }
            }
            else if (item is FileReferenceContent fileReference)
            {
                Console.WriteLine($"  [{item.GetType().Name}] File #{fileReference.FileId}");
            }
        }
    }

    // Clean up
    await thread.DeleteAsync();
    await assistantsClient.DeleteAssistantAsync(agent.Id);
}

async Task SKAgent_As_AFAgentAsync()
{
    Console.WriteLine("\n=== SK Agent Converted as an AF Agent ===\n");

    // Define the assistant
    Assistant assistant = await assistantsClient.CreateAssistantAsync(model, enableCodeInterpreter: true);

    // Create the agent
    OpenAIAssistantAgent skAgent = new(assistant, assistantsClient);

    var agent = skAgent.AsAIAgent();

    var thread = agent.GetNewThread();

    var result = await agent.RunAsync(userInput, thread);
    Console.WriteLine(result);

    // Extracts via breaking glass the code generated by code interpreter tool
    var chatResponse = result.RawRepresentation as ChatResponse;
    StringBuilder generatedCode = new();
    foreach (object? updateRawRepresentation in chatResponse?.RawRepresentation as IEnumerable<object?> ?? [])
    {
        if (updateRawRepresentation is RunStepDetailsUpdate update && update.CodeInterpreterInput is not null)
        {
            generatedCode.Append(update.CodeInterpreterInput);
        }
    }

    if (!string.IsNullOrEmpty(generatedCode.ToString()))
    {
        Console.WriteLine($"\n# {chatResponse?.Messages[0].Role}:Generated Code:\n{generatedCode}");
    }

    // Check for the citations
    foreach (var textContent in result.Messages[0].Contents.OfType<Microsoft.Extensions.AI.TextContent>())
    {
        foreach (var annotation in textContent.Annotations ?? [])
        {
            if (annotation is CitationAnnotation citation)
            {
                if (citation.Url is null)
                {
                    Console.WriteLine($"  [{citation.GetType().Name}] {citation.Snippet}: File #{citation.FileId}");
                }

                foreach (var region in citation.AnnotatedRegions ?? [])
                {
                    if (region is TextSpanAnnotatedRegion textSpanRegion)
                    {
                        Console.WriteLine($"\n[TextSpan Region] {textSpanRegion.StartIndex}-{textSpanRegion.EndIndex}");
                    }
                }
            }
        }
    }

    // Clean up
    if (thread is ChatClientAgentThread chatThread)
    {
        await assistantsClient.DeleteThreadAsync(chatThread.ConversationId);
    }
    await assistantsClient.DeleteAssistantAsync(agent.Id);
}

async Task AFAgentAsync()
{
    Console.WriteLine("\n=== AF Agent ===\n");

    var agent = await assistantsClient.CreateAIAgentAsync(model, tools: [new HostedCodeInterpreterTool()]);

    var thread = agent.GetNewThread();

    var result = await agent.RunAsync(userInput, thread);
    Console.WriteLine(result);

    // Extracts via breaking glass the code generated by code interpreter tool
    var chatResponse = result.RawRepresentation as ChatResponse;
    StringBuilder generatedCode = new();
    foreach (object? updateRawRepresentation in chatResponse?.RawRepresentation as IEnumerable<object?> ?? [])
    {
        if (updateRawRepresentation is RunStepDetailsUpdate update && update.CodeInterpreterInput is not null)
        {
            generatedCode.Append(update.CodeInterpreterInput);
        }
    }

    if (!string.IsNullOrEmpty(generatedCode.ToString()))
    {
        Console.WriteLine($"\n# {chatResponse?.Messages[0].Role}:Generated Code:\n{generatedCode}");
    }

    // Check for the citations
    foreach (var textContent in result.Messages[0].Contents.OfType<Microsoft.Extensions.AI.TextContent>())
    {
        foreach (var annotation in textContent.Annotations ?? [])
        {
            if (annotation is CitationAnnotation citation)
            {
                if (citation.Url is null)
                {
                    Console.WriteLine($"  [{citation.GetType().Name}] {citation.Snippet}: File #{citation.FileId}");
                }

                foreach (var region in citation.AnnotatedRegions ?? [])
                {
                    if (region is TextSpanAnnotatedRegion textSpanRegion)
                    {
                        Console.WriteLine($"\n[TextSpan Region] {textSpanRegion.StartIndex}-{textSpanRegion.EndIndex}");
                    }
                }
            }
        }
    }

    // Clean up
    if (thread is ChatClientAgentThread chatThread)
    {
        await assistantsClient.DeleteThreadAsync(chatThread.ConversationId);
    }
    await assistantsClient.DeleteAssistantAsync(agent.Id);
}
