﻿// Copyright (c) Microsoft. All rights reserved.

using System;
using System.Collections.Generic;
using System.Text.Json.Serialization;

namespace Microsoft.SemanticKernel.Connectors.Amazon;

/// <summary>
/// Tools accessed by the Command R execution settings and Command R request.
/// </summary>
public static class CohereCommandRTools
{
    /// <summary>
    /// The required fields for chat_history.
    /// </summary>
    public sealed class ChatMessage
    {
        /// <summary>
        /// The role for the message. Valid values are USER or CHATBOT. tokens.
        /// </summary>
        [JsonPropertyName("role")]
        public string? Role { get; set; }

        /// <summary>
        /// Text contents of the message.
        /// </summary>
        [JsonPropertyName("message")]
        public string? Message { get; set; }
    }

    /// <summary>
    /// JSON structure for list of texts that the model can cite to generate a more accurate reply.
    /// </summary>
    [Serializable]
    public sealed class Document
    {
        /// <summary>
        /// Possible key field.
        /// </summary>
        [JsonPropertyName("title")]
        public string? Title { get; set; }

        /// <summary>
        /// Possible value field.
        /// </summary>
        [JsonPropertyName("snippet")]
        public string? Snippet { get; set; }
    }

    /// <summary>
    /// Tool parameters.
    /// </summary>
    [Serializable]
    public sealed class Tool
    {
        /// <summary>
        /// Name of the tool.
        /// </summary>
        [JsonPropertyName("name")]
        public string? Name { get; set; }

        /// <summary>
        /// Description of the tool.
        /// </summary>
        [JsonPropertyName("description")]
        public string? Description { get; set; }

        /// <summary>
        /// Definitions for each tool.
        /// </summary>
        [JsonPropertyName("parameter_definitions")]
        public Dictionary<string, ToolParameter> ParameterDefinitions { get; set; } = [];
    }
    /// <summary>
    /// Components of each tool parameter.
    /// </summary>
    [Serializable]
    public sealed class ToolParameter
    {
        /// <summary>
        /// Description of parameter.
        /// </summary>
        [JsonPropertyName("description")]
        public string? Description { get; set; }

        /// <summary>
        /// Parameter type (str, int, etc.) as described in a string.
        /// </summary>
        [JsonPropertyName("type")]
        public string? Type { get; set; }

        /// <summary>
        /// Whether this parameter is required.
        /// </summary>
        [JsonPropertyName("required")]
        public bool? Required { get; set; }
    }

    /// <summary>
    /// Cohere tool result.
    /// </summary>
    [Serializable]
    public sealed class ToolResult
    {
        /// <summary>
        /// The tool call.
        /// </summary>
        [JsonPropertyName("call")]
        public ToolCall? Call { get; set; }

        /// <summary>
        /// Outputs from the tool call.
        /// </summary>
        [JsonPropertyName("outputs")]
        public List<Dictionary<string, string>> Outputs { get; set; } = [];
    }

    /// <summary>
    /// Tool call object to be passed into the tool call.
    /// </summary>
    [Serializable]
    public sealed class ToolCall
    {
        /// <summary>
        /// Name of the tool.
        /// </summary>
        [JsonPropertyName("name")]
        public string? Name { get; set; }

        /// <summary>
        /// Parameters for the tool.
        /// </summary>
        [JsonPropertyName("parameters")]
        public Dictionary<string, string> Parameters { get; set; } = [];

        /// <summary>
        /// Tool call identifier generated by the model.
        /// </summary>
        [JsonPropertyName("generation_id")]
        public string? GenerationId { get; set; }
    }
}
