﻿// Copyright (c) Microsoft. All rights reserved.

using System.Collections.Generic;
using System.Text.Json.Serialization;

namespace Microsoft.SemanticKernel.Connectors.Amazon.Core;

internal static class MistralRequest
{
    /// <summary>
    /// Text generation request structure for Mistral to be passed into the InvokeModelRequest body.
    /// </summary>
    internal sealed class MistralTextGenerationRequest
    {
        /// <summary>
        /// (Required) The prompt that you want to pass to the model, as shown in the following example.
        /// </summary>
        [JsonPropertyName("prompt")]
        public string? Prompt { get; set; }

        /// <summary>
        /// Specify the maximum number of tokens to use in the generated response. The model truncates the response once the generated text exceeds max_tokens
        /// </summary>
        [JsonPropertyName("max_tokens")]
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        public int? MaxTokens { get; set; }

        /// <summary>
        /// A list of stop sequences that if generated by the model, stops the model from generating further output.
        /// </summary>
        [JsonPropertyName("stop")]
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        public IList<string>? StopSequences { get; set; } = [];

        /// <summary>
        /// Controls the randomness of predictions made by the model.
        /// </summary>
        [JsonPropertyName("temperature")]
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        public float? Temperature { get; set; }

        /// <summary>
        /// Controls the diversity of text that the model generates by setting the percentage of most-likely candidates that the model considers for the next token.
        /// </summary>
        [JsonPropertyName("top_p")]
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        public float? TopP { get; set; }

        /// <summary>
        /// Controls the number of most-likely candidates that the model considers for the next token.
        /// </summary>
        [JsonPropertyName("top_k")]
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        public int? TopK { get; set; }
    }
}
