﻿// Copyright (c) Microsoft. All rights reserved.

using System;

namespace Microsoft.SemanticKernel.Plugins.OpenApi;

/// <summary>
/// REST API Security Scheme.
/// </summary>
public sealed class RestApiSecurityScheme
{
    /// <summary>
    /// REQUIRED. The type of the security scheme. Valid values are "apiKey", "http", "oauth2", "openIdConnect".
    /// </summary>
    public string SecuritySchemeType { get; init; }

    /// <summary>
    /// A short description for security scheme. CommonMark syntax MAY be used for rich text representation.
    /// </summary>
    public string? Description { get; init; }

    /// <summary>
    /// REQUIRED. The name of the header, query or cookie parameter to be used.
    /// </summary>
    public string Name { get; init; }

    /// <summary>
    /// REQUIRED. The location of the API key. Valid values are "query", "header" or "cookie".
    /// </summary>
    public RestApiParameterLocation In { get; init; }

    /// <summary>
    /// REQUIRED. The name of the HTTP Authorization scheme to be used
    /// in the Authorization header as defined in RFC7235.
    /// </summary>
    public string Scheme { get; init; }

    /// <summary>
    /// A hint to the client to identify how the bearer token is formatted.
    /// Bearer tokens are usually generated by an authorization server,
    /// so this information is primarily for documentation purposes.
    /// </summary>
    public string? BearerFormat { get; init; }

    /// <summary>
    /// REQUIRED. An object containing configuration information for the flow types supported.
    /// </summary>
    public RestApiOAuthFlows? Flows { get; init; }

    /// <summary>
    /// REQUIRED. OpenId Connect URL to discover OAuth2 configuration values.
    /// </summary>
    public Uri OpenIdConnectUrl { get; init; }

    /// <summary>
    /// Creates an instance of a <see cref="RestApiSecurityScheme"/> class.
    /// </summary>
#pragma warning disable CS8618 // Non-nullable field must contain a non-null value when exiting constructor. Consider adding the 'required' modifier or declaring as nullable.
    internal RestApiSecurityScheme()
#pragma warning restore CS8618 // Non-nullable field must contain a non-null value when exiting constructor. Consider adding the 'required' modifier or declaring as nullable.
    {
    }

    internal void Freeze()
    {
        this.Flows?.Freeze();
    }
}
