﻿// Copyright (c) Microsoft. All rights reserved.

using System;
using System.Collections.Generic;
using System.Text.Json.Serialization;

namespace Microsoft.SemanticKernel.Plugins.Web.Tavily;

/// <summary>
/// Represents a search response.
/// </summary>
#pragma warning disable CA1812 // Instantiated by reflection
internal sealed class TavilySearchResponse
{
    /// <summary>
    /// A short answer to the user's query, generated by an LLM. Included in the response only if include_answer is requested (i.e., set to true, basic, or advanced)
    /// </summary>
    [JsonPropertyName("answer")]
    public string? Answer { get; set; }

    /// <summary>
    /// The search query that was executed.
    /// </summary>
    [JsonPropertyName("query")]
    [JsonRequired]
    public string Query { get; set; }

    /// <summary>
    /// Time in seconds it took to complete the request.
    /// </summary>
    [JsonPropertyName("response_time")]
    [JsonRequired]
    public double ResponseTime { get; set; }

    /// <summary>
    /// List of query-related images. If include_image_descriptions is true, each item will have url and description.
    /// </summary>
    [JsonPropertyName("images")]
    public IList<TavilyImageResult>? Images { get; set; }

    /// <summary>
    /// A list of sorted search results, ranked by relevancy.
    /// </summary>
    [JsonPropertyName("results")]
    [JsonRequired]
    public IList<TavilySearchResult> Results { get; set; }

    /// <summary>
    /// A list of suggested follow up questions related to original query.
    /// </summary>
    [JsonPropertyName("follow_up_questions")]
    public IList<string>? FollowUpQuestions { get; set; }

    /// <summary>
    /// Additional properties that are not explicitly defined in the schema
    /// </summary>
    [JsonExtensionData]
    public IDictionary<string, object> AdditionalProperties { get; set; } = new Dictionary<string, object>();

    /// <summary>
    /// Initializes a new instance of the <see cref="TavilySearchResponse" /> class.
    /// </summary>
    [JsonConstructorAttribute]
    internal TavilySearchResponse(
            string query,
            double responseTime,
            IList<TavilySearchResult> results,
            string? answer,
            IList<TavilyImageResult>? images)
    {
        this.Query = query ?? throw new ArgumentNullException(nameof(query));
        this.ResponseTime = responseTime;
        this.Results = results ?? throw new ArgumentNullException(nameof(results));
        this.Answer = answer;
        this.Images = images;
    }
}
#pragma warning restore CA1812 // Instantiated by reflection
