// Copyright (C) 2019-2020 Zilliz. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance
// with the License. You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software distributed under the License
// is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
// or implied. See the License for the specific language governing permissions and limitations under the License.

package main

import (
	"bufio"
	"bytes"
	"fmt"
	"os"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
	"go.uber.org/zap"

	"github.com/milvus-io/milvus/pkg/v2/log"
	"github.com/milvus-io/milvus/pkg/v2/util/paramtable"
)

// Assert the milvus.yaml file is consistent to paramtable
//
//	 Please be noted that milvus.yaml is generated by code, so don't edit it directly, instead, change the code in paramtable
//		and run `make milvus-tools && ./bin/tools/config gen-yaml && mv milvus.yaml configs/milvus.yaml`.
func TestYamlFile(t *testing.T) {
	log.SetLevel(zap.InfoLevel)
	w := bytes.Buffer{}
	WriteYaml(&w)

	base := paramtable.NewBaseTable()
	f, err := os.Open(fmt.Sprintf("%s/%s", base.GetConfigDir(), "milvus.yaml"))
	assert.NoError(t, err, "expecting configs/milvus.yaml")
	log.Info("Verifying config", zap.String("file", f.Name()))
	defer f.Close()
	fileScanner := bufio.NewScanner(f)
	codeScanner := bufio.NewScanner(&w)

	msg := func(file, code string) string {
		return fmt.Sprintf(`configs/milvus.yaml is not consistent with paramtable, file: [%s], code: [%s]. 
Do not edit milvus.yaml directly, instead, run "make milvus-tools && ./bin/tools/config gen-yaml && mv milvus.yaml configs/milvus.yaml"`, file, code)
	}
	for fileScanner.Scan() {
		if !codeScanner.Scan() {
			assert.FailNow(t, msg(fileScanner.Text(), "EMPTY"))
		}
		if strings.Contains(codeScanner.Text(), "etcd:") || strings.Contains(codeScanner.Text(), "minio:") || strings.Contains(codeScanner.Text(), "pulsar:") {
			// Skip check of endpoints given by .env
			continue
		}
		if fileScanner.Text() != codeScanner.Text() {
			assert.FailNow(t, msg(fileScanner.Text(), codeScanner.Text()))
		}
	}
	if codeScanner.Scan() {
		assert.FailNow(t, msg("EMPTY", codeScanner.Text()))
	}
}
