/*
These stop word lists are from the Snowball project (https://snowballstem.org/)
which carries the following copyright and license:

Copyright (c) 2001, Dr Martin Porter
Copyright (c) 2004,2005, Richard Boulton
Copyright (c) 2013, Yoshiki Shibukawa
Copyright (c) 2006,2007,2009,2010,2011,2014-2019, Olly Betts
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

  1. Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation
     and/or other materials provided with the distribution.
  3. Neither the name of the Snowball project nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

use once_cell::sync::Lazy;
fn fetch_words_from_file(data: &[u8]) -> &'static [&'static str] {
    let s = std::str::from_utf8(data).expect("UTF-8");
    let vec: Box<Vec<String>> = Box::new(
        s.lines()
            .map(|line| line.trim().to_string())
            .filter(|l| !l.is_empty())
            .collect(),
    );
    let vec: &'static Vec<String> = Box::leak(vec);

    let slice: Box<[&str]> = vec
        .iter()
        .map(|s| s.as_str())
        .collect::<Vec<&str>>()
        .into_boxed_slice();
    Box::leak(slice)
}

pub fn fetch_language_stop_words(lang: &str) -> Option<&[&str]> {
    match lang {
        "_english_" => Some(ENGLISH),
        "_danish_" => Some(DANISH),
        "_dutch_" => Some(DUTCH),
        "_finnish_" => Some(FINNISH),
        "_french_" => Some(FRENCH),
        "_german_" => Some(GERMAN),
        "_hungarian_" => Some(HUNGARIAN),
        "_italian_" => Some(ITALIAN),
        "_norwegian_" => Some(NORWEGIAN),
        "_portuguese_" => Some(PORTUGUESE),
        "_russian_" => Some(RUSSIAN),
        "_spanish_" => Some(SPANISH),
        "_swedish_" => Some(SWEDISH),
        "_chinese_" => Some(&*CHINESE),
        _ => None,
    }
}

pub const ENGLISH: &[&str] = &[
    "a", "an", "and", "are", "as", "at", "be", "but", "by", "for", "if", "in", "into", "is", "it",
    "no", "not", "of", "on", "or", "such", "that", "the", "their", "then", "there", "these",
    "they", "this", "to", "was", "will", "with",
];

pub const DANISH: &[&str] = &[
    "og", "i", "jeg", "det", "at", "en", "den", "til", "er", "som", "på", "de", "med", "han", "af",
    "for", "ikke", "der", "var", "mig", "sig", "men", "et", "har", "om", "vi", "min", "havde",
    "ham", "hun", "nu", "over", "da", "fra", "du", "ud", "sin", "dem", "os", "op", "man", "hans",
    "hvor", "eller", "hvad", "skal", "selv", "her", "alle", "vil", "blev", "kunne", "ind", "når",
    "være", "dog", "noget", "ville", "jo", "deres", "efter", "ned", "skulle", "denne", "end",
    "dette", "mit", "også", "under", "have", "dig", "anden", "hende", "mine", "alt", "meget",
    "sit", "sine", "vor", "mod", "disse", "hvis", "din", "nogle", "hos", "blive", "mange", "ad",
    "bliver", "hendes", "været", "thi", "jer", "sådan",
];

pub const DUTCH: &[&str] = &[
    "de", "en", "van", "ik", "te", "dat", "die", "in", "een", "hij", "het", "niet", "zijn", "is",
    "was", "op", "aan", "met", "als", "voor", "had", "er", "maar", "om", "hem", "dan", "zou", "of",
    "wat", "mijn", "men", "dit", "zo", "door", "over", "ze", "zich", "bij", "ook", "tot", "je",
    "mij", "uit", "der", "daar", "haar", "naar", "heb", "hoe", "heeft", "hebben", "deze", "u",
    "want", "nog", "zal", "me", "zij", "nu", "ge", "geen", "omdat", "iets", "worden", "toch", "al",
    "waren", "veel", "meer", "doen", "toen", "moet", "ben", "zonder", "kan", "hun", "dus", "alles",
    "onder", "ja", "eens", "hier", "wie", "werd", "altijd", "doch", "wordt", "wezen", "kunnen",
    "ons", "zelf", "tegen", "na", "reeds", "wil", "kon", "niets", "uw", "iemand", "geweest",
    "andere",
];

pub const FINNISH: &[&str] = &[
    "olla",
    "olen",
    "olet",
    "on",
    "olemme",
    "olette",
    "ovat",
    "ole",
    "oli",
    "olisi",
    "olisit",
    "olisin",
    "olisimme",
    "olisitte",
    "olisivat",
    "olit",
    "olin",
    "olimme",
    "olitte",
    "olivat",
    "ollut",
    "olleet",
    "en",
    "et",
    "ei",
    "emme",
    "ette",
    "eivät",
    "minä",
    "minun",
    "minut",
    "minua",
    "minussa",
    "minusta",
    "minuun",
    "minulla",
    "minulta",
    "minulle",
    "sinä",
    "sinun",
    "sinut",
    "sinua",
    "sinussa",
    "sinusta",
    "sinuun",
    "sinulla",
    "sinulta",
    "sinulle",
    "hän",
    "hänen",
    "hänet",
    "häntä",
    "hänessä",
    "hänestä",
    "häneen",
    "hänellä",
    "häneltä",
    "hänelle",
    "me",
    "meidän",
    "meidät",
    "meitä",
    "meissä",
    "meistä",
    "meihin",
    "meillä",
    "meiltä",
    "meille",
    "te",
    "teidän",
    "teidät",
    "teitä",
    "teissä",
    "teistä",
    "teihin",
    "teillä",
    "teiltä",
    "teille",
    "he",
    "heidän",
    "heidät",
    "heitä",
    "heissä",
    "heistä",
    "heihin",
    "heillä",
    "heiltä",
    "heille",
    "tämä",
    "tämän",
    "tätä",
    "tässä",
    "tästä",
    "tähän",
    "tällä",
    "tältä",
    "tälle",
    "tänä",
    "täksi",
    "tuo",
    "tuon",
    "tuota",
    "tuossa",
    "tuosta",
    "tuohon",
    "tuolla",
    "tuolta",
    "tuolle",
    "tuona",
    "tuoksi",
    "se",
    "sen",
    "sitä",
    "siinä",
    "siitä",
    "siihen",
    "sillä",
    "siltä",
    "sille",
    "sinä",
    "siksi",
    "nämä",
    "näiden",
    "näitä",
    "näissä",
    "näistä",
    "näihin",
    "näillä",
    "näiltä",
    "näille",
    "näinä",
    "näiksi",
    "nuo",
    "noiden",
    "noita",
    "noissa",
    "noista",
    "noihin",
    "noilla",
    "noilta",
    "noille",
    "noina",
    "noiksi",
    "ne",
    "niiden",
    "niitä",
    "niissä",
    "niistä",
    "niihin",
    "niillä",
    "niiltä",
    "niille",
    "niinä",
    "niiksi",
    "kuka",
    "kenen",
    "kenet",
    "ketä",
    "kenessä",
    "kenestä",
    "keneen",
    "kenellä",
    "keneltä",
    "kenelle",
    "kenenä",
    "keneksi",
    "ketkä",
    "keiden",
    "ketkä",
    "keitä",
    "keissä",
    "keistä",
    "keihin",
    "keillä",
    "keiltä",
    "keille",
    "keinä",
    "keiksi",
    "mikä",
    "minkä",
    "minkä",
    "mitä",
    "missä",
    "mistä",
    "mihin",
    "millä",
    "miltä",
    "mille",
    "minä",
    "miksi",
    "mitkä",
    "joka",
    "jonka",
    "jota",
    "jossa",
    "josta",
    "johon",
    "jolla",
    "jolta",
    "jolle",
    "jona",
    "joksi",
    "jotka",
    "joiden",
    "joita",
    "joissa",
    "joista",
    "joihin",
    "joilla",
    "joilta",
    "joille",
    "joina",
    "joiksi",
    "että",
    "ja",
    "jos",
    "koska",
    "kuin",
    "mutta",
    "niin",
    "sekä",
    "sillä",
    "tai",
    "vaan",
    "vai",
    "vaikka",
    "kanssa",
    "mukaan",
    "noin",
    "poikki",
    "yli",
    "kun",
    "nyt",
    "itse",
];

pub const FRENCH: &[&str] = &[
    "au", "aux", "avec", "ce", "ces", "dans", "de", "des", "du", "elle", "en", "et", "eux", "il",
    "je", "la", "le", "leur", "lui", "ma", "mais", "me", "même", "mes", "moi", "mon", "ne", "nos",
    "notre", "nous", "on", "ou", "par", "pas", "pour", "qu", "que", "qui", "sa", "se", "ses",
    "sur", "ta", "te", "tes", "toi", "ton", "tu", "un", "une", "vos", "votre", "vous", "c", "d",
    "j", "l", "à", "m", "n", "s", "t", "y", "étée", "étées", "étant", "suis", "es", "êtes", "sont",
    "serai", "seras", "sera", "serons", "serez", "seront", "serais", "serait", "serions", "seriez",
    "seraient", "étais", "était", "étions", "étiez", "étaient", "fus", "fut", "fûmes", "fûtes",
    "furent", "sois", "soit", "soyons", "soyez", "soient", "fusse", "fusses", "fussions",
    "fussiez", "fussent", "ayant", "eu", "eue", "eues", "eus", "ai", "avons", "avez", "ont",
    "aurai", "aurons", "aurez", "auront", "aurais", "aurait", "aurions", "auriez", "auraient",
    "avais", "avait", "aviez", "avaient", "eut", "eûmes", "eûtes", "eurent", "aie", "aies", "ait",
    "ayons", "ayez", "aient", "eusse", "eusses", "eût", "eussions", "eussiez", "eussent", "ceci",
    "cela", "celà", "cet", "cette", "ici", "ils", "les", "leurs", "quel", "quels", "quelle",
    "quelles", "sans", "soi",
];

pub const GERMAN: &[&str] = &[
    "aber",
    "alle",
    "allem",
    "allen",
    "aller",
    "alles",
    "als",
    "also",
    "am",
    "an",
    "ander",
    "andere",
    "anderem",
    "anderen",
    "anderer",
    "anderes",
    "anderm",
    "andern",
    "anderr",
    "anders",
    "auch",
    "auf",
    "aus",
    "bei",
    "bin",
    "bis",
    "bist",
    "da",
    "damit",
    "dann",
    "der",
    "den",
    "des",
    "dem",
    "die",
    "das",
    "daß",
    "derselbe",
    "derselben",
    "denselben",
    "desselben",
    "demselben",
    "dieselbe",
    "dieselben",
    "dasselbe",
    "dazu",
    "dein",
    "deine",
    "deinem",
    "deinen",
    "deiner",
    "deines",
    "denn",
    "derer",
    "dessen",
    "dich",
    "dir",
    "du",
    "dies",
    "diese",
    "diesem",
    "diesen",
    "dieser",
    "dieses",
    "doch",
    "dort",
    "durch",
    "ein",
    "eine",
    "einem",
    "einen",
    "einer",
    "eines",
    "einig",
    "einige",
    "einigem",
    "einigen",
    "einiger",
    "einiges",
    "einmal",
    "er",
    "ihn",
    "ihm",
    "es",
    "etwas",
    "euer",
    "eure",
    "eurem",
    "euren",
    "eurer",
    "eures",
    "für",
    "gegen",
    "gewesen",
    "hab",
    "habe",
    "haben",
    "hat",
    "hatte",
    "hatten",
    "hier",
    "hin",
    "hinter",
    "ich",
    "mich",
    "mir",
    "ihr",
    "ihre",
    "ihrem",
    "ihren",
    "ihrer",
    "ihres",
    "euch",
    "im",
    "in",
    "indem",
    "ins",
    "ist",
    "jede",
    "jedem",
    "jeden",
    "jeder",
    "jedes",
    "jene",
    "jenem",
    "jenen",
    "jener",
    "jenes",
    "jetzt",
    "kann",
    "kein",
    "keine",
    "keinem",
    "keinen",
    "keiner",
    "keines",
    "können",
    "könnte",
    "machen",
    "man",
    "manche",
    "manchem",
    "manchen",
    "mancher",
    "manches",
    "mein",
    "meine",
    "meinem",
    "meinen",
    "meiner",
    "meines",
    "mit",
    "muss",
    "musste",
    "nach",
    "nicht",
    "nichts",
    "noch",
    "nun",
    "nur",
    "ob",
    "oder",
    "ohne",
    "sehr",
    "sein",
    "seine",
    "seinem",
    "seinen",
    "seiner",
    "seines",
    "selbst",
    "sich",
    "sie",
    "ihnen",
    "sind",
    "so",
    "solche",
    "solchem",
    "solchen",
    "solcher",
    "solches",
    "soll",
    "sollte",
    "sondern",
    "sonst",
    "über",
    "um",
    "und",
    "uns",
    "unse",
    "unsem",
    "unsen",
    "unser",
    "unses",
    "unter",
    "viel",
    "vom",
    "von",
    "vor",
    "während",
    "war",
    "waren",
    "warst",
    "was",
    "weg",
    "weil",
    "weiter",
    "welche",
    "welchem",
    "welchen",
    "welcher",
    "welches",
    "wenn",
    "werde",
    "werden",
    "wie",
    "wieder",
    "will",
    "wir",
    "wird",
    "wirst",
    "wo",
    "wollen",
    "wollte",
    "würde",
    "würden",
    "zu",
    "zum",
    "zur",
    "zwar",
    "zwischen",
];

pub const HUNGARIAN: &[&str] = &[
    "a",
    "ahogy",
    "ahol",
    "aki",
    "akik",
    "akkor",
    "alatt",
    "által",
    "általában",
    "amely",
    "amelyek",
    "amelyekben",
    "amelyeket",
    "amelyet",
    "amelynek",
    "ami",
    "amit",
    "amolyan",
    "amíg",
    "amikor",
    "át",
    "abban",
    "ahhoz",
    "annak",
    "arra",
    "arról",
    "az",
    "azok",
    "azon",
    "azt",
    "azzal",
    "azért",
    "aztán",
    "azután",
    "azonban",
    "bár",
    "be",
    "belül",
    "benne",
    "cikk",
    "cikkek",
    "cikkeket",
    "csak",
    "de",
    "e",
    "eddig",
    "egész",
    "egy",
    "egyes",
    "egyetlen",
    "egyéb",
    "egyik",
    "egyre",
    "ekkor",
    "el",
    "elég",
    "ellen",
    "elő",
    "először",
    "előtt",
    "első",
    "én",
    "éppen",
    "ebben",
    "ehhez",
    "emilyen",
    "ennek",
    "erre",
    "ez",
    "ezt",
    "ezek",
    "ezen",
    "ezzel",
    "ezért",
    "és",
    "fel",
    "felé",
    "hanem",
    "hiszen",
    "hogy",
    "hogyan",
    "igen",
    "így",
    "illetve",
    "ill.",
    "ill",
    "ilyen",
    "ilyenkor",
    "ison",
    "ismét",
    "itt",
    "jó",
    "jól",
    "jobban",
    "kell",
    "kellett",
    "keresztül",
    "keressünk",
    "ki",
    "kívül",
    "között",
    "közül",
    "legalább",
    "lehet",
    "lehetett",
    "legyen",
    "lenne",
    "lenni",
    "lesz",
    "lett",
    "maga",
    "magát",
    "majd",
    "majd",
    "már",
    "más",
    "másik",
    "meg",
    "még",
    "mellett",
    "mert",
    "mely",
    "melyek",
    "mi",
    "mit",
    "míg",
    "miért",
    "milyen",
    "mikor",
    "minden",
    "mindent",
    "mindenki",
    "mindig",
    "mint",
    "mintha",
    "mivel",
    "most",
    "nagy",
    "nagyobb",
    "nagyon",
    "ne",
    "néha",
    "nekem",
    "neki",
    "nem",
    "néhány",
    "nélkül",
    "nincs",
    "olyan",
    "ott",
    "össze",
    "ő",
    "ők",
    "őket",
    "pedig",
    "persze",
    "rá",
    "s",
    "saját",
    "sem",
    "semmi",
    "sok",
    "sokat",
    "sokkal",
    "számára",
    "szemben",
    "szerint",
    "szinte",
    "talán",
    "tehát",
    "teljes",
    "tovább",
    "továbbá",
    "több",
    "úgy",
    "ugyanis",
    "új",
    "újabb",
    "újra",
    "után",
    "utána",
    "utolsó",
    "vagy",
    "vagyis",
    "valaki",
    "valami",
    "valamint",
    "való",
    "vagyok",
    "van",
    "vannak",
    "volt",
    "voltam",
    "voltak",
    "voltunk",
    "vissza",
    "vele",
    "viszont",
    "volna",
];

pub const ITALIAN: &[&str] = &[
    "ad",
    "al",
    "allo",
    "ai",
    "agli",
    "all",
    "agl",
    "alla",
    "alle",
    "con",
    "col",
    "coi",
    "da",
    "dal",
    "dallo",
    "dai",
    "dagli",
    "dall",
    "dagl",
    "dalla",
    "dalle",
    "di",
    "del",
    "dello",
    "dei",
    "degli",
    "dell",
    "degl",
    "della",
    "delle",
    "in",
    "nel",
    "nello",
    "nei",
    "negli",
    "nell",
    "negl",
    "nella",
    "nelle",
    "su",
    "sul",
    "sullo",
    "sui",
    "sugli",
    "sull",
    "sugl",
    "sulla",
    "sulle",
    "per",
    "tra",
    "contro",
    "io",
    "tu",
    "lui",
    "lei",
    "noi",
    "voi",
    "loro",
    "mio",
    "mia",
    "miei",
    "mie",
    "tuo",
    "tua",
    "tuoi",
    "tue",
    "suo",
    "sua",
    "suoi",
    "sue",
    "nostro",
    "nostra",
    "nostri",
    "nostre",
    "vostro",
    "vostra",
    "vostri",
    "vostre",
    "mi",
    "ti",
    "ci",
    "vi",
    "lo",
    "la",
    "li",
    "le",
    "gli",
    "ne",
    "il",
    "un",
    "uno",
    "una",
    "ma",
    "ed",
    "se",
    "perché",
    "anche",
    "come",
    "dov",
    "dove",
    "che",
    "chi",
    "cui",
    "non",
    "più",
    "quale",
    "quanto",
    "quanti",
    "quanta",
    "quante",
    "quello",
    "quelli",
    "quella",
    "quelle",
    "questo",
    "questi",
    "questa",
    "queste",
    "si",
    "tutto",
    "tutti",
    "a",
    "c",
    "e",
    "i",
    "l",
    "o",
    "ho",
    "hai",
    "ha",
    "abbiamo",
    "avete",
    "hanno",
    "abbia",
    "abbiate",
    "abbiano",
    "avrò",
    "avrai",
    "avrà",
    "avremo",
    "avrete",
    "avranno",
    "avrei",
    "avresti",
    "avrebbe",
    "avremmo",
    "avreste",
    "avrebbero",
    "avevo",
    "avevi",
    "aveva",
    "avevamo",
    "avevate",
    "avevano",
    "ebbi",
    "avesti",
    "ebbe",
    "avemmo",
    "aveste",
    "ebbero",
    "avessi",
    "avesse",
    "avessimo",
    "avessero",
    "avendo",
    "avuto",
    "avuta",
    "avuti",
    "avute",
    "sono",
    "sei",
    "è",
    "siamo",
    "siete",
    "sia",
    "siate",
    "siano",
    "sarò",
    "sarai",
    "sarà",
    "saremo",
    "sarete",
    "saranno",
    "sarei",
    "saresti",
    "sarebbe",
    "saremmo",
    "sareste",
    "sarebbero",
    "ero",
    "eri",
    "era",
    "eravamo",
    "eravate",
    "erano",
    "fui",
    "fosti",
    "fu",
    "fummo",
    "foste",
    "furono",
    "fossi",
    "fosse",
    "fossimo",
    "fossero",
    "essendo",
    "faccio",
    "fai",
    "facciamo",
    "fanno",
    "faccia",
    "facciate",
    "facciano",
    "farò",
    "farai",
    "farà",
    "faremo",
    "farete",
    "faranno",
    "farei",
    "faresti",
    "farebbe",
    "faremmo",
    "fareste",
    "farebbero",
    "facevo",
    "facevi",
    "faceva",
    "facevamo",
    "facevate",
    "facevano",
    "feci",
    "facesti",
    "fece",
    "facemmo",
    "faceste",
    "fecero",
    "facessi",
    "facesse",
    "facessimo",
    "facessero",
    "facendo",
    "sto",
    "stai",
    "sta",
    "stiamo",
    "stanno",
    "stia",
    "stiate",
    "stiano",
    "starò",
    "starai",
    "starà",
    "staremo",
    "starete",
    "staranno",
    "starei",
    "staresti",
    "starebbe",
    "staremmo",
    "stareste",
    "starebbero",
    "stavo",
    "stavi",
    "stava",
    "stavamo",
    "stavate",
    "stavano",
    "stetti",
    "stesti",
    "stette",
    "stemmo",
    "steste",
    "stettero",
    "stessi",
    "stesse",
    "stessimo",
    "stessero",
    "stando",
];

pub const NORWEGIAN: &[&str] = &[
    "og",
    "i",
    "jeg",
    "det",
    "at",
    "en",
    "et",
    "den",
    "til",
    "er",
    "som",
    "på",
    "de",
    "med",
    "han",
    "av",
    "ikke",
    "ikkje",
    "der",
    "så",
    "var",
    "meg",
    "seg",
    "men",
    "ett",
    "har",
    "om",
    "vi",
    "min",
    "mitt",
    "ha",
    "hadde",
    "hun",
    "nå",
    "over",
    "da",
    "ved",
    "fra",
    "du",
    "ut",
    "sin",
    "dem",
    "oss",
    "opp",
    "man",
    "kan",
    "hans",
    "hvor",
    "eller",
    "hva",
    "skal",
    "selv",
    "sjøl",
    "her",
    "alle",
    "vil",
    "bli",
    "ble",
    "blei",
    "blitt",
    "kunne",
    "inn",
    "når",
    "være",
    "kom",
    "noen",
    "noe",
    "ville",
    "dere",
    "deres",
    "kun",
    "ja",
    "etter",
    "ned",
    "skulle",
    "denne",
    "for",
    "deg",
    "si",
    "sine",
    "sitt",
    "mot",
    "å",
    "meget",
    "hvorfor",
    "dette",
    "disse",
    "uten",
    "hvordan",
    "ingen",
    "din",
    "ditt",
    "blir",
    "samme",
    "hvilken",
    "hvilke",
    "sånn",
    "inni",
    "mellom",
    "vår",
    "hver",
    "hvem",
    "vors",
    "hvis",
    "både",
    "bare",
    "enn",
    "fordi",
    "før",
    "mange",
    "også",
    "slik",
    "vært",
    "båe",
    "begge",
    "siden",
    "dykk",
    "dykkar",
    "dei",
    "deira",
    "deires",
    "deim",
    "di",
    "då",
    "eg",
    "ein",
    "eit",
    "eitt",
    "elles",
    "honom",
    "hjå",
    "ho",
    "hoe",
    "henne",
    "hennar",
    "hennes",
    "hoss",
    "hossen",
    "ingi",
    "inkje",
    "korleis",
    "korso",
    "kva",
    "kvar",
    "kvarhelst",
    "kven",
    "kvi",
    "kvifor",
    "me",
    "medan",
    "mi",
    "mine",
    "mykje",
    "no",
    "nokon",
    "noka",
    "nokor",
    "noko",
    "nokre",
    "sia",
    "sidan",
    "so",
    "somt",
    "somme",
    "um",
    "upp",
    "vere",
    "vore",
    "verte",
    "vort",
    "varte",
    "vart",
];

pub const PORTUGUESE: &[&str] = &[
    "de",
    "a",
    "o",
    "que",
    "e",
    "do",
    "da",
    "em",
    "um",
    "para",
    "com",
    "não",
    "uma",
    "os",
    "no",
    "se",
    "na",
    "por",
    "mais",
    "as",
    "dos",
    "como",
    "mas",
    "ao",
    "ele",
    "das",
    "à",
    "seu",
    "sua",
    "ou",
    "quando",
    "muito",
    "nos",
    "já",
    "eu",
    "também",
    "só",
    "pelo",
    "pela",
    "até",
    "isso",
    "ela",
    "entre",
    "depois",
    "sem",
    "mesmo",
    "aos",
    "seus",
    "quem",
    "nas",
    "me",
    "esse",
    "eles",
    "você",
    "essa",
    "num",
    "nem",
    "suas",
    "meu",
    "às",
    "minha",
    "numa",
    "pelos",
    "elas",
    "qual",
    "nós",
    "lhe",
    "deles",
    "essas",
    "esses",
    "pelas",
    "este",
    "dele",
    "tu",
    "te",
    "vocês",
    "vos",
    "lhes",
    "meus",
    "minhas",
    "teu",
    "tua",
    "teus",
    "tuas",
    "nosso",
    "nossa",
    "nossos",
    "nossas",
    "dela",
    "delas",
    "esta",
    "estes",
    "estas",
    "aquele",
    "aquela",
    "aqueles",
    "aquelas",
    "isto",
    "aquilo",
    "estou",
    "está",
    "estamos",
    "estão",
    "estive",
    "esteve",
    "estivemos",
    "estiveram",
    "estava",
    "estávamos",
    "estavam",
    "estivera",
    "estivéramos",
    "esteja",
    "estejamos",
    "estejam",
    "estivesse",
    "estivéssemos",
    "estivessem",
    "estiver",
    "estivermos",
    "estiverem",
    "hei",
    "há",
    "havemos",
    "hão",
    "houve",
    "houvemos",
    "houveram",
    "houvera",
    "houvéramos",
    "haja",
    "hajamos",
    "hajam",
    "houvesse",
    "houvéssemos",
    "houvessem",
    "houver",
    "houvermos",
    "houverem",
    "houverei",
    "houverá",
    "houveremos",
    "houverão",
    "houveria",
    "houveríamos",
    "houveriam",
    "sou",
    "somos",
    "são",
    "era",
    "éramos",
    "eram",
    "fui",
    "foi",
    "fomos",
    "foram",
    "fora",
    "fôramos",
    "seja",
    "sejamos",
    "sejam",
    "fosse",
    "fôssemos",
    "fossem",
    "for",
    "formos",
    "forem",
    "serei",
    "será",
    "seremos",
    "serão",
    "seria",
    "seríamos",
    "seriam",
    "tenho",
    "tem",
    "temos",
    "tém",
    "tinha",
    "tínhamos",
    "tinham",
    "tive",
    "teve",
    "tivemos",
    "tiveram",
    "tivera",
    "tivéramos",
    "tenha",
    "tenhamos",
    "tenham",
    "tivesse",
    "tivéssemos",
    "tivessem",
    "tiver",
    "tivermos",
    "tiverem",
    "terei",
    "terá",
    "teremos",
    "terão",
    "teria",
    "teríamos",
    "teriam",
];

pub const RUSSIAN: &[&str] = &[
    "и",
    "в",
    "во",
    "не",
    "что",
    "он",
    "на",
    "я",
    "с",
    "со",
    "как",
    "а",
    "то",
    "все",
    "она",
    "так",
    "его",
    "но",
    "да",
    "ты",
    "к",
    "у",
    "же",
    "вы",
    "за",
    "бы",
    "по",
    "только",
    "ее",
    "мне",
    "было",
    "вот",
    "от",
    "меня",
    "еще",
    "нет",
    "о",
    "из",
    "ему",
    "теперь",
    "когда",
    "даже",
    "ну",
    "вдруг",
    "ли",
    "если",
    "уже",
    "или",
    "ни",
    "быть",
    "был",
    "него",
    "до",
    "вас",
    "нибудь",
    "опять",
    "уж",
    "вам",
    "сказал",
    "ведь",
    "там",
    "потом",
    "себя",
    "ничего",
    "ей",
    "может",
    "они",
    "тут",
    "где",
    "есть",
    "надо",
    "ней",
    "для",
    "мы",
    "тебя",
    "их",
    "чем",
    "была",
    "сам",
    "чтоб",
    "без",
    "будто",
    "человек",
    "чего",
    "раз",
    "тоже",
    "себе",
    "под",
    "жизнь",
    "будет",
    "ж",
    "тогда",
    "кто",
    "этот",
    "говорил",
    "того",
    "потому",
    "этого",
    "какой",
    "совсем",
    "ним",
    "здесь",
    "этом",
    "один",
    "почти",
    "мой",
    "тем",
    "чтобы",
    "нее",
    "кажется",
    "сейчас",
    "были",
    "куда",
    "зачем",
    "сказать",
    "всех",
    "никогда",
    "сегодня",
    "можно",
    "при",
    "наконец",
    "два",
    "об",
    "другой",
    "хоть",
    "после",
    "над",
    "больше",
    "тот",
    "через",
    "эти",
    "нас",
    "про",
    "всего",
    "них",
    "какая",
    "много",
    "разве",
    "сказала",
    "три",
    "эту",
    "моя",
    "впрочем",
    "хорошо",
    "свою",
    "этой",
    "перед",
    "иногда",
    "лучше",
    "чуть",
    "том",
    "нельзя",
    "такой",
    "им",
    "более",
    "всегда",
    "конечно",
    "всю",
    "между",
];

pub const SPANISH: &[&str] = &[
    "de",
    "la",
    "que",
    "el",
    "en",
    "y",
    "a",
    "los",
    "del",
    "se",
    "las",
    "por",
    "un",
    "para",
    "con",
    "no",
    "una",
    "su",
    "al",
    "lo",
    "como",
    "más",
    "pero",
    "sus",
    "le",
    "ya",
    "o",
    "este",
    "sí",
    "porque",
    "esta",
    "entre",
    "cuando",
    "muy",
    "sin",
    "sobre",
    "también",
    "me",
    "hasta",
    "hay",
    "donde",
    "quien",
    "desde",
    "todo",
    "nos",
    "durante",
    "todos",
    "uno",
    "les",
    "ni",
    "contra",
    "otros",
    "ese",
    "eso",
    "ante",
    "ellos",
    "e",
    "esto",
    "mí",
    "antes",
    "algunos",
    "qué",
    "unos",
    "yo",
    "otro",
    "otras",
    "otra",
    "él",
    "tanto",
    "esa",
    "estos",
    "mucho",
    "quienes",
    "nada",
    "muchos",
    "cual",
    "poco",
    "ella",
    "estar",
    "estas",
    "algunas",
    "algo",
    "nosotros",
    "mi",
    "mis",
    "tú",
    "te",
    "ti",
    "tu",
    "tus",
    "ellas",
    "nosotras",
    "vosotros",
    "vosotras",
    "os",
    "mío",
    "mía",
    "míos",
    "mías",
    "tuyo",
    "tuya",
    "tuyos",
    "tuyas",
    "suyo",
    "suya",
    "suyos",
    "suyas",
    "nuestro",
    "nuestra",
    "nuestros",
    "nuestras",
    "vuestro",
    "vuestra",
    "vuestros",
    "vuestras",
    "esos",
    "esas",
    "estoy",
    "estás",
    "está",
    "estamos",
    "estáis",
    "están",
    "esté",
    "estés",
    "estemos",
    "estéis",
    "estén",
    "estaré",
    "estarás",
    "estará",
    "estaremos",
    "estaréis",
    "estarán",
    "estaría",
    "estarías",
    "estaríamos",
    "estaríais",
    "estarían",
    "estaba",
    "estabas",
    "estábamos",
    "estabais",
    "estaban",
    "estuve",
    "estuviste",
    "estuvo",
    "estuvimos",
    "estuvisteis",
    "estuvieron",
    "estuviera",
    "estuvieras",
    "estuviéramos",
    "estuvierais",
    "estuvieran",
    "estuviese",
    "estuvieses",
    "estuviésemos",
    "estuvieseis",
    "estuviesen",
    "estando",
    "estado",
    "estada",
    "estados",
    "estadas",
    "estad",
    "he",
    "has",
    "ha",
    "hemos",
    "habéis",
    "han",
    "haya",
    "hayas",
    "hayamos",
    "hayáis",
    "hayan",
    "habré",
    "habrás",
    "habrá",
    "habremos",
    "habréis",
    "habrán",
    "habría",
    "habrías",
    "habríamos",
    "habríais",
    "habrían",
    "había",
    "habías",
    "habíamos",
    "habíais",
    "habían",
    "hube",
    "hubiste",
    "hubo",
    "hubimos",
    "hubisteis",
    "hubieron",
    "hubiera",
    "hubieras",
    "hubiéramos",
    "hubierais",
    "hubieran",
    "hubiese",
    "hubieses",
    "hubiésemos",
    "hubieseis",
    "hubiesen",
    "habiendo",
    "habido",
    "habida",
    "habidos",
    "habidas",
    "soy",
    "eres",
    "es",
    "somos",
    "sois",
    "son",
    "sea",
    "seas",
    "seamos",
    "seáis",
    "sean",
    "seré",
    "serás",
    "será",
    "seremos",
    "seréis",
    "serán",
    "sería",
    "serías",
    "seríamos",
    "seríais",
    "serían",
    "era",
    "eras",
    "éramos",
    "erais",
    "eran",
    "fui",
    "fuiste",
    "fue",
    "fuimos",
    "fuisteis",
    "fueron",
    "fuera",
    "fueras",
    "fuéramos",
    "fuerais",
    "fueran",
    "fuese",
    "fueses",
    "fuésemos",
    "fueseis",
    "fuesen",
    "siendo",
    "sido",
    "tengo",
    "tienes",
    "tiene",
    "tenemos",
    "tenéis",
    "tienen",
    "tenga",
    "tengas",
    "tengamos",
    "tengáis",
    "tengan",
    "tendré",
    "tendrás",
    "tendrá",
    "tendremos",
    "tendréis",
    "tendrán",
    "tendría",
    "tendrías",
    "tendríamos",
    "tendríais",
    "tendrían",
    "tenía",
    "tenías",
    "teníamos",
    "teníais",
    "tenían",
    "tuve",
    "tuviste",
    "tuvo",
    "tuvimos",
    "tuvisteis",
    "tuvieron",
    "tuviera",
    "tuvieras",
    "tuviéramos",
    "tuvierais",
    "tuvieran",
    "tuviese",
    "tuvieses",
    "tuviésemos",
    "tuvieseis",
    "tuviesen",
    "teniendo",
    "tenido",
    "tenida",
    "tenidos",
    "tenidas",
    "tened",
];

pub const SWEDISH: &[&str] = &[
    "och", "det", "att", "i", "en", "jag", "hon", "som", "han", "på", "den", "med", "var", "sig",
    "för", "så", "till", "är", "men", "ett", "om", "hade", "de", "av", "icke", "mig", "du",
    "henne", "då", "sin", "nu", "har", "inte", "hans", "honom", "skulle", "hennes", "där", "min",
    "man", "ej", "vid", "kunde", "något", "från", "ut", "när", "efter", "upp", "vi", "dem", "vara",
    "vad", "över", "än", "dig", "kan", "sina", "här", "ha", "mot", "alla", "under", "någon",
    "eller", "allt", "mycket", "sedan", "ju", "denna", "själv", "detta", "åt", "utan", "varit",
    "hur", "ingen", "mitt", "ni", "bli", "blev", "oss", "din", "dessa", "några", "deras", "blir",
    "mina", "samma", "vilken", "er", "sådan", "vår", "blivit", "dess", "inom", "mellan", "sådant",
    "varför", "varje", "vilka", "ditt", "vem", "vilket", "sitt", "sådana", "vart", "dina", "vars",
    "vårt", "våra", "ert", "era", "vilkas",
];

pub static CHINESE_DATA: &[u8] = include_bytes!("chinese.txt");

pub static CHINESE: Lazy<&[&str]> = Lazy::new(|| fetch_words_from_file(CHINESE_DATA));

#[cfg(test)]
mod tests {
    use super::fetch_language_stop_words;

    #[test]
    fn test_chinese_stop_words() {
        let words = fetch_language_stop_words("_chinese_").unwrap();
        assert!(
            words.len() == 840,
            "number of system Chinese stop words does not match the expected value"
        );
        for s in words {
            print!("{}\n", s);
        }
    }
}
