// Copyright (C) 2019-2025 Zilliz. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance
// with the License. You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software distributed under the License
// is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
// or implied. See the License for the specific language governing permissions and limitations under the License

//go:build exclude

#include "milvus_plan_parser.h"

#include <cstring>
#include <stdexcept>

// This header is generated by the Go build (cgo)
// and is expected to be available in the include path.
extern "C" {
    #include "libmilvus-planparser.h"
}

namespace milvus {
namespace planparserv2 {

SchemaHandle PlanParser::RegisterSchema(const std::vector<uint8_t>& schema_proto) {
    void* proto_blob = const_cast<void*>(static_cast<const void*>(schema_proto.data()));
    int len = static_cast<int>(schema_proto.size());
    char* err_msg = nullptr;

    SchemaHandle handle = ::RegisterSchema(proto_blob, len, &err_msg);
    if (handle == kInvalidSchemaHandle) {
        std::string err_str = "Unknown error";
        if (err_msg != nullptr) {
            err_str = std::string(err_msg);
            ::Free(err_msg);
        }
        throw std::runtime_error("Failed to register schema: " + err_str);
    }

    return handle;
}

std::string PlanParser::UnregisterSchema(SchemaHandle handle) {
    char* err_msg = nullptr;

    int result = ::UnregisterSchema(handle, &err_msg);
    if (result != 0) {
        std::string err = err_msg ? std::string(err_msg) : "unknown error";
        if (err_msg != nullptr) {
            ::Free(err_msg);
        }
        return err;
    }
    return "";
}

std::vector<uint8_t> PlanParser::Parse(SchemaHandle handle, const std::string& expr) {
    char* c_expr = const_cast<char*>(expr.c_str());
    char* err_msg = nullptr;
    int length = 0;

    void* result = ::Parse(handle, c_expr, &length, &err_msg);
    if (result == nullptr) {
        std::string err_str = "Unknown error";
        if (err_msg != nullptr) {
            err_str = std::string(err_msg);
            ::Free(err_msg);
        }
        throw std::runtime_error("Failed to parse expression: " + err_str);
    }

    std::vector<uint8_t> plan(length);
    if (length > 0) {
        std::memcpy(plan.data(), result, length);
    }

    ::Free(result);

    return plan;
}

} // namespace planparserv2
} // namespace milvus
