import datetime
import random

from mindsdb.api.executor.data_types.response_type import RESPONSE_TYPE
from tests.integration.utils.http_test_helpers import HTTPHelperMixin


def to_dicts(data):
    data = [{
        'date': datetime.datetime.strptime(x[0].split(' ')[0], '%Y-%m-%d').date(),
        'group': x[1],
        'value': x[2]
    } for x in data]
    data.sort(key=lambda x: x['date'])
    return data


class TestHTTP(HTTPHelperMixin):
    @classmethod
    def setup_class(cls):
        cls._sql_via_http_context = {}

    def test_create_model(self, train_finetune_lock):
        self.sql_via_http("DROP DATABASE IF EXISTS test_ts_demo_postgres;", RESPONSE_TYPE.OK)
        sql = '''
        CREATE DATABASE test_ts_demo_postgres
        WITH ENGINE = "postgres",
        PARAMETERS = {
            "user": "demo_user",
            "password": "demo_password",
            "host": "samples.mindsdb.com",
            "port": "5432",
            "database": "demo"
            };
        '''
        resp = self.sql_via_http(sql, RESPONSE_TYPE.OK)

        groups = ['a', 'b']
        selects = []
        for i in range(30):
            day_str = str(datetime.date.today() + datetime.timedelta(days=i))
            for group in groups:
                value = random.randint(0, 10)
                selects.append(f"select '{day_str}' as date, '{group}' as group, {value} as value")
        selects = ' union all '.join(selects)

        self.sql_via_http("DROP VIEW IF EXISTS testv;", RESPONSE_TYPE.OK)
        sql = f'''
            create view testv as (
                select * from test_ts_demo_postgres ({selects})
            )
        '''
        self.sql_via_http(sql, RESPONSE_TYPE.OK)

        self.sql_via_http("DROP MODEL IF EXISTS mindsdb.tstest;", RESPONSE_TYPE.OK)
        with train_finetune_lock.acquire(timeout=600):
            sql = '''
                CREATE MODEL
                    mindsdb.tstest
                FROM mindsdb (select * from testv)
                PREDICT value
                ORDER BY date
                GROUP BY group
                WINDOW 5
                HORIZON 3;
            '''
            resp = self.sql_via_http(sql, RESPONSE_TYPE.TABLE)

            assert len(resp['data']) == 1
            status = resp['column_names'].index('STATUS')
            assert resp['data'][0][status] == 'generating'

            self.await_model('tstest')

    def test_gt_latest_date(self):
        sql = '''
            select p.date, p.group, p.value
            from mindsdb.testv as t join mindsdb.tstest as p
            where t.date > LATEST
        '''
        resp = self.sql_via_http(sql, RESPONSE_TYPE.TABLE)
        data = to_dicts(resp['data'])
        assert len(data) == 6
        assert len([x for x in data if x['group'] == 'a']) == 3
        assert data[0]['date'] == (datetime.date.today() + datetime.timedelta(days=30))

    def test_gt_latest_date_empty_join(self):
        sql = '''
            select p.date, p.group, p.value
            from mindsdb.testv as t join mindsdb.tstest as p
            where t.date > LATEST and t.group = 'wrong'
        '''
        resp = self.sql_via_http(sql, RESPONSE_TYPE.TABLE)
        data = to_dicts(resp['data'])
        assert len(data) == 0

    def test_eq_latest_date(self):
        sql = '''
            select p.date, p.group, p.value
            from mindsdb.testv as t join mindsdb.tstest as p
            where t.date = LATEST
        '''
        resp = self.sql_via_http(sql, RESPONSE_TYPE.TABLE)
        data = to_dicts(resp['data'])
        assert len(data) == 2
        assert len([x for x in data if x['group'] == 'a']) == 1
        assert data[0]['date'] == (datetime.date.today() + datetime.timedelta(days=29))

    def test_gt_particular_date(self):
        since = datetime.date.today() + datetime.timedelta(days=15)
        sql = f'''
            select p.date, p.group, p.value
            from mindsdb.testv as t join mindsdb.tstest as p
            where t.date > '{since}'
        '''
        resp = self.sql_via_http(sql, RESPONSE_TYPE.TABLE)
        data = to_dicts(resp['data'])
        assert len(data) == 34  # 14 * 2 + 6 (4 days, 2 groups, 2*3 horizon)
        assert len([x for x in data if x['group'] == 'a']) == 17  # 14 + 3
        assert data[0]['date'] == (datetime.date.today() + datetime.timedelta(days=16))

    def test_eq_particular_date(self):
        since = datetime.date.today() + datetime.timedelta(days=15)
        sql = f'''
            select p.date, p.group, p.value
            from mindsdb.testv as t join mindsdb.tstest as p
            where t.date = '{since}'
        '''
        resp = self.sql_via_http(sql, RESPONSE_TYPE.TABLE)
        data = to_dicts(resp['data'])
        assert len(data) == 6  # 2 groups * 3 horizon
        assert len([x for x in data if x['group'] == 'a']) == 3
        assert data[0]['date'] == (datetime.date.today() + datetime.timedelta(days=16))
