# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import pytest

from pypinyin import pinyin, load_phrases_dict
from pypinyin.contrib import mmseg

seg_test = mmseg.Seg(mmseg.PrefixSet())
seg_test._prefix_set.train([
    'a',
    'ab',
    'abc',
    'abcd',
    'abd',
    'ac',
    'acd',
    'aff',
    'agf',
    'agfgef',
    'asdf',
    'bbs'
    '中国',
    '中国人',
    '中国人民',
    '中国人民银行',
    '我',
    '北京',
    '天安门',
    '员工',
])


@pytest.mark.parametrize(
    'input, expect', [
        ['', []],
        ['a', ['a']],
        ['abc', ['abc']],
        ['abcefg', ['abc', 'e', 'f', 'g']],
        ['bbcabce', ['bb', 'c', 'abc', 'e']],
        ['北京', ['北京']],
        ['北京,', ['北京', ',']],
        ['北京abc', ['北京', 'abc']],
        ['中国人民银行行长', ['中国人民银行', '行', '长']],
        ['中国人民银行员工', ['中国人民银行', '员工']],
        [
            'abcadbasfgafgasdabcagfaff我是中国人中国人民我爱北京天安门',
            [
                'abc',
                'a',
                'd',
                'b',
                'as',
                'f',
                'g',
                'af',
                'g',
                'asd',
                'abc',
                'agf',
                'aff',
                '我',
                '是',
                '中国人',
                '中国人民',
                '我',
                '爱',
                '北京',
                '天安门',
            ],
        ],
    ]
)
def test_mmseg(input, expect):
    assert list(seg_test.cut(input)) == expect


@pytest.mark.parametrize(
    'input, default_ret, mmseg_ret', [
        [
            '一语中的啊',
            [['yī'], ['yǔ'], ['zhōng'], ['de'], ['a']],
            [['yī'], ['yǔ'], ['zhòng'], ['dì'], ['a']],
        ],
    ]
)
def test_mmseg_for_pinyin(input, default_ret, mmseg_ret):
    assert pinyin(input) == mmseg_ret
    assert pinyin(mmseg.seg.cut(input)) == mmseg_ret


@pytest.mark.parametrize(
    'input, jieba_ret, mmseg_ret', [
        [
            '了局啊',
            [['le'], ['jú'], ['a']],
            [['liǎo'], ['jú'], ['a']],
        ],
    ]
)
def test_mmseg_and_jieba_for_pinyin(input, jieba_ret, mmseg_ret):
    assert pinyin(input) == mmseg_ret
    assert pinyin(mmseg.seg.cut(input)) == mmseg_ret


def test_retrain():
    seg = mmseg.seg
    assert list(seg.cut('啊啊啊')) == ['啊', '啊', '啊']

    load_phrases_dict({'啊啊啊': [['a'], ['a'], ['a']]})
    mmseg.retrain(seg)
    assert list(seg.cut('啊啊啊')) == ['啊啊啊']

    assert list(seg.cut('男孩儿')) == ['男孩儿']


def test_phrases():
    seg = mmseg.seg
    assert list(seg.cut('你要重新考虑这条建议')) == \
           ['你', '要', '重新', '考', '虑', '这', '条', '建', '议']
    load_phrases_dict({'在一起': [['zài'], ['yì'], ['qǐ']]})
    assert list(seg.cut('在一片')) == ['在', '一片']

    # 前缀匹配，后缀是词语
    #
    # 输入头部是另外一个词语的头部，会匹配其他词语的前缀
    # 输入尾部是一个词语
    # 此时这个尾部词语要被分词出来
    assert list(seg.cut('行业')) == ['行业']
    assert list(seg.cut('金融行业')) == ['金', '融', '行业']
    # 整个是词语
    assert list(seg.cut('金融寡头')) == ['金融寡头']
    assert list(seg.cut('服务行业')) == ['服务行业']
    assert list(seg.cut('人员')) == ['人员']
    assert list(seg.cut('服务人员')) == ['服务', '人员']

    assert list(seg.cut('银行')) == ['银行']
    assert list(seg.cut('浦发银行')) == ['浦', '发', '银行']

    assert list(seg.cut('')) == []
    # 整个匹配前缀，但是不是词语
    assert list(seg.cut('金')) == ['金']
    assert list(seg.cut('金融')) == ['金', '融']
    #
    assert list(seg.cut('金融金')) == ['金', '融', '金']
    assert list(seg.cut('金融金融')) == ['金', '融', '金', '融']
    assert list(seg.cut('金融金融金融金融金融金融')) == [
        '金', '融', '金', '融', '金', '融', '金', '融', '金', '融', '金', '融']
    assert list(seg.cut('金融金融金融金融金融金融金')) == [
        '金', '融', '金', '融', '金', '融', '金', '融', '金', '融', '金',
        '融', '金']

    # 没有任何匹配
    assert list(seg.cut('以其昏昏，使人昭昭')) == [
        '以', '其', '昏', '昏', '，', '使', '人', '昭', '昭']

    # 前缀无任何匹配， 后缀是词语
    assert list(seg.cut('以其昏昏行业')) == [
        '以', '其', '昏', '昏', '行业']

    # 前缀是词语
    assert list(seg.cut('行业以其昏昏')) == [
        '行业', '以', '其', '昏', '昏']

    # 中间是词语
    assert list(seg.cut('使人昭昭行业以其昏昏')) == [
        '使', '人', '昭', '昭', '行业', '以', '其', '昏', '昏']


def test_seg_long():
    seg = mmseg.seg
    assert list(seg.cut('''    真的猛士，敢于直面惨淡的人生，敢于正视淋漓的鲜血。
    这是怎样的哀痛者和幸福者？然而造化又常常为庸人设计，以时间的流驶，来洗涤旧迹，
    仅使留下淡红的血色和微漠的悲哀。在这淡红的血色和微漠的悲哀中，又给人暂得偷生，
    维持着这似人非人的世界。我不知道这样的世界何时是一个尽头！
　　我们还在这样的世上活着；我也早觉得有写一点东西的必要了。离三月十八日也已有两星期，
    忘却的救主快要降临了罢，我正有写一点东西的必要了。''')) == [
        ' ',
        ' ',
        ' ',
        ' ',
        '真',
        '的',
        '猛',
        '士',
        '，',
        '敢',
        '于',
        '直',
        '面',
        '惨',
        '淡',
        '的',
        '人',
        '生',
        '，',
        '敢',
        '于',
        '正视',
        '淋',
        '漓',
        '的',
        '鲜血',
        '。',
        '\n',
        ' ',
        ' ',
        ' ',
        ' ',
        '这',
        '是',
        '怎',
        '样',
        '的',
        '哀',
        '痛',
        '者',
        '和',
        '幸',
        '福',
        '者',
        '？',
        '然',
        '而',
        '造化',
        '又',
        '常',
        '常',
        '为',
        '庸',
        '人',
        '设',
        '计',
        '，',
        '以',
        '时间',
        '的',
        '流',
        '驶',
        '，',
        '来',
        '洗涤',
        '旧',
        '迹',
        '，',
        '\n',
        ' ',
        ' ',
        ' ',
        ' ',
        '仅',
        '使',
        '留',
        '下',
        '淡红',
        '的',
        '血色',
        '和',
        '微',
        '漠',
        '的',
        '悲',
        '哀',
        '。',
        '在',
        '这',
        '淡红',
        '的',
        '血色',
        '和',
        '微',
        '漠',
        '的',
        '悲',
        '哀',
        '中',
        '，',
        '又',
        '给',
        '人',
        '暂',
        '得',
        '偷',
        '生',
        '，',
        '\n',
        ' ',
        ' ',
        ' ',
        ' ',
        '维',
        '持',
        '着',
        '这',
        '似',
        '人',
        '非',
        '人',
        '的',
        '世',
        '界',
        '。',
        '我',
        '不',
        '知道',
        '这样',
        '的',
        '世',
        '界',
        '何时',
        '是',
        '一个',
        '尽头',
        '！',
        '\n',
        '\u3000',
        '\u3000',
        '我们',
        '还',
        '在',
        '这样',
        '的',
        '世上',
        '活着',
        '；',
        '我',
        '也',
        '早',
        '觉得',
        '有',
        '写',
        '一点',
        '东',
        '西',
        '的',
        '必要',
        '了',
        '。',
        '离',
        '三',
        '月',
        '十',
        '八',
        '日',
        '也',
        '已',
        '有',
        '两',
        '星期',
        '，',
        '\n',
        ' ',
        ' ',
        ' ',
        ' ',
        '忘',
        '却',
        '的',
        '救',
        '主',
        '快要',
        '降临',
        '了',
        '罢',
        '，',
        '我',
        '正',
        '有',
        '写',
        '一点',
        '东',
        '西',
        '的',
        '必要',
        '了',
        '。']

    assert list(seg.cut(
        '人们常常把人与自然对立起来，宣称要征服自然。殊不知在大自然面前，'
        '人类永远只是一个天真幼稚的孩童，而他却要作自然的主人！'
        '他只是大自然机体上普通的一部分，正像一株小草只是她的普通一部分一样，'
        '有什么资格与自然对立！ 如果说自然的智慧是大海，那么，'
        '人类的智慧就只是大海中的一个小水滴，虽然这个水滴也映照着大海，'
        '但毕竟不是大海。可是，人们竟然不自量力 地宣称要用滴水来代替大海。')) == [
               '人们',
               '常',
               '常',
               '把',
               '人',
               '与',
               '自',
               '然',
               '对',
               '立',
               '起',
               '来',
               '，',
               '宣称',
               '要',
               '征服',
               '自',
               '然',
               '。',
               '殊不知',
               '在',
               '大自然',
               '面',
               '前',
               '，',
               '人',
               '类',
               '永远',
               '只是',
               '一个',
               '天',
               '真',
               '幼',
               '稚',
               '的',
               '孩',
               '童',
               '，',
               '而',
               '他',
               '却',
               '要',
               '作',
               '自',
               '然',
               '的',
               '主',
               '人',
               '！',
               '他',
               '只是',
               '大自然',
               '机体',
               '上',
               '普',
               '通',
               '的',
               '一',
               '部分',
               '，',
               '正像',
               '一',
               '株',
               '小',
               '草',
               '只是',
               '她',
               '的',
               '普',
               '通',
               '一',
               '部分',
               '一',
               '样',
               '，',
               '有',
               '什么',
               '资',
               '格',
               '与',
               '自',
               '然',
               '对',
               '立',
               '！',
               ' ',
               '如',
               '果',
               '说',
               '自',
               '然',
               '的',
               '智',
               '慧',
               '是',
               '大海',
               '，',
               '那么',
               '，',
               '人',
               '类',
               '的',
               '智',
               '慧',
               '就',
               '只是',
               '大海',
               '中的',
               '一个',
               '小',
               '水',
               '滴',
               '，',
               '虽',
               '然',
               '这个',
               '水',
               '滴',
               '也',
               '映',
               '照着',
               '大海',
               '，',
               '但',
               '毕',
               '竟',
               '不是',
               '大海',
               '。',
               '可是',
               '，',
               '人们',
               '竟',
               '然',
               '不自量力',
               ' ',
               '地',
               '宣称',
               '要',
               '用',
               '滴',
               '水',
               '来',
               '代',
               '替',
               '大海',
               '。']

    assert list(seg.cut(
        '该负责人表示，银行保险机构具有外部性强、财务杠杆率高、'
        '信息不对称严重等特征，不同于一般工商企业，对其股东股权必须从严管理。'
        '总体上，银保监会将坚持“两个不变”。一是坚持鼓励社会资本参与银行保险机构改革、'
        '优化股东结构的积极取向不变。银保监会将继续畅通社会资本投资入股银行保险机构的渠道，'
        '优化股东结构、充实机构资本，重点引入注重机构长远发展、资本实力雄厚、管理经验'
        '丰富的战略性股东。二是坚持严惩股东违法违规行为、规范公司治理的高压态势不变。'
        '银保监会将继续深入排查整治违法违规股东股权，依法清理规范股权关系，'
        '对股东严重违法违规行为，将坚决予以惩戒，发现一起、查处一起。')) == [
               '该',
               '负',
               '责',
               '人',
               '表',
               '示',
               '，',
               '银行',
               '保',
               '险',
               '机',
               '构',
               '具有',
               '外',
               '部',
               '性',
               '强',
               '、',
               '财',
               '务',
               '杠杆',
               '率',
               '高',
               '、',
               '信息',
               '不对',
               '称',
               '严重',
               '等',
               '特',
               '征',
               '，',
               '不同',
               '于',
               '一般',
               '工',
               '商',
               '企',
               '业',
               '，',
               '对',
               '其',
               '股',
               '东',
               '股',
               '权',
               '必',
               '须',
               '从',
               '严',
               '管',
               '理',
               '。',
               '总体',
               '上',
               '，',
               '银',
               '保',
               '监',
               '会',
               '将',
               '坚',
               '持',
               '“',
               '两个',
               '不变',
               '”',
               '。',
               '一',
               '是',
               '坚',
               '持',
               '鼓',
               '励',
               '社会',
               '资',
               '本',
               '参与',
               '银行',
               '保',
               '险',
               '机',
               '构',
               '改',
               '革',
               '、',
               '优化',
               '股',
               '东',
               '结构',
               '的',
               '积',
               '极',
               '取',
               '向',
               '不变',
               '。',
               '银',
               '保',
               '监',
               '会',
               '将',
               '继',
               '续',
               '畅',
               '通',
               '社会',
               '资',
               '本',
               '投',
               '资',
               '入',
               '股',
               '银行',
               '保',
               '险',
               '机',
               '构',
               '的',
               '渠道',
               '，',
               '优化',
               '股',
               '东',
               '结构',
               '、',
               '充',
               '实',
               '机',
               '构',
               '资',
               '本',
               '，',
               '重点',
               '引',
               '入',
               '注重',
               '机',
               '构',
               '长远',
               '发展',
               '、',
               '资',
               '本',
               '实',
               '力',
               '雄',
               '厚',
               '、',
               '管',
               '理',
               '经',
               '验',
               '丰',
               '富',
               '的',
               '战',
               '略',
               '性',
               '股',
               '东',
               '。',
               '二',
               '是',
               '坚',
               '持',
               '严',
               '惩',
               '股',
               '东',
               '违',
               '法',
               '违',
               '规',
               '行为',
               '、',
               '规',
               '范',
               '公',
               '司',
               '治',
               '理',
               '的',
               '高压',
               '态',
               '势',
               '不变',
               '。',
               '银',
               '保',
               '监',
               '会',
               '将',
               '继',
               '续',
               '深',
               '入',
               '排',
               '查',
               '整',
               '治',
               '违',
               '法',
               '违',
               '规',
               '股',
               '东',
               '股',
               '权',
               '，',
               '依',
               '法',
               '清',
               '理',
               '规',
               '范',
               '股',
               '权',
               '关系',
               '，',
               '对',
               '股',
               '东',
               '严重',
               '违',
               '法',
               '违',
               '规',
               '行为',
               '，',
               '将',
               '坚',
               '决',
               '予以',
               '惩',
               '戒',
               '，',
               '发现',
               '一起',
               '、',
               '查处',
               '一起',
               '。']


@pytest.mark.parametrize(
    'words,no_non_phrases,expected', [
        ('银行行长', True, ['银行', '行长']),
        ('银行行长', False, ['银行行', '长']),
        ('银行', True, ['银行']),
        ('银行', False, ['银行']),
        ('银行行员', True, ['银行行员']),
        ('银行行员', False, ['银行行员']),
        ('金融资本', True, ['金', '融', '资', '本']),
        ('金融资本', False, ['金融资本']),
        ('金融行业', True, ['金', '融', '行业']),
        ('金融行业', False, ['金融', '行业']),
        ('军工', True, ['军', '工']),
        ('军工', False, ['军', '工']),
        ('浦发银行', True, ['浦', '发', '银行']),
        ('浦发银行', False, ['浦', '发', '银行']),
    ]
)
def test_issue_347(words, no_non_phrases, expected):
    prefix_set = mmseg.PrefixSet()
    prefix_set.train([
        '银行', '行长', '银行行员',
        '金融', '融资', '资本', '金融资本',
        '行业',
    ])
    seg = mmseg.Seg(prefix_set, no_non_phrases=no_non_phrases)
    assert list(seg.cut(words)) == expected


if __name__ == '__main__':
    import pytest

    pytest.cmdline.main()
