/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.neo4j.collection;

import static org.junit.jupiter.api.Assertions.assertEquals;

import java.util.concurrent.atomic.AtomicInteger;
import org.junit.jupiter.api.Test;
import org.neo4j.graphdb.Resource;

class PrimitiveLongResourceCollectionsTest {

    // ITERATOR

    @Test
    void simpleIterator() {
        // Given
        CountingResource resource = new CountingResource();
        PrimitiveLongResourceIterator iterator = PrimitiveLongResourceCollections.iterator(resource, 1, 2, 3, 4);

        // Then
        assertContent(iterator, 1, 2, 3, 4);

        // When
        iterator.close();

        // Then
        assertEquals(1, resource.closeCount(), "exactly one call to close");
    }

    // FILTER

    // CONCAT

    @Test
    void concatIterators() {
        // Given
        CountingResource resource = new CountingResource();
        PrimitiveLongResourceIterator first = PrimitiveLongResourceCollections.iterator(resource, 1, 2);
        PrimitiveLongResourceIterator second = PrimitiveLongResourceCollections.iterator(resource, 3, 4);

        // When
        PrimitiveLongResourceIterator concat = PrimitiveLongResourceCollections.concat(first, second);

        // Then
        assertContent(concat, 1, 2, 3, 4);

        // When
        concat.close();

        // Then
        assertEquals(2, resource.closeCount(), "all concatenated iterators are closed");
    }

    private static void assertContent(PrimitiveLongResourceIterator iterator, long... expected) {
        int i = 0;
        while (iterator.hasNext()) {
            assertEquals(expected[i++], iterator.next(), "has expected value");
        }
        assertEquals(expected.length, i, "has all expected values");
    }

    private static class CountingResource implements Resource {
        private final AtomicInteger closed = new AtomicInteger();

        @Override
        public void close() {
            closed.incrementAndGet();
        }

        int closeCount() {
            return closed.get();
        }
    }
}
