/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.neo4j.cypher.internal.ast.semantics

import org.neo4j.cypher.internal.ast.SemanticCheckInTest.SemanticCheckWithDefaultContext
import org.neo4j.cypher.internal.expressions.PropertyKeyName
import org.neo4j.cypher.internal.expressions.Variable
import org.neo4j.cypher.internal.util.symbols.CTAny
import org.neo4j.cypher.internal.util.symbols.CTDate
import org.neo4j.cypher.internal.util.symbols.CTDateTime
import org.neo4j.cypher.internal.util.symbols.CTDuration
import org.neo4j.cypher.internal.util.symbols.CTInteger
import org.neo4j.cypher.internal.util.symbols.CTLocalDateTime
import org.neo4j.cypher.internal.util.symbols.CTLocalTime
import org.neo4j.cypher.internal.util.symbols.CTMap
import org.neo4j.cypher.internal.util.symbols.CTNode
import org.neo4j.cypher.internal.util.symbols.CTPoint
import org.neo4j.cypher.internal.util.symbols.CTRelationship
import org.neo4j.cypher.internal.util.symbols.CTTime
import org.neo4j.cypher.internal.util.symbols.StorableType
import org.neo4j.gqlstatus.GqlParams

class PropertyTest extends SemanticFunSuite {

  Seq(CTMap, CTPoint, CTDate, CTTime, CTLocalTime, CTLocalDateTime, CTDateTime, CTDuration).foreach { cypherType =>
    test(s"accepts property access on a $cypherType") {
      val mapExpr: Variable = variable("map")
      val propertyKey: PropertyKeyName = propertyKeyName("prop")

      val beforeState = SemanticState.clean.newChildScope.declareVariable(mapExpr, cypherType).right.get

      val propExpr = property(mapExpr, propertyKey)
      val result = SemanticExpressionCheck.simple(propExpr).run(beforeState)

      result.errors shouldBe empty
      types(propExpr)(result.state) should equal(CTAny.covariant)
    }
  }

  test("accepts property access on a node") {
    val mapExpr: Variable = variable("map")
    val propertyKey: PropertyKeyName = propertyKeyName("prop")

    val beforeState = SemanticState.clean.newChildScope.declareVariable(mapExpr, CTNode).right.get

    val propExpr = property(mapExpr, propertyKey)
    val result = SemanticExpressionCheck.simple(property(mapExpr, propertyKey)).run(beforeState)

    result.errors shouldBe empty
    types(propExpr)(result.state) should equal(StorableType.storableType)
  }

  test("accepts property access on a relationship") {
    val mapExpr: Variable = variable("map")
    val propertyKey: PropertyKeyName = propertyKeyName("prop")

    val beforeState = SemanticState.clean.newChildScope.declareVariable(mapExpr, CTRelationship).right.get

    val propExpr = property(mapExpr, propertyKey)
    val result = SemanticExpressionCheck.simple(propExpr).run(beforeState)

    result.errors shouldBe empty
    types(propExpr)(result.state) should equal(StorableType.storableType)
  }

  test("refuses property access on an Integer") {
    val mapExpr: Variable = variable("map")
    val propertyKey: PropertyKeyName = propertyKeyName("prop")

    val beforeState = SemanticState.clean.newChildScope.declareVariable(mapExpr, CTInteger).right.get

    val result = SemanticExpressionCheck.simple(property(mapExpr, propertyKey)).run(beforeState)

    result.errors should equal(Seq(SemanticError.invalidEntityType(
      "Integer",
      GqlParams.StringParam.ident.process("map"),
      List(
        "Map",
        "Node",
        "Relationship",
        "Point",
        "Duration",
        "Date",
        "Time",
        "LocalTime",
        "LocalDateTime",
        "DateTime"
      ),
      "Type mismatch: expected Map, Node, Relationship, Point, Duration, Date, Time, LocalTime, LocalDateTime or DateTime but was Integer",
      pos
    )))
  }
}
