/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.neo4j.cypher.internal.parser.v25.ast.factory

import org.neo4j.cypher.internal.ast.factory.neo4j.completion.CodeCompletionCore.CandidateRule
import org.neo4j.cypher.internal.ast.factory.neo4j.completion.CodeCompletionCore.CandidatesCollection
import org.neo4j.cypher.internal.parser.CypherErrorVocabulary
import org.neo4j.cypher.internal.parser.v25.Cypher25Parser
import org.neo4j.cypher.internal.util.test_helpers.CypherFunSuite

class Cypher25VocabularyTest extends CypherFunSuite {
  val conf = new Cypher25ErrorStrategyConf
  val vocab = new CypherErrorVocabulary(conf)

  test("user facing token names are human readable") {
    Range.inclusive(1, vocab.getMaxTokenType).foreach { token =>
      val displayName = vocab.getDisplayName(token)
      val symbolicName = vocab.getSymbolicName(token)
      val literalName = vocab.getLiteralName(token)
      withClue(s"displayName=$displayName, symbolicName=$symbolicName, literalName=$literalName\n") {
        expectedDisplayNames.get(token) match {
          case Some(expected) =>
            if (displayName != expected) println(token)
            displayName shouldBe expected
          case _ => fail(
              s"""Hello! I can see that you have added a new token type ($symbolicName)
                 |You need to add an expected display name in ${getClass.getName}.
                 |The reason why we fail for every new token here is that these names
                 |are user facing and we want to make sure all have a readable name.
                 |""".stripMargin
            )
        }
      }
    }
  }

  test("rules of interest have a display name") {
    conf.preferredRules.foreach { r =>
      withClue(Cypher25Parser.ruleNames(r))(ruleDisplayName(r) should not be empty)
    }
    Range(0, Cypher25Parser.ruleNames.length).foreach { r =>
      withClue(Cypher25Parser.ruleNames(r))(ruleDisplayName(r) shouldBe expectedRuleDisplayNames.get(r).toSeq)
    }
  }

  private def ruleDisplayName(r: Int): Seq[String] = {
    val candidates = new CandidatesCollection()
    candidates.rules.put(java.lang.Integer.valueOf(r), new CandidateRule(-1, java.util.List.of[java.lang.Integer]()))
    vocab.expected(candidates)
  }

  private val expectedDisplayNames: Map[Int, String] = Map(
    Cypher25Parser.SPACE -> "' '",
    Cypher25Parser.SINGLE_LINE_COMMENT -> "'//'",
    Cypher25Parser.DECIMAL_DOUBLE -> "a float value",
    Cypher25Parser.UNSIGNED_DECIMAL_INTEGER -> "an integer value",
    Cypher25Parser.UNSIGNED_HEX_INTEGER -> "a hexadecimal integer value",
    Cypher25Parser.UNSIGNED_OCTAL_INTEGER -> "an octal integer value",
    Cypher25Parser.ACCESS -> "'ACCESS'",
    Cypher25Parser.ACTIVE -> "'ACTIVE'",
    Cypher25Parser.ADMIN -> "'ADMIN'",
    Cypher25Parser.ADMINISTRATOR -> "'ADMINISTRATOR'",
    Cypher25Parser.ALIAS -> "'ALIAS'",
    Cypher25Parser.ALIASES -> "'ALIASES'",
    Cypher25Parser.ALL_SHORTEST_PATHS -> "'allShortestPaths'",
    Cypher25Parser.ALL -> "'ALL'",
    Cypher25Parser.ALTER -> "'ALTER'",
    Cypher25Parser.AND -> "'AND'",
    Cypher25Parser.ANY -> "'ANY'",
    Cypher25Parser.ARRAY -> "'ARRAY'",
    Cypher25Parser.AS -> "'AS'",
    Cypher25Parser.ASC -> "'ASC'",
    Cypher25Parser.ASCENDING -> "'ASCENDING'",
    Cypher25Parser.ASSIGN -> "'ASSIGN'",
    Cypher25Parser.AT -> "'AT'",
    Cypher25Parser.AUTH -> "'AUTH'",
    Cypher25Parser.BAR -> "'|'",
    Cypher25Parser.BINDINGS -> "'BINDINGS'",
    Cypher25Parser.BOOL -> "'BOOL'",
    Cypher25Parser.BOOLEAN -> "'BOOLEAN'",
    Cypher25Parser.BOOSTED -> "'BOOSTED'",
    Cypher25Parser.BOTH -> "'BOTH'",
    Cypher25Parser.BREAK -> "'BREAK'",
    Cypher25Parser.BUILT -> "'BUILT'",
    Cypher25Parser.BY -> "'BY'",
    Cypher25Parser.CALL -> "'CALL'",
    Cypher25Parser.CASCADE -> "'CASCADE'",
    Cypher25Parser.CASE -> "'CASE'",
    Cypher25Parser.CHANGE -> "'CHANGE'",
    Cypher25Parser.CIDR -> "'CIDR'",
    Cypher25Parser.COLLECT -> "'COLLECT'",
    Cypher25Parser.COLON -> "':'",
    Cypher25Parser.COLONCOLON -> "'::'",
    Cypher25Parser.COMMA -> "','",
    Cypher25Parser.COMMAND -> "'COMMAND'",
    Cypher25Parser.COMMANDS -> "'COMMANDS'",
    Cypher25Parser.COMPOSITE -> "'COMPOSITE'",
    Cypher25Parser.CONCURRENT -> "'CONCURRENT'",
    Cypher25Parser.CONSTRAINT -> "'CONSTRAINT'",
    Cypher25Parser.CONSTRAINTS -> "'CONSTRAINTS'",
    Cypher25Parser.CONTAINS -> "'CONTAINS'",
    Cypher25Parser.COPY -> "'COPY'",
    Cypher25Parser.CONTINUE -> "'CONTINUE'",
    Cypher25Parser.COUNT -> "'COUNT'",
    Cypher25Parser.CREATE -> "'CREATE'",
    Cypher25Parser.CSV -> "'CSV'",
    Cypher25Parser.CURRENT -> "'CURRENT'",
    Cypher25Parser.DATA -> "'DATA'",
    Cypher25Parser.DATABASE -> "'DATABASE'",
    Cypher25Parser.DATABASES -> "'DATABASES'",
    Cypher25Parser.DATE -> "'DATE'",
    Cypher25Parser.DATETIME -> "'DATETIME'",
    Cypher25Parser.DBMS -> "'DBMS'",
    Cypher25Parser.DEALLOCATE -> "'DEALLOCATE'",
    Cypher25Parser.DEFAULT -> "'DEFAULT'",
    Cypher25Parser.DEFINED -> "'DEFINED'",
    Cypher25Parser.DELETE -> "'DELETE'",
    Cypher25Parser.DENY -> "'DENY'",
    Cypher25Parser.DESC -> "'DESC'",
    Cypher25Parser.DESCENDING -> "'DESCENDING'",
    Cypher25Parser.DESTROY -> "'DESTROY'",
    Cypher25Parser.DETACH -> "'DETACH'",
    Cypher25Parser.DIFFERENT -> "'DIFFERENT'",
    Cypher25Parser.DOLLAR -> "'$'",
    Cypher25Parser.DISTINCT -> "'DISTINCT'",
    Cypher25Parser.DIVIDE -> "'/'",
    Cypher25Parser.DOT -> "'.'",
    Cypher25Parser.DOTDOT -> "'..'",
    Cypher25Parser.DOUBLEBAR -> "'||'",
    Cypher25Parser.DRIVER -> "'DRIVER'",
    Cypher25Parser.DROP -> "'DROP'",
    Cypher25Parser.DRYRUN -> "'DRYRUN'",
    Cypher25Parser.DUMP -> "'DUMP'",
    Cypher25Parser.DURATION -> "'DURATION'",
    Cypher25Parser.EACH -> "'EACH'",
    Cypher25Parser.EDGE -> "'EDGE'",
    Cypher25Parser.ENABLE -> "'ENABLE'",
    Cypher25Parser.ELEMENT -> "'ELEMENT'",
    Cypher25Parser.ELEMENTS -> "'ELEMENTS'",
    Cypher25Parser.ELSE -> "'ELSE'",
    Cypher25Parser.ENCRYPTED -> "'ENCRYPTED'",
    Cypher25Parser.END -> "'END'",
    Cypher25Parser.ENDS -> "'ENDS'",
    Cypher25Parser.EQ -> "'='",
    Cypher25Parser.ErrorChar -> "'ErrorChar'",
    Cypher25Parser.EXECUTABLE -> "'EXECUTABLE'",
    Cypher25Parser.EXECUTE -> "'EXECUTE'",
    Cypher25Parser.EXIST -> "'EXIST'",
    Cypher25Parser.EXISTENCE -> "'EXISTENCE'",
    Cypher25Parser.EXISTS -> "'EXISTS'",
    Cypher25Parser.ERROR -> "'ERROR'",
    Cypher25Parser.FAIL -> "'FAIL'",
    Cypher25Parser.FALSE -> "'FALSE'",
    Cypher25Parser.FIELDTERMINATOR -> "'FIELDTERMINATOR'",
    Cypher25Parser.FINISH -> "'FINISH'",
    Cypher25Parser.FLOAT -> "'FLOAT'",
    Cypher25Parser.FOR -> "'FOR'",
    Cypher25Parser.FOREACH -> "'FOREACH'",
    Cypher25Parser.FROM -> "'FROM'",
    Cypher25Parser.FULLTEXT -> "'FULLTEXT'",
    Cypher25Parser.FUNCTION -> "'FUNCTION'",
    Cypher25Parser.FUNCTIONS -> "'FUNCTIONS'",
    Cypher25Parser.GE -> "'>='",
    Cypher25Parser.GRANT -> "'GRANT'",
    Cypher25Parser.GRAPH -> "'GRAPH'",
    Cypher25Parser.GRAPHS -> "'GRAPHS'",
    Cypher25Parser.GROUP -> "'GROUP'",
    Cypher25Parser.GROUPS -> "'GROUPS'",
    Cypher25Parser.GT -> "'>'",
    Cypher25Parser.HEADERS -> "'HEADERS'",
    Cypher25Parser.HOME -> "'HOME'",
    Cypher25Parser.ID -> "'ID'",
    Cypher25Parser.IF -> "'IF'",
    Cypher25Parser.IMPERSONATE -> "'IMPERSONATE'",
    Cypher25Parser.IMMUTABLE -> "'IMMUTABLE'",
    Cypher25Parser.IN -> "'IN'",
    Cypher25Parser.INDEX -> "'INDEX'",
    Cypher25Parser.INDEXES -> "'INDEXES'",
    Cypher25Parser.INF -> "'INF'",
    Cypher25Parser.INFINITY -> "'INFINITY'",
    Cypher25Parser.INSERT -> "'INSERT'",
    Cypher25Parser.INT -> "'INT'",
    Cypher25Parser.INTEGER -> "'INTEGER'",
    Cypher25Parser.IS -> "'IS'",
    Cypher25Parser.JOIN -> "'JOIN'",
    Cypher25Parser.KEY -> "'KEY'",
    Cypher25Parser.LABEL -> "'LABEL'",
    Cypher25Parser.LABELS -> "'LABELS'",
    Cypher25Parser.AMPERSAND -> "'&'",
    Cypher25Parser.EXCLAMATION_MARK -> "'!'",
    Cypher25Parser.LBRACKET -> "'['",
    Cypher25Parser.LCURLY -> "'{'",
    Cypher25Parser.LE -> "'<='",
    Cypher25Parser.LEADING -> "'LEADING'",
    Cypher25Parser.LIMITROWS -> "'LIMIT'",
    Cypher25Parser.LIST -> "'LIST'",
    Cypher25Parser.LOAD -> "'LOAD'",
    Cypher25Parser.LOCAL -> "'LOCAL'",
    Cypher25Parser.LOOKUP -> "'LOOKUP'",
    Cypher25Parser.LPAREN -> "'('",
    Cypher25Parser.LT -> "'<'",
    Cypher25Parser.MANAGEMENT -> "'MANAGEMENT'",
    Cypher25Parser.MAP -> "'MAP'",
    Cypher25Parser.MATCH -> "'MATCH'",
    Cypher25Parser.MERGE -> "'MERGE'",
    Cypher25Parser.MINUS -> "'-'",
    Cypher25Parser.PERCENT -> "'%'",
    Cypher25Parser.INVALID_NEQ -> "'!='",
    Cypher25Parser.NEQ -> "'<>'",
    Cypher25Parser.NAME -> "'NAME'",
    Cypher25Parser.NAMES -> "'NAMES'",
    Cypher25Parser.NAN -> "'NAN'",
    Cypher25Parser.NFC -> "'NFC'",
    Cypher25Parser.NFD -> "'NFD'",
    Cypher25Parser.NFKC -> "'NFKC'",
    Cypher25Parser.NFKD -> "'NFKD'",
    Cypher25Parser.NEW -> "'NEW'",
    Cypher25Parser.NODE -> "'NODE'",
    Cypher25Parser.NODETACH -> "'NODETACH'",
    Cypher25Parser.NODES -> "'NODES'",
    Cypher25Parser.NONE -> "'NONE'",
    Cypher25Parser.NORMALIZE -> "'NORMALIZE'",
    Cypher25Parser.NORMALIZED -> "'NORMALIZED'",
    Cypher25Parser.NOT -> "'NOT'",
    Cypher25Parser.NOTHING -> "'NOTHING'",
    Cypher25Parser.NOWAIT -> "'NOWAIT'",
    Cypher25Parser.NULL -> "'NULL'",
    Cypher25Parser.OF -> "'OF'",
    Cypher25Parser.OFFSET -> "'OFFSET'",
    Cypher25Parser.ON -> "'ON'",
    Cypher25Parser.ONLY -> "'ONLY'",
    Cypher25Parser.OPTIONAL -> "'OPTIONAL'",
    Cypher25Parser.OPTIONS -> "'OPTIONS'",
    Cypher25Parser.OPTION -> "'OPTION'",
    Cypher25Parser.OR -> "'OR'",
    Cypher25Parser.ORDER -> "'ORDER'",
    Cypher25Parser.PASSWORD -> "'PASSWORD'",
    Cypher25Parser.PASSWORDS -> "'PASSWORDS'",
    Cypher25Parser.PATH -> "'PATH'",
    Cypher25Parser.PATHS -> "'PATHS'",
    Cypher25Parser.PLAINTEXT -> "'PLAINTEXT'",
    Cypher25Parser.PLUS -> "'+'",
    Cypher25Parser.PLUSEQUAL -> "'+='",
    Cypher25Parser.POINT -> "'POINT'",
    Cypher25Parser.POPULATED -> "'POPULATED'",
    Cypher25Parser.POW -> "'^'",
    Cypher25Parser.PRIMARY -> "'PRIMARY'",
    Cypher25Parser.PRIMARIES -> "'PRIMARIES'",
    Cypher25Parser.PRIVILEGE -> "'PRIVILEGE'",
    Cypher25Parser.PRIVILEGES -> "'PRIVILEGES'",
    Cypher25Parser.PROCEDURE -> "'PROCEDURE'",
    Cypher25Parser.PROCEDURES -> "'PROCEDURES'",
    Cypher25Parser.PROPERTIES -> "'PROPERTIES'",
    Cypher25Parser.PROPERTY -> "'PROPERTY'",
    Cypher25Parser.PROVIDER -> "'PROVIDER'",
    Cypher25Parser.PROVIDERS -> "'PROVIDERS'",
    Cypher25Parser.QUESTION -> "'?'",
    Cypher25Parser.RANGE -> "'RANGE'",
    Cypher25Parser.RBRACKET -> "']'",
    Cypher25Parser.RCURLY -> "'}'",
    Cypher25Parser.READ -> "'READ'",
    Cypher25Parser.REALLOCATE -> "'REALLOCATE'",
    Cypher25Parser.REDUCE -> "'REDUCE'",
    Cypher25Parser.RENAME -> "'RENAME'",
    Cypher25Parser.REGEQ -> "'=~'",
    Cypher25Parser.REL -> "'REL'",
    Cypher25Parser.RELATIONSHIP -> "'RELATIONSHIP'",
    Cypher25Parser.RELATIONSHIPS -> "'RELATIONSHIPS'",
    Cypher25Parser.REMOVE -> "'REMOVE'",
    Cypher25Parser.REPEATABLE -> "'REPEATABLE'",
    Cypher25Parser.REPLACE -> "'REPLACE'",
    Cypher25Parser.REPORT -> "'REPORT'",
    Cypher25Parser.REQUIRE -> "'REQUIRE'",
    Cypher25Parser.REQUIRED -> "'REQUIRED'",
    Cypher25Parser.RESTRICT -> "'RESTRICT'",
    Cypher25Parser.RETURN -> "'RETURN'",
    Cypher25Parser.REVOKE -> "'REVOKE'",
    Cypher25Parser.ROLE -> "'ROLE'",
    Cypher25Parser.ROLES -> "'ROLES'",
    Cypher25Parser.ROW -> "'ROW'",
    Cypher25Parser.ROWS -> "'ROWS'",
    Cypher25Parser.RPAREN -> "')'",
    Cypher25Parser.SCAN -> "'SCAN'",
    Cypher25Parser.SECONDARY -> "'SECONDARY'",
    Cypher25Parser.SECONDARIES -> "'SECONDARIES'",
    Cypher25Parser.SEC -> "'SEC'",
    Cypher25Parser.SECOND -> "'SECOND'",
    Cypher25Parser.SECONDS -> "'SECONDS'",
    Cypher25Parser.SEEK -> "'SEEK'",
    Cypher25Parser.SEMICOLON -> "';'",
    Cypher25Parser.SERVER -> "'SERVER'",
    Cypher25Parser.SERVERS -> "'SERVERS'",
    Cypher25Parser.SET -> "'SET'",
    Cypher25Parser.SETTING -> "'SETTING'",
    Cypher25Parser.SETTINGS -> "'SETTINGS'",
    Cypher25Parser.SHORTEST_PATH -> "'shortestPath'",
    Cypher25Parser.SHORTEST -> "'SHORTEST'",
    Cypher25Parser.SHOW -> "'SHOW'",
    Cypher25Parser.SIGNED -> "'SIGNED'",
    Cypher25Parser.SINGLE -> "'SINGLE'",
    Cypher25Parser.SKIPROWS -> "'SKIP'",
    Cypher25Parser.START -> "'START'",
    Cypher25Parser.STARTS -> "'STARTS'",
    Cypher25Parser.STATUS -> "'STATUS'",
    Cypher25Parser.STOP -> "'STOP'",
    Cypher25Parser.STRING -> "'STRING'",
    Cypher25Parser.SUPPORTED -> "'SUPPORTED'",
    Cypher25Parser.SUSPENDED -> "'SUSPENDED'",
    Cypher25Parser.TARGET -> "'TARGET'",
    Cypher25Parser.TERMINATE -> "'TERMINATE'",
    Cypher25Parser.TEXT -> "'TEXT'",
    Cypher25Parser.THEN -> "'THEN'",
    Cypher25Parser.TIME -> "'TIME'",
    Cypher25Parser.TIMES -> "'*'",
    Cypher25Parser.TIMESTAMP -> "'TIMESTAMP'",
    Cypher25Parser.TIMEZONE -> "'TIMEZONE'",
    Cypher25Parser.TO -> "'TO'",
    Cypher25Parser.TOPOLOGY -> "'TOPOLOGY'",
    Cypher25Parser.TRAILING -> "'TRAILING'",
    Cypher25Parser.TRANSACTION -> "'TRANSACTION'",
    Cypher25Parser.TRANSACTIONS -> "'TRANSACTIONS'",
    Cypher25Parser.TRAVERSE -> "'TRAVERSE'",
    Cypher25Parser.TRIM -> "'TRIM'",
    Cypher25Parser.TRUE -> "'TRUE'",
    Cypher25Parser.TYPE -> "'TYPE'",
    Cypher25Parser.TYPED -> "'TYPED'",
    Cypher25Parser.TYPES -> "'TYPES'",
    Cypher25Parser.UNION -> "'UNION'",
    Cypher25Parser.UNIQUE -> "'UNIQUE'",
    Cypher25Parser.UNIQUENESS -> "'UNIQUENESS'",
    Cypher25Parser.UNWIND -> "'UNWIND'",
    Cypher25Parser.URL -> "'URL'",
    Cypher25Parser.USE -> "'USE'",
    Cypher25Parser.USER -> "'USER'",
    Cypher25Parser.USERS -> "'USERS'",
    Cypher25Parser.USING -> "'USING'",
    Cypher25Parser.VALUE -> "'VALUE'",
    Cypher25Parser.VARCHAR -> "'VARCHAR'",
    Cypher25Parser.VECTOR -> "'VECTOR'",
    Cypher25Parser.VERTEX -> "'VERTEX'",
    Cypher25Parser.WAIT -> "'WAIT'",
    Cypher25Parser.WHEN -> "'WHEN'",
    Cypher25Parser.WHERE -> "'WHERE'",
    Cypher25Parser.WITH -> "'WITH'",
    Cypher25Parser.WITHOUT -> "'WITHOUT'",
    Cypher25Parser.WRITE -> "'WRITE'",
    Cypher25Parser.XOR -> "'XOR'",
    Cypher25Parser.YIELD -> "'YIELD'",
    Cypher25Parser.ZONE -> "'ZONE'",
    Cypher25Parser.ZONED -> "'ZONED'",
    Cypher25Parser.EXTENDED_IDENTIFIER -> "an identifier",
    Cypher25Parser.IDENTIFIER -> "an identifier",
    Cypher25Parser.ARROW_LINE -> "'-'",
    Cypher25Parser.ARROW_LEFT_HEAD -> "'<'",
    Cypher25Parser.ARROW_RIGHT_HEAD -> "'>'",
    Cypher25Parser.STRING_LITERAL1 -> "a string value",
    Cypher25Parser.STRING_LITERAL2 -> "a string value",
    Cypher25Parser.MULTI_LINE_COMMENT -> "'/*'",
    Cypher25Parser.ESCAPED_SYMBOLIC_NAME -> "an identifier"
  )

  private val expectedRuleDisplayNames: Map[Int, String] = Map(
    Cypher25Parser.RULE_expression -> "an expression",
    Cypher25Parser.RULE_expression1 -> "an expression",
    Cypher25Parser.RULE_expression2 -> "an expression",
    Cypher25Parser.RULE_expression3 -> "an expression",
    Cypher25Parser.RULE_expression4 -> "an expression",
    Cypher25Parser.RULE_expression5 -> "an expression",
    Cypher25Parser.RULE_expression6 -> "an expression",
    Cypher25Parser.RULE_expression7 -> "an expression",
    Cypher25Parser.RULE_expression8 -> "an expression",
    Cypher25Parser.RULE_expression9 -> "an expression",
    Cypher25Parser.RULE_expression10 -> "an expression",
    Cypher25Parser.RULE_expression11 -> "an expression",
    Cypher25Parser.RULE_pattern -> "a graph pattern",
    Cypher25Parser.RULE_numberLiteral -> "a number",
    Cypher25Parser.RULE_parameter -> "a parameter",
    Cypher25Parser.RULE_variable -> "a variable name",
    Cypher25Parser.RULE_symbolicAliasName -> "a database name",
    Cypher25Parser.RULE_stringLiteral -> "a string",
    Cypher25Parser.RULE_symbolicNameString -> "an identifier",
    Cypher25Parser.RULE_symbolicLabelNameString -> "an identifier",
    Cypher25Parser.RULE_escapedSymbolicNameString -> "an identifier",
    Cypher25Parser.RULE_unescapedSymbolicNameString -> "an identifier",
    Cypher25Parser.RULE_unescapedLabelSymbolicNameString -> "an identifier"
  )
}
