/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.neo4j.internal.id;

/**
 * Utility containing methods to validate ids generated by {@link IdGenerator}.
 * Takes into account special reserved value {@link #INTEGER_MINUS_ONE}.
 */
public final class IdValidator {
    /**
     * Invalid and reserved id value. Represents special values, f.ex. the end of a relationships/property chain.
     * Please use {@link IdValidator} to validate generated ids.
     */
    public static final long INTEGER_MINUS_ONE = 0xFFFFFFFFL; // 4294967295L;

    private IdValidator() {}

    /**
     * Checks if the given id is reserved, i.e. {@link #INTEGER_MINUS_ONE}.
     *
     * @param id the id to check.
     * @return <code>true</code> if the given id is {@link #INTEGER_MINUS_ONE}, <code>false</code>
     * otherwise.
     * @see #INTEGER_MINUS_ONE
     */
    public static boolean isReservedId(long id) {
        return id == INTEGER_MINUS_ONE;
    }

    /**
     * Asserts that the given id is valid:
     * <ul>
     * <li>non-negative
     * <li>less than the given max id
     * <li>not equal to {@link #INTEGER_MINUS_ONE}
     * </ul>
     *
     * @param id the id to check.
     * @param maxId the max allowed id.
     * @see #INTEGER_MINUS_ONE
     */
    public static void assertValidId(IdType idType, long id, long maxId) {
        if (isReservedId(id)) {
            throw new ReservedIdException(id);
        }
        assertIdWithinCapacity(idType, id, maxId);
    }

    /**
     * Asserts that the given id is valid with respect to given max id:
     * <ul>
     * <li>non-negative
     * <li>less than the given max id
     * </ul>
     *
     * @param idType
     * @param id the id to check.
     * @param maxId the max allowed id.
     */
    public static void assertIdWithinCapacity(IdType idType, long id, long maxId) {
        if (id < 0) {
            throw new NegativeIdException(id);
        }
        assertIdWithinMaxCapacity(idType, id, maxId);
    }

    public static void assertIdWithinMaxCapacity(IdType idType, long id, long maxId) {
        if (id > maxId) {
            throw new IdCapacityExceededException(idType, id, maxId);
        }
    }

    public static boolean hasReservedIdInRange(long startIdInclusive, long endIdExclusive) {
        return startIdInclusive <= INTEGER_MINUS_ONE && endIdExclusive > INTEGER_MINUS_ONE;
    }
}
