/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.neo4j.kernel.impl.api;

import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.params.provider.Arguments.arguments;

import java.util.stream.Stream;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.neo4j.exceptions.KernelException;
import org.neo4j.graphdb.NotInTransactionException;
import org.neo4j.internal.kernel.api.Read;
import org.neo4j.internal.kernel.api.SchemaRead;
import org.neo4j.internal.kernel.api.Write;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.api.security.AnonymousContext;
import org.neo4j.values.storable.Values;

class KernelTransactionAssertOpenTest extends KernelTransactionTestBase {
    private static Stream<Arguments> parameters() {
        // Some samples of operations that should react to transaction killing. Not exhaustive.
        return Stream.of(
                arguments("nodeExists", (Operation) (read, write, schema) -> read.nodeExists(0)),
                arguments("singleRelationship", (Operation) (read, write, schema) -> read.singleRelationship(0, null)),
                arguments("nodeCreate", (Operation) (read, write, schema) -> write.nodeCreate()),
                arguments("relationshipSetProperty", (Operation)
                        (read, write, schema) -> write.relationshipSetProperty(0, 2, Values.longValue(42))),
                arguments("indexesGetAll", (Operation) (read, write, schema) -> schema.indexesGetAll()));
    }

    @ParameterizedTest
    @MethodSource("parameters")
    void shouldThrowNotInTransactionWhenTransactionClosedAndAccessingOperations(
            String name, Operation transactionOperation) throws KernelException {
        KernelTransaction transaction = newTransaction(AnonymousContext.write());

        transaction.commit();

        assertThrows(
                NotInTransactionException.class,
                () -> transactionOperation.operate(
                        transaction.dataRead(), transaction.dataWrite(), transaction.schemaRead()));
    }

    @ParameterizedTest
    @MethodSource("parameters")
    void shouldThrowNotInTransactionWhenTransactionClosedAndAttemptingOperations(
            String name, Operation transactionOperation) throws KernelException {
        KernelTransaction transaction = newTransaction(AnonymousContext.write());

        Read read = transaction.dataRead();
        Write write = transaction.dataWrite();
        SchemaRead schemaRead = transaction.schemaRead();

        transaction.commit();

        assertThrows(NotInTransactionException.class, () -> transactionOperation.operate(read, write, schemaRead));
    }

    interface Operation {
        void operate(Read read, Write write, SchemaRead schemaRead) throws KernelException;
    }
}
