/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.neo4j.kernel.impl.index.schema;

import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.Map;
import java.util.function.IntSupplier;
import org.assertj.core.api.SoftAssertions;
import org.assertj.core.api.junit.jupiter.InjectSoftAssertions;
import org.assertj.core.api.junit.jupiter.SoftAssertionsExtension;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.internal.kernel.api.PropertyIndexQuery;
import org.neo4j.internal.kernel.api.TokenPredicate;
import org.neo4j.internal.schema.IndexCapability;
import org.neo4j.internal.schema.IndexQuery;
import org.neo4j.values.storable.CoordinateReferenceSystem;
import org.neo4j.values.storable.PointValue;
import org.neo4j.values.storable.Values;

@ExtendWith(SoftAssertionsExtension.class)
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
abstract class SupportPartitionedScanTestSuite {
    protected static final Map<Query, Boolean> NO_SUPPORT = createNoSupport();

    @InjectSoftAssertions
    SoftAssertions softly;

    private final IndexCapability capability;
    private final Map<Query, Boolean> supported;

    SupportPartitionedScanTestSuite(IndexCapability capability, Map<Query, Boolean> supported) {
        this.capability = capability;
        this.supported = supported;
    }

    @Test
    final void throwOnEmptyQueries() {
        softly.assertThatThrownBy(capability::supportPartitionedScan, "require non empty queries")
                .isInstanceOf(IllegalArgumentException.class)
                .hasMessageContaining("Expected non empty array");
    }

    @Test
    final void throwOnNullQueries() {
        softly.assertThatThrownBy(() -> capability.supportPartitionedScan(null, null), "require all non-null queries")
                .isInstanceOf(IllegalArgumentException.class)
                .hasMessageContaining("Expected array without null elements");
    }

    @Test
    final void testSupported() {
        supported.forEach((query, expected) -> softly.assertThat(
                        capability.supportPartitionedScan(query.queries(idGenerator())))
                .as("PartitionedScan with query '%s', is %s expected to be supported", query, expected ? "" : "not")
                .isEqualTo(expected));
    }

    protected static IntSupplier idGenerator() {
        return new IntSupplier() {
            private int i = 1 << 7;

            @Override
            public int getAsInt() {
                return ++i;
            }
        };
    }

    protected static Map<Query, Boolean> supports(Query... queries) {
        final var supported = new EnumMap<>(NO_SUPPORT);
        for (final var query : queries) {
            supported.put(query, true);
        }
        return supported;
    }

    private static Map<Query, Boolean> createNoSupport() {
        final var noSupport = new EnumMap<Query, Boolean>(Query.class);
        for (final var query : Query.values()) {
            noSupport.put(query, false);
        }
        return Collections.unmodifiableMap(noSupport);
    }

    protected enum Query {
        // singular queries

        TOKEN_LOOKUP(TokenPredicate::new),

        ALL_ENTRIES(id -> PropertyIndexQuery.allEntries()),

        EXISTS(PropertyIndexQuery::exists),

        EXACT_NUMBER(id -> PropertyIndexQuery.exact(id, 999)),
        EXACT_NUMBER_ARRAY(id -> PropertyIndexQuery.exact(id, new int[] {-999, 999})),
        EXACT_TEXT(id -> PropertyIndexQuery.exact(id, "foo")),
        EXACT_TEXT_ARRAY(id -> PropertyIndexQuery.exact(id, new String[] {"foo", "bar"})),
        EXACT_GEOMETRY(id -> PropertyIndexQuery.exact(
                id, Values.pointValue(CoordinateReferenceSystem.WGS_84, 12.994807, 55.612088))),
        EXACT_GEOMETRY_ARRAY(id -> PropertyIndexQuery.exact(id, new PointValue[] {
            Values.pointValue(CoordinateReferenceSystem.WGS_84, 12.994807, 55.612088),
            Values.pointValue(CoordinateReferenceSystem.WGS_84, -0.101008, 51.503773)
        })),
        EXACT_TEMPORAL(id -> PropertyIndexQuery.exact(id, ZonedDateTime.now())),
        EXACT_TEMPORAL_ARRAY(id -> PropertyIndexQuery.exact(
                id,
                new ZonedDateTime[] {ZonedDateTime.now(), ZonedDateTime.now().plusWeeks(2)})),
        EXACT_BOOLEAN(id -> PropertyIndexQuery.exact(id, true)),
        EXACT_BOOLEAN_ARRAY(id -> PropertyIndexQuery.exact(id, new boolean[] {false, true})),

        RANGE_NUMBER(id -> PropertyIndexQuery.range(id, Integer.MIN_VALUE / 2, true, Integer.MAX_VALUE / 2, false)),
        RANGE_NUMBER_ARRAY(id -> PropertyIndexQuery.range(
                id, Values.of(new int[] {-999, -999}), true, Values.of(new int[] {999, 999}), false)),
        RANGE_TEXT(id -> PropertyIndexQuery.range(id, Values.of("abc"), true, Values.of("xyz"), false)),
        RANGE_TEXT_ARRAY(id -> PropertyIndexQuery.range(
                id, Values.of(new String[] {"abc", "def"}), true, Values.of(new String[] {"xyz", "uvw"}), false)),
        RANGE_GEOMETRY(id -> PropertyIndexQuery.range(
                id,
                Values.pointValue(CoordinateReferenceSystem.WGS_84, 0, 0),
                true,
                Values.pointValue(CoordinateReferenceSystem.WGS_84, 0, 90),
                true)),
        RANGE_GEOMETRY_ARRAY(id -> PropertyIndexQuery.range(
                id,
                Values.of(new PointValue[] {
                    Values.pointValue(CoordinateReferenceSystem.WGS_84, 0, 0),
                    Values.pointValue(CoordinateReferenceSystem.WGS_84, 180, 0)
                }),
                true,
                Values.of(new PointValue[] {
                    Values.pointValue(CoordinateReferenceSystem.WGS_84, 0, 90),
                    Values.pointValue(CoordinateReferenceSystem.WGS_84, 180, -90)
                }),
                true)),
        RANGE_TEMPORAL(id -> PropertyIndexQuery.range(
                id,
                Values.of(ZonedDateTime.ofInstant(Instant.EPOCH, ZoneOffset.UTC)),
                false,
                Values.of(ZonedDateTime.now()),
                true)),
        RANGE_TEMPORAL_ARRAY(id -> PropertyIndexQuery.range(
                id,
                Values.of(new ZonedDateTime[] {
                    ZonedDateTime.ofInstant(Instant.EPOCH, ZoneOffset.UTC),
                    ZonedDateTime.now().minusWeeks(1)
                }),
                false,
                Values.of(new ZonedDateTime[] {
                    ZonedDateTime.now(), ZonedDateTime.now().plusWeeks(1)
                }),
                true)),
        RANGE_BOOLEAN(id -> PropertyIndexQuery.range(id, Values.of(false), true, Values.of(true), true)),
        RANGE_BOOLEAN_ARRAY(id -> PropertyIndexQuery.range(
                id, Values.of(new boolean[] {false, true}), true, Values.of(new boolean[] {true, true}), true)),

        BOUNDING_BOX(id -> PropertyIndexQuery.boundingBox(
                id,
                Values.pointValue(CoordinateReferenceSystem.WGS_84, 0, 0),
                Values.pointValue(CoordinateReferenceSystem.WGS_84, 0, 90))),

        STRING_PREFIX(id -> PropertyIndexQuery.stringPrefix(id, Values.utf8Value("prefix"))),
        STRING_SUFFIX(id -> PropertyIndexQuery.stringSuffix(id, Values.utf8Value("suffix"))),
        STRING_CONTAINS(id -> PropertyIndexQuery.stringContains(id, Values.utf8Value("contains"))),

        FULLTEXT_SEARCH(id -> PropertyIndexQuery.fulltextSearch("contains any of these words")),

        // composite queries

        COMPOSITE_TOKEN_LOOKUP_TOKEN_LOOKUP(TOKEN_LOOKUP, TOKEN_LOOKUP),
        COMPOSITE_TOKEN_LOOKUP_ALL_ENTRIES(TOKEN_LOOKUP, ALL_ENTRIES),
        COMPOSITE_TOKEN_LOOKUP_EXISTS(TOKEN_LOOKUP, EXISTS),
        COMPOSITE_TOKEN_LOOKUP_EXACT_NUMBER(TOKEN_LOOKUP, EXACT_NUMBER),
        COMPOSITE_TOKEN_LOOKUP_EXACT_NUMBER_ARRAY(TOKEN_LOOKUP, EXACT_NUMBER_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_EXACT_TEXT(TOKEN_LOOKUP, EXACT_TEXT),
        COMPOSITE_TOKEN_LOOKUP_EXACT_TEXT_ARRAY(TOKEN_LOOKUP, EXACT_TEXT_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_EXACT_GEOMETRY(TOKEN_LOOKUP, EXACT_GEOMETRY),
        COMPOSITE_TOKEN_LOOKUP_EXACT_GEOMETRY_ARRAY(TOKEN_LOOKUP, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_EXACT_TEMPORAL(TOKEN_LOOKUP, EXACT_TEMPORAL),
        COMPOSITE_TOKEN_LOOKUP_EXACT_TEMPORAL_ARRAY(TOKEN_LOOKUP, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_EXACT_BOOLEAN(TOKEN_LOOKUP, EXACT_BOOLEAN),
        COMPOSITE_TOKEN_LOOKUP_EXACT_BOOLEAN_ARRAY(TOKEN_LOOKUP, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_RANGE_NUMBER(TOKEN_LOOKUP, RANGE_NUMBER),
        COMPOSITE_TOKEN_LOOKUP_RANGE_NUMBER_ARRAY(TOKEN_LOOKUP, RANGE_NUMBER_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_RANGE_TEXT(TOKEN_LOOKUP, RANGE_TEXT),
        COMPOSITE_TOKEN_LOOKUP_RANGE_TEXT_ARRAY(TOKEN_LOOKUP, RANGE_TEXT_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_RANGE_GEOMETRY(TOKEN_LOOKUP, RANGE_GEOMETRY),
        COMPOSITE_TOKEN_LOOKUP_RANGE_GEOMETRY_ARRAY(TOKEN_LOOKUP, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_RANGE_TEMPORAL(TOKEN_LOOKUP, RANGE_TEMPORAL),
        COMPOSITE_TOKEN_LOOKUP_RANGE_TEMPORAL_ARRAY(TOKEN_LOOKUP, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_RANGE_BOOLEAN(TOKEN_LOOKUP, RANGE_BOOLEAN),
        COMPOSITE_TOKEN_LOOKUP_RANGE_BOOLEAN_ARRAY(TOKEN_LOOKUP, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_TOKEN_LOOKUP_BOUNDING_BOX(TOKEN_LOOKUP, BOUNDING_BOX),
        COMPOSITE_TOKEN_LOOKUP_STRING_PREFIX(TOKEN_LOOKUP, STRING_PREFIX),
        COMPOSITE_TOKEN_LOOKUP_STRING_SUFFIX(TOKEN_LOOKUP, STRING_SUFFIX),
        COMPOSITE_TOKEN_LOOKUP_STRING_CONTAINS(TOKEN_LOOKUP, STRING_CONTAINS),
        COMPOSITE_TOKEN_LOOKUP_FULLTEXT_SEARCH(TOKEN_LOOKUP, FULLTEXT_SEARCH),

        COMPOSITE_ALL_ENTRIES_TOKEN_LOOKUP(ALL_ENTRIES, TOKEN_LOOKUP),
        COMPOSITE_ALL_ENTRIES_ALL_ENTRIES(ALL_ENTRIES, ALL_ENTRIES),
        COMPOSITE_ALL_ENTRIES_EXISTS(ALL_ENTRIES, EXISTS),
        COMPOSITE_ALL_ENTRIES_EXACT_NUMBER(ALL_ENTRIES, EXACT_NUMBER),
        COMPOSITE_ALL_ENTRIES_EXACT_NUMBER_ARRAY(ALL_ENTRIES, EXACT_NUMBER_ARRAY),
        COMPOSITE_ALL_ENTRIES_EXACT_TEXT(ALL_ENTRIES, EXACT_TEXT),
        COMPOSITE_ALL_ENTRIES_EXACT_TEXT_ARRAY(ALL_ENTRIES, EXACT_TEXT_ARRAY),
        COMPOSITE_ALL_ENTRIES_EXACT_GEOMETRY(ALL_ENTRIES, EXACT_GEOMETRY),
        COMPOSITE_ALL_ENTRIES_EXACT_GEOMETRY_ARRAY(ALL_ENTRIES, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_ALL_ENTRIES_EXACT_TEMPORAL(ALL_ENTRIES, EXACT_TEMPORAL),
        COMPOSITE_ALL_ENTRIES_EXACT_TEMPORAL_ARRAY(ALL_ENTRIES, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_ALL_ENTRIES_EXACT_BOOLEAN(ALL_ENTRIES, EXACT_BOOLEAN),
        COMPOSITE_ALL_ENTRIES_EXACT_BOOLEAN_ARRAY(ALL_ENTRIES, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_ALL_ENTRIES_RANGE_NUMBER(ALL_ENTRIES, RANGE_NUMBER),
        COMPOSITE_ALL_ENTRIES_RANGE_NUMBER_ARRAY(ALL_ENTRIES, RANGE_NUMBER_ARRAY),
        COMPOSITE_ALL_ENTRIES_RANGE_TEXT(ALL_ENTRIES, RANGE_TEXT),
        COMPOSITE_ALL_ENTRIES_RANGE_TEXT_ARRAY(ALL_ENTRIES, RANGE_TEXT_ARRAY),
        COMPOSITE_ALL_ENTRIES_RANGE_GEOMETRY(ALL_ENTRIES, RANGE_GEOMETRY),
        COMPOSITE_ALL_ENTRIES_RANGE_GEOMETRY_ARRAY(ALL_ENTRIES, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_ALL_ENTRIES_RANGE_TEMPORAL(ALL_ENTRIES, RANGE_TEMPORAL),
        COMPOSITE_ALL_ENTRIES_RANGE_TEMPORAL_ARRAY(ALL_ENTRIES, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_ALL_ENTRIES_RANGE_BOOLEAN(ALL_ENTRIES, RANGE_BOOLEAN),
        COMPOSITE_ALL_ENTRIES_RANGE_BOOLEAN_ARRAY(ALL_ENTRIES, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_ALL_ENTRIES_BOUNDING_BOX(ALL_ENTRIES, BOUNDING_BOX),
        COMPOSITE_ALL_ENTRIES_STRING_PREFIX(ALL_ENTRIES, STRING_PREFIX),
        COMPOSITE_ALL_ENTRIES_STRING_SUFFIX(ALL_ENTRIES, STRING_SUFFIX),
        COMPOSITE_ALL_ENTRIES_STRING_CONTAINS(ALL_ENTRIES, STRING_CONTAINS),
        COMPOSITE_ALL_ENTRIES_FULLTEXT_SEARCH(ALL_ENTRIES, FULLTEXT_SEARCH),

        COMPOSITE_EXISTS_TOKEN_LOOKUP(EXISTS, TOKEN_LOOKUP),
        COMPOSITE_EXISTS_ALL_ENTRIES(EXISTS, ALL_ENTRIES),
        COMPOSITE_EXISTS_EXISTS(EXISTS, EXISTS),
        COMPOSITE_EXISTS_EXACT_NUMBER(EXISTS, EXACT_NUMBER),
        COMPOSITE_EXISTS_EXACT_NUMBER_ARRAY(EXISTS, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXISTS_EXACT_TEXT(EXISTS, EXACT_TEXT),
        COMPOSITE_EXISTS_EXACT_TEXT_ARRAY(EXISTS, EXACT_TEXT_ARRAY),
        COMPOSITE_EXISTS_EXACT_GEOMETRY(EXISTS, EXACT_GEOMETRY),
        COMPOSITE_EXISTS_EXACT_GEOMETRY_ARRAY(EXISTS, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXISTS_EXACT_TEMPORAL(EXISTS, EXACT_TEMPORAL),
        COMPOSITE_EXISTS_EXACT_TEMPORAL_ARRAY(EXISTS, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXISTS_EXACT_BOOLEAN(EXISTS, EXACT_BOOLEAN),
        COMPOSITE_EXISTS_EXACT_BOOLEAN_ARRAY(EXISTS, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXISTS_RANGE_NUMBER(EXISTS, RANGE_NUMBER),
        COMPOSITE_EXISTS_RANGE_NUMBER_ARRAY(EXISTS, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXISTS_RANGE_TEXT(EXISTS, RANGE_TEXT),
        COMPOSITE_EXISTS_RANGE_TEXT_ARRAY(EXISTS, RANGE_TEXT_ARRAY),
        COMPOSITE_EXISTS_RANGE_GEOMETRY(EXISTS, RANGE_GEOMETRY),
        COMPOSITE_EXISTS_RANGE_GEOMETRY_ARRAY(EXISTS, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXISTS_RANGE_TEMPORAL(EXISTS, RANGE_TEMPORAL),
        COMPOSITE_EXISTS_RANGE_TEMPORAL_ARRAY(EXISTS, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXISTS_RANGE_BOOLEAN(EXISTS, RANGE_BOOLEAN),
        COMPOSITE_EXISTS_RANGE_BOOLEAN_ARRAY(EXISTS, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXISTS_BOUNDING_BOX(EXISTS, BOUNDING_BOX),
        COMPOSITE_EXISTS_STRING_PREFIX(EXISTS, STRING_PREFIX),
        COMPOSITE_EXISTS_STRING_SUFFIX(EXISTS, STRING_SUFFIX),
        COMPOSITE_EXISTS_STRING_CONTAINS(EXISTS, STRING_CONTAINS),
        COMPOSITE_EXISTS_FULLTEXT_SEARCH(EXISTS, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_NUMBER_TOKEN_LOOKUP(EXACT_NUMBER, TOKEN_LOOKUP),
        COMPOSITE_EXACT_NUMBER_ALL_ENTRIES(EXACT_NUMBER, ALL_ENTRIES),
        COMPOSITE_EXACT_NUMBER_EXISTS(EXACT_NUMBER, EXISTS),
        COMPOSITE_EXACT_NUMBER_EXACT_NUMBER(EXACT_NUMBER, EXACT_NUMBER),
        COMPOSITE_EXACT_NUMBER_EXACT_NUMBER_ARRAY(EXACT_NUMBER, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_NUMBER_EXACT_TEXT(EXACT_NUMBER, EXACT_TEXT),
        COMPOSITE_EXACT_NUMBER_EXACT_TEXT_ARRAY(EXACT_NUMBER, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_NUMBER_EXACT_GEOMETRY(EXACT_NUMBER, EXACT_GEOMETRY),
        COMPOSITE_EXACT_NUMBER_EXACT_GEOMETRY_ARRAY(EXACT_NUMBER, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_NUMBER_EXACT_TEMPORAL(EXACT_NUMBER, EXACT_TEMPORAL),
        COMPOSITE_EXACT_NUMBER_EXACT_TEMPORAL_ARRAY(EXACT_NUMBER, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_NUMBER_EXACT_BOOLEAN(EXACT_NUMBER, EXACT_BOOLEAN),
        COMPOSITE_EXACT_NUMBER_EXACT_BOOLEAN_ARRAY(EXACT_NUMBER, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_NUMBER_RANGE_NUMBER(EXACT_NUMBER, RANGE_NUMBER),
        COMPOSITE_EXACT_NUMBER_RANGE_NUMBER_ARRAY(EXACT_NUMBER, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_NUMBER_RANGE_TEXT(EXACT_NUMBER, RANGE_TEXT),
        COMPOSITE_EXACT_NUMBER_RANGE_TEXT_ARRAY(EXACT_NUMBER, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_NUMBER_RANGE_GEOMETRY(EXACT_NUMBER, RANGE_GEOMETRY),
        COMPOSITE_EXACT_NUMBER_RANGE_GEOMETRY_ARRAY(EXACT_NUMBER, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_NUMBER_RANGE_TEMPORAL(EXACT_NUMBER, RANGE_TEMPORAL),
        COMPOSITE_EXACT_NUMBER_RANGE_TEMPORAL_ARRAY(EXACT_NUMBER, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_NUMBER_RANGE_BOOLEAN(EXACT_NUMBER, RANGE_BOOLEAN),
        COMPOSITE_EXACT_NUMBER_RANGE_BOOLEAN_ARRAY(EXACT_NUMBER, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_NUMBER_BOUNDING_BOX(EXACT_NUMBER, BOUNDING_BOX),
        COMPOSITE_EXACT_NUMBER_STRING_PREFIX(EXACT_NUMBER, STRING_PREFIX),
        COMPOSITE_EXACT_NUMBER_STRING_SUFFIX(EXACT_NUMBER, STRING_SUFFIX),
        COMPOSITE_EXACT_NUMBER_STRING_CONTAINS(EXACT_NUMBER, STRING_CONTAINS),
        COMPOSITE_EXACT_NUMBER_FULLTEXT_SEARCH(EXACT_NUMBER, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_NUMBER_ARRAY_TOKEN_LOOKUP(EXACT_NUMBER_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_EXACT_NUMBER_ARRAY_ALL_ENTRIES(EXACT_NUMBER_ARRAY, ALL_ENTRIES),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXISTS(EXACT_NUMBER_ARRAY, EXISTS),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_NUMBER(EXACT_NUMBER_ARRAY, EXACT_NUMBER),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_NUMBER_ARRAY(EXACT_NUMBER_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_TEXT(EXACT_NUMBER_ARRAY, EXACT_TEXT),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_TEXT_ARRAY(EXACT_NUMBER_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_GEOMETRY(EXACT_NUMBER_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_GEOMETRY_ARRAY(EXACT_NUMBER_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_TEMPORAL(EXACT_NUMBER_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_TEMPORAL_ARRAY(EXACT_NUMBER_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_BOOLEAN(EXACT_NUMBER_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_EXACT_NUMBER_ARRAY_EXACT_BOOLEAN_ARRAY(EXACT_NUMBER_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_NUMBER(EXACT_NUMBER_ARRAY, RANGE_NUMBER),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_NUMBER_ARRAY(EXACT_NUMBER_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_TEXT(EXACT_NUMBER_ARRAY, RANGE_TEXT),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_TEXT_ARRAY(EXACT_NUMBER_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_GEOMETRY(EXACT_NUMBER_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_GEOMETRY_ARRAY(EXACT_NUMBER_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_TEMPORAL(EXACT_NUMBER_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_TEMPORAL_ARRAY(EXACT_NUMBER_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_BOOLEAN(EXACT_NUMBER_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_EXACT_NUMBER_ARRAY_RANGE_BOOLEAN_ARRAY(EXACT_NUMBER_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_NUMBER_ARRAY_BOUNDING_BOX(EXACT_NUMBER_ARRAY, BOUNDING_BOX),
        COMPOSITE_EXACT_NUMBER_ARRAY_STRING_PREFIX(EXACT_NUMBER_ARRAY, STRING_PREFIX),
        COMPOSITE_EXACT_NUMBER_ARRAY_STRING_SUFFIX(EXACT_NUMBER_ARRAY, STRING_SUFFIX),
        COMPOSITE_EXACT_NUMBER_ARRAY_STRING_CONTAINS(EXACT_NUMBER_ARRAY, STRING_CONTAINS),
        COMPOSITE_EXACT_NUMBER_ARRAY_FULLTEXT_SEARCH(EXACT_NUMBER_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_TEXT_TOKEN_LOOKUP(EXACT_TEXT, TOKEN_LOOKUP),
        COMPOSITE_EXACT_TEXT_ALL_ENTRIES(EXACT_TEXT, ALL_ENTRIES),
        COMPOSITE_EXACT_TEXT_EXISTS(EXACT_TEXT, EXISTS),
        COMPOSITE_EXACT_TEXT_EXACT_NUMBER(EXACT_TEXT, EXACT_NUMBER),
        COMPOSITE_EXACT_TEXT_EXACT_NUMBER_ARRAY(EXACT_TEXT, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_TEXT_EXACT_TEXT(EXACT_TEXT, EXACT_TEXT),
        COMPOSITE_EXACT_TEXT_EXACT_TEXT_ARRAY(EXACT_TEXT, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_TEXT_EXACT_GEOMETRY(EXACT_TEXT, EXACT_GEOMETRY),
        COMPOSITE_EXACT_TEXT_EXACT_GEOMETRY_ARRAY(EXACT_TEXT, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_TEXT_EXACT_TEMPORAL(EXACT_TEXT, EXACT_TEMPORAL),
        COMPOSITE_EXACT_TEXT_EXACT_TEMPORAL_ARRAY(EXACT_TEXT, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_TEXT_EXACT_BOOLEAN(EXACT_TEXT, EXACT_BOOLEAN),
        COMPOSITE_EXACT_TEXT_EXACT_BOOLEAN_ARRAY(EXACT_TEXT, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_TEXT_RANGE_NUMBER(EXACT_TEXT, RANGE_NUMBER),
        COMPOSITE_EXACT_TEXT_RANGE_NUMBER_ARRAY(EXACT_TEXT, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_TEXT_RANGE_TEXT(EXACT_TEXT, RANGE_TEXT),
        COMPOSITE_EXACT_TEXT_RANGE_TEXT_ARRAY(EXACT_TEXT, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_TEXT_RANGE_GEOMETRY(EXACT_TEXT, RANGE_GEOMETRY),
        COMPOSITE_EXACT_TEXT_RANGE_GEOMETRY_ARRAY(EXACT_TEXT, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_TEXT_RANGE_TEMPORAL(EXACT_TEXT, RANGE_TEMPORAL),
        COMPOSITE_EXACT_TEXT_RANGE_TEMPORAL_ARRAY(EXACT_TEXT, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_TEXT_RANGE_BOOLEAN(EXACT_TEXT, RANGE_BOOLEAN),
        COMPOSITE_EXACT_TEXT_RANGE_BOOLEAN_ARRAY(EXACT_TEXT, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_TEXT_BOUNDING_BOX(EXACT_TEXT, BOUNDING_BOX),
        COMPOSITE_EXACT_TEXT_STRING_PREFIX(EXACT_TEXT, STRING_PREFIX),
        COMPOSITE_EXACT_TEXT_STRING_SUFFIX(EXACT_TEXT, STRING_SUFFIX),
        COMPOSITE_EXACT_TEXT_STRING_CONTAINS(EXACT_TEXT, STRING_CONTAINS),
        COMPOSITE_EXACT_TEXT_FULLTEXT_SEARCH(EXACT_TEXT, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_TEXT_ARRAY_TOKEN_LOOKUP(EXACT_TEXT_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_EXACT_TEXT_ARRAY_ALL_ENTRIES(EXACT_TEXT_ARRAY, ALL_ENTRIES),
        COMPOSITE_EXACT_TEXT_ARRAY_EXISTS(EXACT_TEXT_ARRAY, EXISTS),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_NUMBER(EXACT_TEXT_ARRAY, EXACT_NUMBER),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_NUMBER_ARRAY(EXACT_TEXT_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_TEXT(EXACT_TEXT_ARRAY, EXACT_TEXT),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_TEXT_ARRAY(EXACT_TEXT_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_GEOMETRY(EXACT_TEXT_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_GEOMETRY_ARRAY(EXACT_TEXT_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_TEMPORAL(EXACT_TEXT_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_TEMPORAL_ARRAY(EXACT_TEXT_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_BOOLEAN(EXACT_TEXT_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_EXACT_TEXT_ARRAY_EXACT_BOOLEAN_ARRAY(EXACT_TEXT_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_NUMBER(EXACT_TEXT_ARRAY, RANGE_NUMBER),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_NUMBER_ARRAY(EXACT_TEXT_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_TEXT(EXACT_TEXT_ARRAY, RANGE_TEXT),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_TEXT_ARRAY(EXACT_TEXT_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_GEOMETRY(EXACT_TEXT_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_GEOMETRY_ARRAY(EXACT_TEXT_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_TEMPORAL(EXACT_TEXT_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_TEMPORAL_ARRAY(EXACT_TEXT_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_BOOLEAN(EXACT_TEXT_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_EXACT_TEXT_ARRAY_RANGE_BOOLEAN_ARRAY(EXACT_TEXT_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_TEXT_ARRAY_BOUNDING_BOX(EXACT_TEXT_ARRAY, BOUNDING_BOX),
        COMPOSITE_EXACT_TEXT_ARRAY_STRING_PREFIX(EXACT_TEXT_ARRAY, STRING_PREFIX),
        COMPOSITE_EXACT_TEXT_ARRAY_STRING_SUFFIX(EXACT_TEXT_ARRAY, STRING_SUFFIX),
        COMPOSITE_EXACT_TEXT_ARRAY_STRING_CONTAINS(EXACT_TEXT_ARRAY, STRING_CONTAINS),
        COMPOSITE_EXACT_TEXT_ARRAY_FULLTEXT_SEARCH(EXACT_TEXT_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_GEOMETRY_TOKEN_LOOKUP(EXACT_GEOMETRY, TOKEN_LOOKUP),
        COMPOSITE_EXACT_GEOMETRY_ALL_ENTRIES(EXACT_GEOMETRY, ALL_ENTRIES),
        COMPOSITE_EXACT_GEOMETRY_EXISTS(EXACT_GEOMETRY, EXISTS),
        COMPOSITE_EXACT_GEOMETRY_EXACT_NUMBER(EXACT_GEOMETRY, EXACT_NUMBER),
        COMPOSITE_EXACT_GEOMETRY_EXACT_NUMBER_ARRAY(EXACT_GEOMETRY, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_EXACT_TEXT(EXACT_GEOMETRY, EXACT_TEXT),
        COMPOSITE_EXACT_GEOMETRY_EXACT_TEXT_ARRAY(EXACT_GEOMETRY, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_EXACT_GEOMETRY(EXACT_GEOMETRY, EXACT_GEOMETRY),
        COMPOSITE_EXACT_GEOMETRY_EXACT_GEOMETRY_ARRAY(EXACT_GEOMETRY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_EXACT_TEMPORAL(EXACT_GEOMETRY, EXACT_TEMPORAL),
        COMPOSITE_EXACT_GEOMETRY_EXACT_TEMPORAL_ARRAY(EXACT_GEOMETRY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_EXACT_BOOLEAN(EXACT_GEOMETRY, EXACT_BOOLEAN),
        COMPOSITE_EXACT_GEOMETRY_EXACT_BOOLEAN_ARRAY(EXACT_GEOMETRY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_RANGE_NUMBER(EXACT_GEOMETRY, RANGE_NUMBER),
        COMPOSITE_EXACT_GEOMETRY_RANGE_NUMBER_ARRAY(EXACT_GEOMETRY, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_RANGE_TEXT(EXACT_GEOMETRY, RANGE_TEXT),
        COMPOSITE_EXACT_GEOMETRY_RANGE_TEXT_ARRAY(EXACT_GEOMETRY, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_RANGE_GEOMETRY(EXACT_GEOMETRY, RANGE_GEOMETRY),
        COMPOSITE_EXACT_GEOMETRY_RANGE_GEOMETRY_ARRAY(EXACT_GEOMETRY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_RANGE_TEMPORAL(EXACT_GEOMETRY, RANGE_TEMPORAL),
        COMPOSITE_EXACT_GEOMETRY_RANGE_TEMPORAL_ARRAY(EXACT_GEOMETRY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_RANGE_BOOLEAN(EXACT_GEOMETRY, RANGE_BOOLEAN),
        COMPOSITE_EXACT_GEOMETRY_RANGE_BOOLEAN_ARRAY(EXACT_GEOMETRY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_BOUNDING_BOX(EXACT_GEOMETRY, BOUNDING_BOX),
        COMPOSITE_EXACT_GEOMETRY_STRING_PREFIX(EXACT_GEOMETRY, STRING_PREFIX),
        COMPOSITE_EXACT_GEOMETRY_STRING_SUFFIX(EXACT_GEOMETRY, STRING_SUFFIX),
        COMPOSITE_EXACT_GEOMETRY_STRING_CONTAINS(EXACT_GEOMETRY, STRING_CONTAINS),
        COMPOSITE_EXACT_GEOMETRY_FULLTEXT_SEARCH(EXACT_GEOMETRY, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_GEOMETRY_ARRAY_TOKEN_LOOKUP(EXACT_GEOMETRY_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_ALL_ENTRIES(EXACT_GEOMETRY_ARRAY, ALL_ENTRIES),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXISTS(EXACT_GEOMETRY_ARRAY, EXISTS),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_NUMBER(EXACT_GEOMETRY_ARRAY, EXACT_NUMBER),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_NUMBER_ARRAY(EXACT_GEOMETRY_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_TEXT(EXACT_GEOMETRY_ARRAY, EXACT_TEXT),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_TEXT_ARRAY(EXACT_GEOMETRY_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_GEOMETRY(EXACT_GEOMETRY_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_GEOMETRY_ARRAY(EXACT_GEOMETRY_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_TEMPORAL(EXACT_GEOMETRY_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_TEMPORAL_ARRAY(EXACT_GEOMETRY_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_BOOLEAN(EXACT_GEOMETRY_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_EXACT_BOOLEAN_ARRAY(EXACT_GEOMETRY_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_NUMBER(EXACT_GEOMETRY_ARRAY, RANGE_NUMBER),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_NUMBER_ARRAY(EXACT_GEOMETRY_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_TEXT(EXACT_GEOMETRY_ARRAY, RANGE_TEXT),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_TEXT_ARRAY(EXACT_GEOMETRY_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_GEOMETRY(EXACT_GEOMETRY_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_GEOMETRY_ARRAY(EXACT_GEOMETRY_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_TEMPORAL(EXACT_GEOMETRY_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_TEMPORAL_ARRAY(EXACT_GEOMETRY_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_BOOLEAN(EXACT_GEOMETRY_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_RANGE_BOOLEAN_ARRAY(EXACT_GEOMETRY_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_BOUNDING_BOX(EXACT_GEOMETRY_ARRAY, BOUNDING_BOX),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_STRING_PREFIX(EXACT_GEOMETRY_ARRAY, STRING_PREFIX),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_STRING_SUFFIX(EXACT_GEOMETRY_ARRAY, STRING_SUFFIX),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_STRING_CONTAINS(EXACT_GEOMETRY_ARRAY, STRING_CONTAINS),
        COMPOSITE_EXACT_GEOMETRY_ARRAY_FULLTEXT_SEARCH(EXACT_GEOMETRY_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_TEMPORAL_TOKEN_LOOKUP(EXACT_TEMPORAL, TOKEN_LOOKUP),
        COMPOSITE_EXACT_TEMPORAL_ALL_ENTRIES(EXACT_TEMPORAL, ALL_ENTRIES),
        COMPOSITE_EXACT_TEMPORAL_EXISTS(EXACT_TEMPORAL, EXISTS),
        COMPOSITE_EXACT_TEMPORAL_EXACT_NUMBER(EXACT_TEMPORAL, EXACT_NUMBER),
        COMPOSITE_EXACT_TEMPORAL_EXACT_NUMBER_ARRAY(EXACT_TEMPORAL, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_EXACT_TEXT(EXACT_TEMPORAL, EXACT_TEXT),
        COMPOSITE_EXACT_TEMPORAL_EXACT_TEXT_ARRAY(EXACT_TEMPORAL, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_EXACT_GEOMETRY(EXACT_TEMPORAL, EXACT_GEOMETRY),
        COMPOSITE_EXACT_TEMPORAL_EXACT_GEOMETRY_ARRAY(EXACT_TEMPORAL, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_EXACT_TEMPORAL(EXACT_TEMPORAL, EXACT_TEMPORAL),
        COMPOSITE_EXACT_TEMPORAL_EXACT_TEMPORAL_ARRAY(EXACT_TEMPORAL, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_EXACT_BOOLEAN(EXACT_TEMPORAL, EXACT_BOOLEAN),
        COMPOSITE_EXACT_TEMPORAL_EXACT_BOOLEAN_ARRAY(EXACT_TEMPORAL, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_RANGE_NUMBER(EXACT_TEMPORAL, RANGE_NUMBER),
        COMPOSITE_EXACT_TEMPORAL_RANGE_NUMBER_ARRAY(EXACT_TEMPORAL, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_RANGE_TEXT(EXACT_TEMPORAL, RANGE_TEXT),
        COMPOSITE_EXACT_TEMPORAL_RANGE_TEXT_ARRAY(EXACT_TEMPORAL, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_RANGE_GEOMETRY(EXACT_TEMPORAL, RANGE_GEOMETRY),
        COMPOSITE_EXACT_TEMPORAL_RANGE_GEOMETRY_ARRAY(EXACT_TEMPORAL, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_RANGE_TEMPORAL(EXACT_TEMPORAL, RANGE_TEMPORAL),
        COMPOSITE_EXACT_TEMPORAL_RANGE_TEMPORAL_ARRAY(EXACT_TEMPORAL, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_RANGE_BOOLEAN(EXACT_TEMPORAL, RANGE_BOOLEAN),
        COMPOSITE_EXACT_TEMPORAL_RANGE_BOOLEAN_ARRAY(EXACT_TEMPORAL, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_BOUNDING_BOX(EXACT_TEMPORAL, BOUNDING_BOX),
        COMPOSITE_EXACT_TEMPORAL_STRING_PREFIX(EXACT_TEMPORAL, STRING_PREFIX),
        COMPOSITE_EXACT_TEMPORAL_STRING_SUFFIX(EXACT_TEMPORAL, STRING_SUFFIX),
        COMPOSITE_EXACT_TEMPORAL_STRING_CONTAINS(EXACT_TEMPORAL, STRING_CONTAINS),
        COMPOSITE_EXACT_TEMPORAL_FULLTEXT_SEARCH(EXACT_TEMPORAL, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_TEMPORAL_ARRAY_TOKEN_LOOKUP(EXACT_TEMPORAL_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_ALL_ENTRIES(EXACT_TEMPORAL_ARRAY, ALL_ENTRIES),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXISTS(EXACT_TEMPORAL_ARRAY, EXISTS),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_NUMBER(EXACT_TEMPORAL_ARRAY, EXACT_NUMBER),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_NUMBER_ARRAY(EXACT_TEMPORAL_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_TEXT(EXACT_TEMPORAL_ARRAY, EXACT_TEXT),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_TEXT_ARRAY(EXACT_TEMPORAL_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_GEOMETRY(EXACT_TEMPORAL_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_GEOMETRY_ARRAY(EXACT_TEMPORAL_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_TEMPORAL(EXACT_TEMPORAL_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_TEMPORAL_ARRAY(EXACT_TEMPORAL_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_BOOLEAN(EXACT_TEMPORAL_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_EXACT_BOOLEAN_ARRAY(EXACT_TEMPORAL_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_NUMBER(EXACT_TEMPORAL_ARRAY, RANGE_NUMBER),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_NUMBER_ARRAY(EXACT_TEMPORAL_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_TEXT(EXACT_TEMPORAL_ARRAY, RANGE_TEXT),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_TEXT_ARRAY(EXACT_TEMPORAL_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_GEOMETRY(EXACT_TEMPORAL_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_GEOMETRY_ARRAY(EXACT_TEMPORAL_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_TEMPORAL(EXACT_TEMPORAL_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_TEMPORAL_ARRAY(EXACT_TEMPORAL_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_BOOLEAN(EXACT_TEMPORAL_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_RANGE_BOOLEAN_ARRAY(EXACT_TEMPORAL_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_BOUNDING_BOX(EXACT_TEMPORAL_ARRAY, BOUNDING_BOX),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_STRING_PREFIX(EXACT_TEMPORAL_ARRAY, STRING_PREFIX),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_STRING_SUFFIX(EXACT_TEMPORAL_ARRAY, STRING_SUFFIX),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_STRING_CONTAINS(EXACT_TEMPORAL_ARRAY, STRING_CONTAINS),
        COMPOSITE_EXACT_TEMPORAL_ARRAY_FULLTEXT_SEARCH(EXACT_TEMPORAL_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_BOOLEAN_TOKEN_LOOKUP(EXACT_BOOLEAN, TOKEN_LOOKUP),
        COMPOSITE_EXACT_BOOLEAN_ALL_ENTRIES(EXACT_BOOLEAN, ALL_ENTRIES),
        COMPOSITE_EXACT_BOOLEAN_EXISTS(EXACT_BOOLEAN, EXISTS),
        COMPOSITE_EXACT_BOOLEAN_EXACT_NUMBER(EXACT_BOOLEAN, EXACT_NUMBER),
        COMPOSITE_EXACT_BOOLEAN_EXACT_NUMBER_ARRAY(EXACT_BOOLEAN, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_EXACT_TEXT(EXACT_BOOLEAN, EXACT_TEXT),
        COMPOSITE_EXACT_BOOLEAN_EXACT_TEXT_ARRAY(EXACT_BOOLEAN, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_EXACT_GEOMETRY(EXACT_BOOLEAN, EXACT_GEOMETRY),
        COMPOSITE_EXACT_BOOLEAN_EXACT_GEOMETRY_ARRAY(EXACT_BOOLEAN, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_EXACT_TEMPORAL(EXACT_BOOLEAN, EXACT_TEMPORAL),
        COMPOSITE_EXACT_BOOLEAN_EXACT_TEMPORAL_ARRAY(EXACT_BOOLEAN, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_EXACT_BOOLEAN(EXACT_BOOLEAN, EXACT_BOOLEAN),
        COMPOSITE_EXACT_BOOLEAN_EXACT_BOOLEAN_ARRAY(EXACT_BOOLEAN, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_RANGE_NUMBER(EXACT_BOOLEAN, RANGE_NUMBER),
        COMPOSITE_EXACT_BOOLEAN_RANGE_NUMBER_ARRAY(EXACT_BOOLEAN, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_RANGE_TEXT(EXACT_BOOLEAN, RANGE_TEXT),
        COMPOSITE_EXACT_BOOLEAN_RANGE_TEXT_ARRAY(EXACT_BOOLEAN, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_RANGE_GEOMETRY(EXACT_BOOLEAN, RANGE_GEOMETRY),
        COMPOSITE_EXACT_BOOLEAN_RANGE_GEOMETRY_ARRAY(EXACT_BOOLEAN, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_RANGE_TEMPORAL(EXACT_BOOLEAN, RANGE_TEMPORAL),
        COMPOSITE_EXACT_BOOLEAN_RANGE_TEMPORAL_ARRAY(EXACT_BOOLEAN, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_RANGE_BOOLEAN(EXACT_BOOLEAN, RANGE_BOOLEAN),
        COMPOSITE_EXACT_BOOLEAN_RANGE_BOOLEAN_ARRAY(EXACT_BOOLEAN, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_BOUNDING_BOX(EXACT_BOOLEAN, BOUNDING_BOX),
        COMPOSITE_EXACT_BOOLEAN_STRING_PREFIX(EXACT_BOOLEAN, STRING_PREFIX),
        COMPOSITE_EXACT_BOOLEAN_STRING_SUFFIX(EXACT_BOOLEAN, STRING_SUFFIX),
        COMPOSITE_EXACT_BOOLEAN_STRING_CONTAINS(EXACT_BOOLEAN, STRING_CONTAINS),
        COMPOSITE_EXACT_BOOLEAN_FULLTEXT_SEARCH(EXACT_BOOLEAN, FULLTEXT_SEARCH),

        COMPOSITE_EXACT_BOOLEAN_ARRAY_TOKEN_LOOKUP(EXACT_BOOLEAN_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_ALL_ENTRIES(EXACT_BOOLEAN_ARRAY, ALL_ENTRIES),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXISTS(EXACT_BOOLEAN_ARRAY, EXISTS),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_NUMBER(EXACT_BOOLEAN_ARRAY, EXACT_NUMBER),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_NUMBER_ARRAY(EXACT_BOOLEAN_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_TEXT(EXACT_BOOLEAN_ARRAY, EXACT_TEXT),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_TEXT_ARRAY(EXACT_BOOLEAN_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_GEOMETRY(EXACT_BOOLEAN_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_GEOMETRY_ARRAY(EXACT_BOOLEAN_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_TEMPORAL(EXACT_BOOLEAN_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_TEMPORAL_ARRAY(EXACT_BOOLEAN_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_BOOLEAN(EXACT_BOOLEAN_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_EXACT_BOOLEAN_ARRAY(EXACT_BOOLEAN_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_NUMBER(EXACT_BOOLEAN_ARRAY, RANGE_NUMBER),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_NUMBER_ARRAY(EXACT_BOOLEAN_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_TEXT(EXACT_BOOLEAN_ARRAY, RANGE_TEXT),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_TEXT_ARRAY(EXACT_BOOLEAN_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_GEOMETRY(EXACT_BOOLEAN_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_GEOMETRY_ARRAY(EXACT_BOOLEAN_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_TEMPORAL(EXACT_BOOLEAN_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_TEMPORAL_ARRAY(EXACT_BOOLEAN_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_BOOLEAN(EXACT_BOOLEAN_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_RANGE_BOOLEAN_ARRAY(EXACT_BOOLEAN_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_BOUNDING_BOX(EXACT_BOOLEAN_ARRAY, BOUNDING_BOX),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_STRING_PREFIX(EXACT_BOOLEAN_ARRAY, STRING_PREFIX),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_STRING_SUFFIX(EXACT_BOOLEAN_ARRAY, STRING_SUFFIX),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_STRING_CONTAINS(EXACT_BOOLEAN_ARRAY, STRING_CONTAINS),
        COMPOSITE_EXACT_BOOLEAN_ARRAY_FULLTEXT_SEARCH(EXACT_BOOLEAN_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_NUMBER_TOKEN_LOOKUP(RANGE_NUMBER, TOKEN_LOOKUP),
        COMPOSITE_RANGE_NUMBER_ALL_ENTRIES(RANGE_NUMBER, ALL_ENTRIES),
        COMPOSITE_RANGE_NUMBER_EXISTS(RANGE_NUMBER, EXISTS),
        COMPOSITE_RANGE_NUMBER_EXACT_NUMBER(RANGE_NUMBER, EXACT_NUMBER),
        COMPOSITE_RANGE_NUMBER_EXACT_NUMBER_ARRAY(RANGE_NUMBER, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_NUMBER_EXACT_TEXT(RANGE_NUMBER, EXACT_TEXT),
        COMPOSITE_RANGE_NUMBER_EXACT_TEXT_ARRAY(RANGE_NUMBER, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_NUMBER_EXACT_GEOMETRY(RANGE_NUMBER, EXACT_GEOMETRY),
        COMPOSITE_RANGE_NUMBER_EXACT_GEOMETRY_ARRAY(RANGE_NUMBER, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_NUMBER_EXACT_TEMPORAL(RANGE_NUMBER, EXACT_TEMPORAL),
        COMPOSITE_RANGE_NUMBER_EXACT_TEMPORAL_ARRAY(RANGE_NUMBER, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_NUMBER_EXACT_BOOLEAN(RANGE_NUMBER, EXACT_BOOLEAN),
        COMPOSITE_RANGE_NUMBER_EXACT_BOOLEAN_ARRAY(RANGE_NUMBER, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_NUMBER_RANGE_NUMBER(RANGE_NUMBER, RANGE_NUMBER),
        COMPOSITE_RANGE_NUMBER_RANGE_NUMBER_ARRAY(RANGE_NUMBER, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_NUMBER_RANGE_TEXT(RANGE_NUMBER, RANGE_TEXT),
        COMPOSITE_RANGE_NUMBER_RANGE_TEXT_ARRAY(RANGE_NUMBER, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_NUMBER_RANGE_GEOMETRY(RANGE_NUMBER, RANGE_GEOMETRY),
        COMPOSITE_RANGE_NUMBER_RANGE_GEOMETRY_ARRAY(RANGE_NUMBER, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_NUMBER_RANGE_TEMPORAL(RANGE_NUMBER, RANGE_TEMPORAL),
        COMPOSITE_RANGE_NUMBER_RANGE_TEMPORAL_ARRAY(RANGE_NUMBER, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_NUMBER_RANGE_BOOLEAN(RANGE_NUMBER, RANGE_BOOLEAN),
        COMPOSITE_RANGE_NUMBER_RANGE_BOOLEAN_ARRAY(RANGE_NUMBER, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_NUMBER_BOUNDING_BOX(RANGE_NUMBER, BOUNDING_BOX),
        COMPOSITE_RANGE_NUMBER_STRING_PREFIX(RANGE_NUMBER, STRING_PREFIX),
        COMPOSITE_RANGE_NUMBER_STRING_SUFFIX(RANGE_NUMBER, STRING_SUFFIX),
        COMPOSITE_RANGE_NUMBER_STRING_CONTAINS(RANGE_NUMBER, STRING_CONTAINS),
        COMPOSITE_RANGE_NUMBER_FULLTEXT_SEARCH(RANGE_NUMBER, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_NUMBER_ARRAY_TOKEN_LOOKUP(RANGE_NUMBER_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_RANGE_NUMBER_ARRAY_ALL_ENTRIES(RANGE_NUMBER_ARRAY, ALL_ENTRIES),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXISTS(RANGE_NUMBER_ARRAY, EXISTS),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_NUMBER(RANGE_NUMBER_ARRAY, EXACT_NUMBER),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_NUMBER_ARRAY(RANGE_NUMBER_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_TEXT(RANGE_NUMBER_ARRAY, EXACT_TEXT),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_TEXT_ARRAY(RANGE_NUMBER_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_GEOMETRY(RANGE_NUMBER_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_GEOMETRY_ARRAY(RANGE_NUMBER_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_TEMPORAL(RANGE_NUMBER_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_TEMPORAL_ARRAY(RANGE_NUMBER_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_BOOLEAN(RANGE_NUMBER_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_RANGE_NUMBER_ARRAY_EXACT_BOOLEAN_ARRAY(RANGE_NUMBER_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_NUMBER(RANGE_NUMBER_ARRAY, RANGE_NUMBER),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_NUMBER_ARRAY(RANGE_NUMBER_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_TEXT(RANGE_NUMBER_ARRAY, RANGE_TEXT),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_TEXT_ARRAY(RANGE_NUMBER_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_GEOMETRY(RANGE_NUMBER_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_GEOMETRY_ARRAY(RANGE_NUMBER_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_TEMPORAL(RANGE_NUMBER_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_TEMPORAL_ARRAY(RANGE_NUMBER_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_BOOLEAN(RANGE_NUMBER_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_RANGE_NUMBER_ARRAY_RANGE_BOOLEAN_ARRAY(RANGE_NUMBER_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_NUMBER_ARRAY_BOUNDING_BOX(RANGE_NUMBER_ARRAY, BOUNDING_BOX),
        COMPOSITE_RANGE_NUMBER_ARRAY_STRING_PREFIX(RANGE_NUMBER_ARRAY, STRING_PREFIX),
        COMPOSITE_RANGE_NUMBER_ARRAY_STRING_SUFFIX(RANGE_NUMBER_ARRAY, STRING_SUFFIX),
        COMPOSITE_RANGE_NUMBER_ARRAY_STRING_CONTAINS(RANGE_NUMBER_ARRAY, STRING_CONTAINS),
        COMPOSITE_RANGE_NUMBER_ARRAY_FULLTEXT_SEARCH(RANGE_NUMBER_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_TEXT_TOKEN_LOOKUP(RANGE_TEXT, TOKEN_LOOKUP),
        COMPOSITE_RANGE_TEXT_ALL_ENTRIES(RANGE_TEXT, ALL_ENTRIES),
        COMPOSITE_RANGE_TEXT_EXISTS(RANGE_TEXT, EXISTS),
        COMPOSITE_RANGE_TEXT_EXACT_NUMBER(RANGE_TEXT, EXACT_NUMBER),
        COMPOSITE_RANGE_TEXT_EXACT_NUMBER_ARRAY(RANGE_TEXT, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_TEXT_EXACT_TEXT(RANGE_TEXT, EXACT_TEXT),
        COMPOSITE_RANGE_TEXT_EXACT_TEXT_ARRAY(RANGE_TEXT, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_TEXT_EXACT_GEOMETRY(RANGE_TEXT, EXACT_GEOMETRY),
        COMPOSITE_RANGE_TEXT_EXACT_GEOMETRY_ARRAY(RANGE_TEXT, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_TEXT_EXACT_TEMPORAL(RANGE_TEXT, EXACT_TEMPORAL),
        COMPOSITE_RANGE_TEXT_EXACT_TEMPORAL_ARRAY(RANGE_TEXT, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_TEXT_EXACT_BOOLEAN(RANGE_TEXT, EXACT_BOOLEAN),
        COMPOSITE_RANGE_TEXT_EXACT_BOOLEAN_ARRAY(RANGE_TEXT, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_TEXT_RANGE_NUMBER(RANGE_TEXT, RANGE_NUMBER),
        COMPOSITE_RANGE_TEXT_RANGE_NUMBER_ARRAY(RANGE_TEXT, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_TEXT_RANGE_TEXT(RANGE_TEXT, RANGE_TEXT),
        COMPOSITE_RANGE_TEXT_RANGE_TEXT_ARRAY(RANGE_TEXT, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_TEXT_RANGE_GEOMETRY(RANGE_TEXT, RANGE_GEOMETRY),
        COMPOSITE_RANGE_TEXT_RANGE_GEOMETRY_ARRAY(RANGE_TEXT, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_TEXT_RANGE_TEMPORAL(RANGE_TEXT, RANGE_TEMPORAL),
        COMPOSITE_RANGE_TEXT_RANGE_TEMPORAL_ARRAY(RANGE_TEXT, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_TEXT_RANGE_BOOLEAN(RANGE_TEXT, RANGE_BOOLEAN),
        COMPOSITE_RANGE_TEXT_RANGE_BOOLEAN_ARRAY(RANGE_TEXT, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_TEXT_BOUNDING_BOX(RANGE_TEXT, BOUNDING_BOX),
        COMPOSITE_RANGE_TEXT_STRING_PREFIX(RANGE_TEXT, STRING_PREFIX),
        COMPOSITE_RANGE_TEXT_STRING_SUFFIX(RANGE_TEXT, STRING_SUFFIX),
        COMPOSITE_RANGE_TEXT_STRING_CONTAINS(RANGE_TEXT, STRING_CONTAINS),
        COMPOSITE_RANGE_TEXT_FULLTEXT_SEARCH(RANGE_TEXT, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_TEXT_ARRAY_TOKEN_LOOKUP(RANGE_TEXT_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_RANGE_TEXT_ARRAY_ALL_ENTRIES(RANGE_TEXT_ARRAY, ALL_ENTRIES),
        COMPOSITE_RANGE_TEXT_ARRAY_EXISTS(RANGE_TEXT_ARRAY, EXISTS),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_NUMBER(RANGE_TEXT_ARRAY, EXACT_NUMBER),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_NUMBER_ARRAY(RANGE_TEXT_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_TEXT(RANGE_TEXT_ARRAY, EXACT_TEXT),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_TEXT_ARRAY(RANGE_TEXT_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_GEOMETRY(RANGE_TEXT_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_GEOMETRY_ARRAY(RANGE_TEXT_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_TEMPORAL(RANGE_TEXT_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_TEMPORAL_ARRAY(RANGE_TEXT_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_BOOLEAN(RANGE_TEXT_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_RANGE_TEXT_ARRAY_EXACT_BOOLEAN_ARRAY(RANGE_TEXT_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_NUMBER(RANGE_TEXT_ARRAY, RANGE_NUMBER),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_NUMBER_ARRAY(RANGE_TEXT_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_TEXT(RANGE_TEXT_ARRAY, RANGE_TEXT),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_TEXT_ARRAY(RANGE_TEXT_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_GEOMETRY(RANGE_TEXT_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_GEOMETRY_ARRAY(RANGE_TEXT_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_TEMPORAL(RANGE_TEXT_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_TEMPORAL_ARRAY(RANGE_TEXT_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_BOOLEAN(RANGE_TEXT_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_RANGE_TEXT_ARRAY_RANGE_BOOLEAN_ARRAY(RANGE_TEXT_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_TEXT_ARRAY_BOUNDING_BOX(RANGE_TEXT_ARRAY, BOUNDING_BOX),
        COMPOSITE_RANGE_TEXT_ARRAY_STRING_PREFIX(RANGE_TEXT_ARRAY, STRING_PREFIX),
        COMPOSITE_RANGE_TEXT_ARRAY_STRING_SUFFIX(RANGE_TEXT_ARRAY, STRING_SUFFIX),
        COMPOSITE_RANGE_TEXT_ARRAY_STRING_CONTAINS(RANGE_TEXT_ARRAY, STRING_CONTAINS),
        COMPOSITE_RANGE_TEXT_ARRAY_FULLTEXT_SEARCH(RANGE_TEXT_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_GEOMETRY_TOKEN_LOOKUP(RANGE_GEOMETRY, TOKEN_LOOKUP),
        COMPOSITE_RANGE_GEOMETRY_ALL_ENTRIES(RANGE_GEOMETRY, ALL_ENTRIES),
        COMPOSITE_RANGE_GEOMETRY_EXISTS(RANGE_GEOMETRY, EXISTS),
        COMPOSITE_RANGE_GEOMETRY_EXACT_NUMBER(RANGE_GEOMETRY, EXACT_NUMBER),
        COMPOSITE_RANGE_GEOMETRY_EXACT_NUMBER_ARRAY(RANGE_GEOMETRY, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_EXACT_TEXT(RANGE_GEOMETRY, EXACT_TEXT),
        COMPOSITE_RANGE_GEOMETRY_EXACT_TEXT_ARRAY(RANGE_GEOMETRY, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_EXACT_GEOMETRY(RANGE_GEOMETRY, EXACT_GEOMETRY),
        COMPOSITE_RANGE_GEOMETRY_EXACT_GEOMETRY_ARRAY(RANGE_GEOMETRY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_EXACT_TEMPORAL(RANGE_GEOMETRY, EXACT_TEMPORAL),
        COMPOSITE_RANGE_GEOMETRY_EXACT_TEMPORAL_ARRAY(RANGE_GEOMETRY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_EXACT_BOOLEAN(RANGE_GEOMETRY, EXACT_BOOLEAN),
        COMPOSITE_RANGE_GEOMETRY_EXACT_BOOLEAN_ARRAY(RANGE_GEOMETRY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_RANGE_NUMBER(RANGE_GEOMETRY, RANGE_NUMBER),
        COMPOSITE_RANGE_GEOMETRY_RANGE_NUMBER_ARRAY(RANGE_GEOMETRY, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_RANGE_TEXT(RANGE_GEOMETRY, RANGE_TEXT),
        COMPOSITE_RANGE_GEOMETRY_RANGE_TEXT_ARRAY(RANGE_GEOMETRY, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_RANGE_GEOMETRY(RANGE_GEOMETRY, RANGE_GEOMETRY),
        COMPOSITE_RANGE_GEOMETRY_RANGE_GEOMETRY_ARRAY(RANGE_GEOMETRY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_RANGE_TEMPORAL(RANGE_GEOMETRY, RANGE_TEMPORAL),
        COMPOSITE_RANGE_GEOMETRY_RANGE_TEMPORAL_ARRAY(RANGE_GEOMETRY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_RANGE_BOOLEAN(RANGE_GEOMETRY, RANGE_BOOLEAN),
        COMPOSITE_RANGE_GEOMETRY_RANGE_BOOLEAN_ARRAY(RANGE_GEOMETRY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_BOUNDING_BOX(RANGE_GEOMETRY, BOUNDING_BOX),
        COMPOSITE_RANGE_GEOMETRY_STRING_PREFIX(RANGE_GEOMETRY, STRING_PREFIX),
        COMPOSITE_RANGE_GEOMETRY_STRING_SUFFIX(RANGE_GEOMETRY, STRING_SUFFIX),
        COMPOSITE_RANGE_GEOMETRY_STRING_CONTAINS(RANGE_GEOMETRY, STRING_CONTAINS),
        COMPOSITE_RANGE_GEOMETRY_FULLTEXT_SEARCH(RANGE_GEOMETRY, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_GEOMETRY_ARRAY_TOKEN_LOOKUP(RANGE_GEOMETRY_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_ALL_ENTRIES(RANGE_GEOMETRY_ARRAY, ALL_ENTRIES),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXISTS(RANGE_GEOMETRY_ARRAY, EXISTS),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_NUMBER(RANGE_GEOMETRY_ARRAY, EXACT_NUMBER),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_NUMBER_ARRAY(RANGE_GEOMETRY_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_TEXT(RANGE_GEOMETRY_ARRAY, EXACT_TEXT),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_TEXT_ARRAY(RANGE_GEOMETRY_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_GEOMETRY(RANGE_GEOMETRY_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_GEOMETRY_ARRAY(RANGE_GEOMETRY_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_TEMPORAL(RANGE_GEOMETRY_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_TEMPORAL_ARRAY(RANGE_GEOMETRY_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_BOOLEAN(RANGE_GEOMETRY_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_EXACT_BOOLEAN_ARRAY(RANGE_GEOMETRY_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_NUMBER(RANGE_GEOMETRY_ARRAY, RANGE_NUMBER),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_NUMBER_ARRAY(RANGE_GEOMETRY_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_TEXT(RANGE_GEOMETRY_ARRAY, RANGE_TEXT),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_TEXT_ARRAY(RANGE_GEOMETRY_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_GEOMETRY(RANGE_GEOMETRY_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_GEOMETRY_ARRAY(RANGE_GEOMETRY_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_TEMPORAL(RANGE_GEOMETRY_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_TEMPORAL_ARRAY(RANGE_GEOMETRY_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_BOOLEAN(RANGE_GEOMETRY_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_RANGE_BOOLEAN_ARRAY(RANGE_GEOMETRY_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_BOUNDING_BOX(RANGE_GEOMETRY_ARRAY, BOUNDING_BOX),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_STRING_PREFIX(RANGE_GEOMETRY_ARRAY, STRING_PREFIX),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_STRING_SUFFIX(RANGE_GEOMETRY_ARRAY, STRING_SUFFIX),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_STRING_CONTAINS(RANGE_GEOMETRY_ARRAY, STRING_CONTAINS),
        COMPOSITE_RANGE_GEOMETRY_ARRAY_FULLTEXT_SEARCH(RANGE_GEOMETRY_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_TEMPORAL_TOKEN_LOOKUP(RANGE_TEMPORAL, TOKEN_LOOKUP),
        COMPOSITE_RANGE_TEMPORAL_ALL_ENTRIES(RANGE_TEMPORAL, ALL_ENTRIES),
        COMPOSITE_RANGE_TEMPORAL_EXISTS(RANGE_TEMPORAL, EXISTS),
        COMPOSITE_RANGE_TEMPORAL_EXACT_NUMBER(RANGE_TEMPORAL, EXACT_NUMBER),
        COMPOSITE_RANGE_TEMPORAL_EXACT_NUMBER_ARRAY(RANGE_TEMPORAL, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_EXACT_TEXT(RANGE_TEMPORAL, EXACT_TEXT),
        COMPOSITE_RANGE_TEMPORAL_EXACT_TEXT_ARRAY(RANGE_TEMPORAL, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_EXACT_GEOMETRY(RANGE_TEMPORAL, EXACT_GEOMETRY),
        COMPOSITE_RANGE_TEMPORAL_EXACT_GEOMETRY_ARRAY(RANGE_TEMPORAL, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_EXACT_TEMPORAL(RANGE_TEMPORAL, EXACT_TEMPORAL),
        COMPOSITE_RANGE_TEMPORAL_EXACT_TEMPORAL_ARRAY(RANGE_TEMPORAL, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_EXACT_BOOLEAN(RANGE_TEMPORAL, EXACT_BOOLEAN),
        COMPOSITE_RANGE_TEMPORAL_EXACT_BOOLEAN_ARRAY(RANGE_TEMPORAL, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_RANGE_NUMBER(RANGE_TEMPORAL, RANGE_NUMBER),
        COMPOSITE_RANGE_TEMPORAL_RANGE_NUMBER_ARRAY(RANGE_TEMPORAL, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_RANGE_TEXT(RANGE_TEMPORAL, RANGE_TEXT),
        COMPOSITE_RANGE_TEMPORAL_RANGE_TEXT_ARRAY(RANGE_TEMPORAL, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_RANGE_GEOMETRY(RANGE_TEMPORAL, RANGE_GEOMETRY),
        COMPOSITE_RANGE_TEMPORAL_RANGE_GEOMETRY_ARRAY(RANGE_TEMPORAL, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_RANGE_TEMPORAL(RANGE_TEMPORAL, RANGE_TEMPORAL),
        COMPOSITE_RANGE_TEMPORAL_RANGE_TEMPORAL_ARRAY(RANGE_TEMPORAL, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_RANGE_BOOLEAN(RANGE_TEMPORAL, RANGE_BOOLEAN),
        COMPOSITE_RANGE_TEMPORAL_RANGE_BOOLEAN_ARRAY(RANGE_TEMPORAL, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_BOUNDING_BOX(RANGE_TEMPORAL, BOUNDING_BOX),
        COMPOSITE_RANGE_TEMPORAL_STRING_PREFIX(RANGE_TEMPORAL, STRING_PREFIX),
        COMPOSITE_RANGE_TEMPORAL_STRING_SUFFIX(RANGE_TEMPORAL, STRING_SUFFIX),
        COMPOSITE_RANGE_TEMPORAL_STRING_CONTAINS(RANGE_TEMPORAL, STRING_CONTAINS),
        COMPOSITE_RANGE_TEMPORAL_FULLTEXT_SEARCH(RANGE_TEMPORAL, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_TEMPORAL_ARRAY_TOKEN_LOOKUP(RANGE_TEMPORAL_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_ALL_ENTRIES(RANGE_TEMPORAL_ARRAY, ALL_ENTRIES),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXISTS(RANGE_TEMPORAL_ARRAY, EXISTS),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_NUMBER(RANGE_TEMPORAL_ARRAY, EXACT_NUMBER),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_NUMBER_ARRAY(RANGE_TEMPORAL_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_TEXT(RANGE_TEMPORAL_ARRAY, EXACT_TEXT),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_TEXT_ARRAY(RANGE_TEMPORAL_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_GEOMETRY(RANGE_TEMPORAL_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_GEOMETRY_ARRAY(RANGE_TEMPORAL_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_TEMPORAL(RANGE_TEMPORAL_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_TEMPORAL_ARRAY(RANGE_TEMPORAL_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_BOOLEAN(RANGE_TEMPORAL_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_EXACT_BOOLEAN_ARRAY(RANGE_TEMPORAL_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_NUMBER(RANGE_TEMPORAL_ARRAY, RANGE_NUMBER),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_NUMBER_ARRAY(RANGE_TEMPORAL_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_TEXT(RANGE_TEMPORAL_ARRAY, RANGE_TEXT),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_TEXT_ARRAY(RANGE_TEMPORAL_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_GEOMETRY(RANGE_TEMPORAL_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_GEOMETRY_ARRAY(RANGE_TEMPORAL_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_TEMPORAL(RANGE_TEMPORAL_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_TEMPORAL_ARRAY(RANGE_TEMPORAL_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_BOOLEAN(RANGE_TEMPORAL_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_RANGE_BOOLEAN_ARRAY(RANGE_TEMPORAL_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_BOUNDING_BOX(RANGE_TEMPORAL_ARRAY, BOUNDING_BOX),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_STRING_PREFIX(RANGE_TEMPORAL_ARRAY, STRING_PREFIX),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_STRING_SUFFIX(RANGE_TEMPORAL_ARRAY, STRING_SUFFIX),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_STRING_CONTAINS(RANGE_TEMPORAL_ARRAY, STRING_CONTAINS),
        COMPOSITE_RANGE_TEMPORAL_ARRAY_FULLTEXT_SEARCH(RANGE_TEMPORAL_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_BOOLEAN_TOKEN_LOOKUP(RANGE_BOOLEAN, TOKEN_LOOKUP),
        COMPOSITE_RANGE_BOOLEAN_ALL_ENTRIES(RANGE_BOOLEAN, ALL_ENTRIES),
        COMPOSITE_RANGE_BOOLEAN_EXISTS(RANGE_BOOLEAN, EXISTS),
        COMPOSITE_RANGE_BOOLEAN_EXACT_NUMBER(RANGE_BOOLEAN, EXACT_NUMBER),
        COMPOSITE_RANGE_BOOLEAN_EXACT_NUMBER_ARRAY(RANGE_BOOLEAN, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_EXACT_TEXT(RANGE_BOOLEAN, EXACT_TEXT),
        COMPOSITE_RANGE_BOOLEAN_EXACT_TEXT_ARRAY(RANGE_BOOLEAN, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_EXACT_GEOMETRY(RANGE_BOOLEAN, EXACT_GEOMETRY),
        COMPOSITE_RANGE_BOOLEAN_EXACT_GEOMETRY_ARRAY(RANGE_BOOLEAN, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_EXACT_TEMPORAL(RANGE_BOOLEAN, EXACT_TEMPORAL),
        COMPOSITE_RANGE_BOOLEAN_EXACT_TEMPORAL_ARRAY(RANGE_BOOLEAN, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_EXACT_BOOLEAN(RANGE_BOOLEAN, EXACT_BOOLEAN),
        COMPOSITE_RANGE_BOOLEAN_EXACT_BOOLEAN_ARRAY(RANGE_BOOLEAN, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_RANGE_NUMBER(RANGE_BOOLEAN, RANGE_NUMBER),
        COMPOSITE_RANGE_BOOLEAN_RANGE_NUMBER_ARRAY(RANGE_BOOLEAN, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_RANGE_TEXT(RANGE_BOOLEAN, RANGE_TEXT),
        COMPOSITE_RANGE_BOOLEAN_RANGE_TEXT_ARRAY(RANGE_BOOLEAN, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_RANGE_GEOMETRY(RANGE_BOOLEAN, RANGE_GEOMETRY),
        COMPOSITE_RANGE_BOOLEAN_RANGE_GEOMETRY_ARRAY(RANGE_BOOLEAN, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_RANGE_TEMPORAL(RANGE_BOOLEAN, RANGE_TEMPORAL),
        COMPOSITE_RANGE_BOOLEAN_RANGE_TEMPORAL_ARRAY(RANGE_BOOLEAN, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_RANGE_BOOLEAN(RANGE_BOOLEAN, RANGE_BOOLEAN),
        COMPOSITE_RANGE_BOOLEAN_RANGE_BOOLEAN_ARRAY(RANGE_BOOLEAN, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_BOUNDING_BOX(RANGE_BOOLEAN, BOUNDING_BOX),
        COMPOSITE_RANGE_BOOLEAN_STRING_PREFIX(RANGE_BOOLEAN, STRING_PREFIX),
        COMPOSITE_RANGE_BOOLEAN_STRING_SUFFIX(RANGE_BOOLEAN, STRING_SUFFIX),
        COMPOSITE_RANGE_BOOLEAN_STRING_CONTAINS(RANGE_BOOLEAN, STRING_CONTAINS),
        COMPOSITE_RANGE_BOOLEAN_FULLTEXT_SEARCH(RANGE_BOOLEAN, FULLTEXT_SEARCH),

        COMPOSITE_RANGE_BOOLEAN_ARRAY_TOKEN_LOOKUP(RANGE_BOOLEAN_ARRAY, TOKEN_LOOKUP),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_ALL_ENTRIES(RANGE_BOOLEAN_ARRAY, ALL_ENTRIES),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXISTS(RANGE_BOOLEAN_ARRAY, EXISTS),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_NUMBER(RANGE_BOOLEAN_ARRAY, EXACT_NUMBER),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_NUMBER_ARRAY(RANGE_BOOLEAN_ARRAY, EXACT_NUMBER_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_TEXT(RANGE_BOOLEAN_ARRAY, EXACT_TEXT),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_TEXT_ARRAY(RANGE_BOOLEAN_ARRAY, EXACT_TEXT_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_GEOMETRY(RANGE_BOOLEAN_ARRAY, EXACT_GEOMETRY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_GEOMETRY_ARRAY(RANGE_BOOLEAN_ARRAY, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_TEMPORAL(RANGE_BOOLEAN_ARRAY, EXACT_TEMPORAL),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_TEMPORAL_ARRAY(RANGE_BOOLEAN_ARRAY, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_BOOLEAN(RANGE_BOOLEAN_ARRAY, EXACT_BOOLEAN),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_EXACT_BOOLEAN_ARRAY(RANGE_BOOLEAN_ARRAY, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_NUMBER(RANGE_BOOLEAN_ARRAY, RANGE_NUMBER),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_NUMBER_ARRAY(RANGE_BOOLEAN_ARRAY, RANGE_NUMBER_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_TEXT(RANGE_BOOLEAN_ARRAY, RANGE_TEXT),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_TEXT_ARRAY(RANGE_BOOLEAN_ARRAY, RANGE_TEXT_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_GEOMETRY(RANGE_BOOLEAN_ARRAY, RANGE_GEOMETRY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_GEOMETRY_ARRAY(RANGE_BOOLEAN_ARRAY, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_TEMPORAL(RANGE_BOOLEAN_ARRAY, RANGE_TEMPORAL),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_TEMPORAL_ARRAY(RANGE_BOOLEAN_ARRAY, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_BOOLEAN(RANGE_BOOLEAN_ARRAY, RANGE_BOOLEAN),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_RANGE_BOOLEAN_ARRAY(RANGE_BOOLEAN_ARRAY, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_BOUNDING_BOX(RANGE_BOOLEAN_ARRAY, BOUNDING_BOX),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_STRING_PREFIX(RANGE_BOOLEAN_ARRAY, STRING_PREFIX),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_STRING_SUFFIX(RANGE_BOOLEAN_ARRAY, STRING_SUFFIX),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_STRING_CONTAINS(RANGE_BOOLEAN_ARRAY, STRING_CONTAINS),
        COMPOSITE_RANGE_BOOLEAN_ARRAY_FULLTEXT_SEARCH(RANGE_BOOLEAN_ARRAY, FULLTEXT_SEARCH),

        COMPOSITE_BOUNDING_BOX_TOKEN_LOOKUP(BOUNDING_BOX, TOKEN_LOOKUP),
        COMPOSITE_BOUNDING_BOX_ALL_ENTRIES(BOUNDING_BOX, ALL_ENTRIES),
        COMPOSITE_BOUNDING_BOX_EXISTS(BOUNDING_BOX, EXISTS),
        COMPOSITE_BOUNDING_BOX_EXACT_NUMBER(BOUNDING_BOX, EXACT_NUMBER),
        COMPOSITE_BOUNDING_BOX_EXACT_NUMBER_ARRAY(BOUNDING_BOX, EXACT_NUMBER_ARRAY),
        COMPOSITE_BOUNDING_BOX_EXACT_TEXT(BOUNDING_BOX, EXACT_TEXT),
        COMPOSITE_BOUNDING_BOX_EXACT_TEXT_ARRAY(BOUNDING_BOX, EXACT_TEXT_ARRAY),
        COMPOSITE_BOUNDING_BOX_EXACT_GEOMETRY(BOUNDING_BOX, EXACT_GEOMETRY),
        COMPOSITE_BOUNDING_BOX_EXACT_GEOMETRY_ARRAY(BOUNDING_BOX, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_BOUNDING_BOX_EXACT_TEMPORAL(BOUNDING_BOX, EXACT_TEMPORAL),
        COMPOSITE_BOUNDING_BOX_EXACT_TEMPORAL_ARRAY(BOUNDING_BOX, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_BOUNDING_BOX_EXACT_BOOLEAN(BOUNDING_BOX, EXACT_BOOLEAN),
        COMPOSITE_BOUNDING_BOX_EXACT_BOOLEAN_ARRAY(BOUNDING_BOX, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_BOUNDING_BOX_RANGE_NUMBER(BOUNDING_BOX, RANGE_NUMBER),
        COMPOSITE_BOUNDING_BOX_RANGE_NUMBER_ARRAY(BOUNDING_BOX, RANGE_NUMBER_ARRAY),
        COMPOSITE_BOUNDING_BOX_RANGE_TEXT(BOUNDING_BOX, RANGE_TEXT),
        COMPOSITE_BOUNDING_BOX_RANGE_TEXT_ARRAY(BOUNDING_BOX, RANGE_TEXT_ARRAY),
        COMPOSITE_BOUNDING_BOX_RANGE_GEOMETRY(BOUNDING_BOX, RANGE_GEOMETRY),
        COMPOSITE_BOUNDING_BOX_RANGE_GEOMETRY_ARRAY(BOUNDING_BOX, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_BOUNDING_BOX_RANGE_TEMPORAL(BOUNDING_BOX, RANGE_TEMPORAL),
        COMPOSITE_BOUNDING_BOX_RANGE_TEMPORAL_ARRAY(BOUNDING_BOX, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_BOUNDING_BOX_RANGE_BOOLEAN(BOUNDING_BOX, RANGE_BOOLEAN),
        COMPOSITE_BOUNDING_BOX_RANGE_BOOLEAN_ARRAY(BOUNDING_BOX, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_BOUNDING_BOX_BOUNDING_BOX(BOUNDING_BOX, BOUNDING_BOX),
        COMPOSITE_BOUNDING_BOX_STRING_PREFIX(BOUNDING_BOX, STRING_PREFIX),
        COMPOSITE_BOUNDING_BOX_STRING_SUFFIX(BOUNDING_BOX, STRING_SUFFIX),
        COMPOSITE_BOUNDING_BOX_STRING_CONTAINS(BOUNDING_BOX, STRING_CONTAINS),
        COMPOSITE_BOUNDING_BOX_FULLTEXT_SEARCH(BOUNDING_BOX, FULLTEXT_SEARCH),

        COMPOSITE_STRING_PREFIX_TOKEN_LOOKUP(STRING_PREFIX, TOKEN_LOOKUP),
        COMPOSITE_STRING_PREFIX_ALL_ENTRIES(STRING_PREFIX, ALL_ENTRIES),
        COMPOSITE_STRING_PREFIX_EXISTS(STRING_PREFIX, EXISTS),
        COMPOSITE_STRING_PREFIX_EXACT_NUMBER(STRING_PREFIX, EXACT_NUMBER),
        COMPOSITE_STRING_PREFIX_EXACT_NUMBER_ARRAY(STRING_PREFIX, EXACT_NUMBER_ARRAY),
        COMPOSITE_STRING_PREFIX_EXACT_TEXT(STRING_PREFIX, EXACT_TEXT),
        COMPOSITE_STRING_PREFIX_EXACT_TEXT_ARRAY(STRING_PREFIX, EXACT_TEXT_ARRAY),
        COMPOSITE_STRING_PREFIX_EXACT_GEOMETRY(STRING_PREFIX, EXACT_GEOMETRY),
        COMPOSITE_STRING_PREFIX_EXACT_GEOMETRY_ARRAY(STRING_PREFIX, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_STRING_PREFIX_EXACT_TEMPORAL(STRING_PREFIX, EXACT_TEMPORAL),
        COMPOSITE_STRING_PREFIX_EXACT_TEMPORAL_ARRAY(STRING_PREFIX, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_STRING_PREFIX_EXACT_BOOLEAN(STRING_PREFIX, EXACT_BOOLEAN),
        COMPOSITE_STRING_PREFIX_EXACT_BOOLEAN_ARRAY(STRING_PREFIX, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_STRING_PREFIX_RANGE_NUMBER(STRING_PREFIX, RANGE_NUMBER),
        COMPOSITE_STRING_PREFIX_RANGE_NUMBER_ARRAY(STRING_PREFIX, RANGE_NUMBER_ARRAY),
        COMPOSITE_STRING_PREFIX_RANGE_TEXT(STRING_PREFIX, RANGE_TEXT),
        COMPOSITE_STRING_PREFIX_RANGE_TEXT_ARRAY(STRING_PREFIX, RANGE_TEXT_ARRAY),
        COMPOSITE_STRING_PREFIX_RANGE_GEOMETRY(STRING_PREFIX, RANGE_GEOMETRY),
        COMPOSITE_STRING_PREFIX_RANGE_GEOMETRY_ARRAY(STRING_PREFIX, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_STRING_PREFIX_RANGE_TEMPORAL(STRING_PREFIX, RANGE_TEMPORAL),
        COMPOSITE_STRING_PREFIX_RANGE_TEMPORAL_ARRAY(STRING_PREFIX, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_STRING_PREFIX_RANGE_BOOLEAN(STRING_PREFIX, RANGE_BOOLEAN),
        COMPOSITE_STRING_PREFIX_RANGE_BOOLEAN_ARRAY(STRING_PREFIX, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_STRING_PREFIX_BOUNDING_BOX(STRING_PREFIX, BOUNDING_BOX),
        COMPOSITE_STRING_PREFIX_STRING_PREFIX(STRING_PREFIX, STRING_PREFIX),
        COMPOSITE_STRING_PREFIX_STRING_SUFFIX(STRING_PREFIX, STRING_SUFFIX),
        COMPOSITE_STRING_PREFIX_STRING_CONTAINS(STRING_PREFIX, STRING_CONTAINS),
        COMPOSITE_STRING_PREFIX_FULLTEXT_SEARCH(STRING_PREFIX, FULLTEXT_SEARCH),

        COMPOSITE_STRING_SUFFIX_TOKEN_LOOKUP(STRING_SUFFIX, TOKEN_LOOKUP),
        COMPOSITE_STRING_SUFFIX_ALL_ENTRIES(STRING_SUFFIX, ALL_ENTRIES),
        COMPOSITE_STRING_SUFFIX_EXISTS(STRING_SUFFIX, EXISTS),
        COMPOSITE_STRING_SUFFIX_EXACT_NUMBER(STRING_SUFFIX, EXACT_NUMBER),
        COMPOSITE_STRING_SUFFIX_EXACT_NUMBER_ARRAY(STRING_SUFFIX, EXACT_NUMBER_ARRAY),
        COMPOSITE_STRING_SUFFIX_EXACT_TEXT(STRING_SUFFIX, EXACT_TEXT),
        COMPOSITE_STRING_SUFFIX_EXACT_TEXT_ARRAY(STRING_SUFFIX, EXACT_TEXT_ARRAY),
        COMPOSITE_STRING_SUFFIX_EXACT_GEOMETRY(STRING_SUFFIX, EXACT_GEOMETRY),
        COMPOSITE_STRING_SUFFIX_EXACT_GEOMETRY_ARRAY(STRING_SUFFIX, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_STRING_SUFFIX_EXACT_TEMPORAL(STRING_SUFFIX, EXACT_TEMPORAL),
        COMPOSITE_STRING_SUFFIX_EXACT_TEMPORAL_ARRAY(STRING_SUFFIX, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_STRING_SUFFIX_EXACT_BOOLEAN(STRING_SUFFIX, EXACT_BOOLEAN),
        COMPOSITE_STRING_SUFFIX_EXACT_BOOLEAN_ARRAY(STRING_SUFFIX, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_STRING_SUFFIX_RANGE_NUMBER(STRING_SUFFIX, RANGE_NUMBER),
        COMPOSITE_STRING_SUFFIX_RANGE_NUMBER_ARRAY(STRING_SUFFIX, RANGE_NUMBER_ARRAY),
        COMPOSITE_STRING_SUFFIX_RANGE_TEXT(STRING_SUFFIX, RANGE_TEXT),
        COMPOSITE_STRING_SUFFIX_RANGE_TEXT_ARRAY(STRING_SUFFIX, RANGE_TEXT_ARRAY),
        COMPOSITE_STRING_SUFFIX_RANGE_GEOMETRY(STRING_SUFFIX, RANGE_GEOMETRY),
        COMPOSITE_STRING_SUFFIX_RANGE_GEOMETRY_ARRAY(STRING_SUFFIX, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_STRING_SUFFIX_RANGE_TEMPORAL(STRING_SUFFIX, RANGE_TEMPORAL),
        COMPOSITE_STRING_SUFFIX_RANGE_TEMPORAL_ARRAY(STRING_SUFFIX, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_STRING_SUFFIX_RANGE_BOOLEAN(STRING_SUFFIX, RANGE_BOOLEAN),
        COMPOSITE_STRING_SUFFIX_RANGE_BOOLEAN_ARRAY(STRING_SUFFIX, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_STRING_SUFFIX_BOUNDING_BOX(STRING_SUFFIX, BOUNDING_BOX),
        COMPOSITE_STRING_SUFFIX_STRING_PREFIX(STRING_SUFFIX, STRING_PREFIX),
        COMPOSITE_STRING_SUFFIX_STRING_SUFFIX(STRING_SUFFIX, STRING_SUFFIX),
        COMPOSITE_STRING_SUFFIX_STRING_CONTAINS(STRING_SUFFIX, STRING_CONTAINS),
        COMPOSITE_STRING_SUFFIX_FULLTEXT_SEARCH(STRING_SUFFIX, FULLTEXT_SEARCH),

        COMPOSITE_STRING_CONTAINS_TOKEN_LOOKUP(STRING_CONTAINS, TOKEN_LOOKUP),
        COMPOSITE_STRING_CONTAINS_ALL_ENTRIES(STRING_CONTAINS, ALL_ENTRIES),
        COMPOSITE_STRING_CONTAINS_EXISTS(STRING_CONTAINS, EXISTS),
        COMPOSITE_STRING_CONTAINS_EXACT_NUMBER(STRING_CONTAINS, EXACT_NUMBER),
        COMPOSITE_STRING_CONTAINS_EXACT_NUMBER_ARRAY(STRING_CONTAINS, EXACT_NUMBER_ARRAY),
        COMPOSITE_STRING_CONTAINS_EXACT_TEXT(STRING_CONTAINS, EXACT_TEXT),
        COMPOSITE_STRING_CONTAINS_EXACT_TEXT_ARRAY(STRING_CONTAINS, EXACT_TEXT_ARRAY),
        COMPOSITE_STRING_CONTAINS_EXACT_GEOMETRY(STRING_CONTAINS, EXACT_GEOMETRY),
        COMPOSITE_STRING_CONTAINS_EXACT_GEOMETRY_ARRAY(STRING_CONTAINS, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_STRING_CONTAINS_EXACT_TEMPORAL(STRING_CONTAINS, EXACT_TEMPORAL),
        COMPOSITE_STRING_CONTAINS_EXACT_TEMPORAL_ARRAY(STRING_CONTAINS, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_STRING_CONTAINS_EXACT_BOOLEAN(STRING_CONTAINS, EXACT_BOOLEAN),
        COMPOSITE_STRING_CONTAINS_EXACT_BOOLEAN_ARRAY(STRING_CONTAINS, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_STRING_CONTAINS_RANGE_NUMBER(STRING_CONTAINS, RANGE_NUMBER),
        COMPOSITE_STRING_CONTAINS_RANGE_NUMBER_ARRAY(STRING_CONTAINS, RANGE_NUMBER_ARRAY),
        COMPOSITE_STRING_CONTAINS_RANGE_TEXT(STRING_CONTAINS, RANGE_TEXT),
        COMPOSITE_STRING_CONTAINS_RANGE_TEXT_ARRAY(STRING_CONTAINS, RANGE_TEXT_ARRAY),
        COMPOSITE_STRING_CONTAINS_RANGE_GEOMETRY(STRING_CONTAINS, RANGE_GEOMETRY),
        COMPOSITE_STRING_CONTAINS_RANGE_GEOMETRY_ARRAY(STRING_CONTAINS, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_STRING_CONTAINS_RANGE_TEMPORAL(STRING_CONTAINS, RANGE_TEMPORAL),
        COMPOSITE_STRING_CONTAINS_RANGE_TEMPORAL_ARRAY(STRING_CONTAINS, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_STRING_CONTAINS_RANGE_BOOLEAN(STRING_CONTAINS, RANGE_BOOLEAN),
        COMPOSITE_STRING_CONTAINS_RANGE_BOOLEAN_ARRAY(STRING_CONTAINS, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_STRING_CONTAINS_BOUNDING_BOX(STRING_CONTAINS, BOUNDING_BOX),
        COMPOSITE_STRING_CONTAINS_STRING_PREFIX(STRING_CONTAINS, STRING_PREFIX),
        COMPOSITE_STRING_CONTAINS_STRING_SUFFIX(STRING_CONTAINS, STRING_SUFFIX),
        COMPOSITE_STRING_CONTAINS_STRING_CONTAINS(STRING_CONTAINS, STRING_CONTAINS),
        COMPOSITE_STRING_CONTAINS_FULLTEXT_SEARCH(STRING_CONTAINS, FULLTEXT_SEARCH),

        COMPOSITE_FULLTEXT_SEARCH_TOKEN_LOOKUP(FULLTEXT_SEARCH, TOKEN_LOOKUP),
        COMPOSITE_FULLTEXT_SEARCH_ALL_ENTRIES(FULLTEXT_SEARCH, ALL_ENTRIES),
        COMPOSITE_FULLTEXT_SEARCH_EXISTS(FULLTEXT_SEARCH, EXISTS),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_NUMBER(FULLTEXT_SEARCH, EXACT_NUMBER),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_NUMBER_ARRAY(FULLTEXT_SEARCH, EXACT_NUMBER_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_TEXT(FULLTEXT_SEARCH, EXACT_TEXT),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_TEXT_ARRAY(FULLTEXT_SEARCH, EXACT_TEXT_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_GEOMETRY(FULLTEXT_SEARCH, EXACT_GEOMETRY),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_GEOMETRY_ARRAY(FULLTEXT_SEARCH, EXACT_GEOMETRY_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_TEMPORAL(FULLTEXT_SEARCH, EXACT_TEMPORAL),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_TEMPORAL_ARRAY(FULLTEXT_SEARCH, EXACT_TEMPORAL_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_BOOLEAN(FULLTEXT_SEARCH, EXACT_BOOLEAN),
        COMPOSITE_FULLTEXT_SEARCH_EXACT_BOOLEAN_ARRAY(FULLTEXT_SEARCH, EXACT_BOOLEAN_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_NUMBER(FULLTEXT_SEARCH, RANGE_NUMBER),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_NUMBER_ARRAY(FULLTEXT_SEARCH, RANGE_NUMBER_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_TEXT(FULLTEXT_SEARCH, RANGE_TEXT),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_TEXT_ARRAY(FULLTEXT_SEARCH, RANGE_TEXT_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_GEOMETRY(FULLTEXT_SEARCH, RANGE_GEOMETRY),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_GEOMETRY_ARRAY(FULLTEXT_SEARCH, RANGE_GEOMETRY_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_TEMPORAL(FULLTEXT_SEARCH, RANGE_TEMPORAL),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_TEMPORAL_ARRAY(FULLTEXT_SEARCH, RANGE_TEMPORAL_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_BOOLEAN(FULLTEXT_SEARCH, RANGE_BOOLEAN),
        COMPOSITE_FULLTEXT_SEARCH_RANGE_BOOLEAN_ARRAY(FULLTEXT_SEARCH, RANGE_BOOLEAN_ARRAY),
        COMPOSITE_FULLTEXT_SEARCH_BOUNDING_BOX(FULLTEXT_SEARCH, BOUNDING_BOX),
        COMPOSITE_FULLTEXT_SEARCH_STRING_PREFIX(FULLTEXT_SEARCH, STRING_PREFIX),
        COMPOSITE_FULLTEXT_SEARCH_STRING_SUFFIX(FULLTEXT_SEARCH, STRING_SUFFIX),
        COMPOSITE_FULLTEXT_SEARCH_STRING_CONTAINS(FULLTEXT_SEARCH, STRING_CONTAINS),
        COMPOSITE_FULLTEXT_SEARCH_FULLTEXT_SEARCH(FULLTEXT_SEARCH, FULLTEXT_SEARCH),
        ;

        private final IndexQueryFromId[] queryFromIds;

        Query(IndexQueryFromId... queryFromIds) {
            this.queryFromIds = queryFromIds;
        }

        Query(Query... queries) {
            this(Arrays.stream(queries)
                    .map(Query::queryFromIds)
                    .flatMap(Arrays::stream)
                    .toArray(IndexQueryFromId[]::new));
        }

        IndexQueryFromId[] queryFromIds() {
            return queryFromIds;
        }

        IndexQuery[] queries(IntSupplier ids) {
            return Arrays.stream(queryFromIds)
                    .map(query -> query.fromId(ids.getAsInt()))
                    .toArray(IndexQuery[]::new);
        }

        @FunctionalInterface
        interface IndexQueryFromId {
            IndexQuery fromId(int id);
        }
    }
}
