/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.search.fetch.subphase;

import com.carrotsearch.randomizedtesting.annotations.ParametersFactory;

import org.opensearch.action.search.SearchResponse;
import org.opensearch.common.settings.Settings;
import org.opensearch.core.common.bytes.BytesReference;
import org.opensearch.core.xcontent.MediaTypeRegistry;
import org.opensearch.core.xcontent.XContentHelper;
import org.opensearch.index.query.MatchQueryBuilder;
import org.opensearch.index.query.QueryBuilder;
import org.opensearch.index.query.QueryBuilders;
import org.opensearch.index.query.TermQueryBuilder;
import org.opensearch.search.SearchHit;
import org.opensearch.test.ParameterizedStaticSettingsOpenSearchIntegTestCase;

import java.util.Arrays;
import java.util.Collection;

import static org.opensearch.index.query.QueryBuilders.boolQuery;
import static org.opensearch.index.query.QueryBuilders.constantScoreQuery;
import static org.opensearch.index.query.QueryBuilders.matchAllQuery;
import static org.opensearch.index.query.QueryBuilders.matchQuery;
import static org.opensearch.index.query.QueryBuilders.queryStringQuery;
import static org.opensearch.index.query.QueryBuilders.rangeQuery;
import static org.opensearch.index.query.QueryBuilders.termQuery;
import static org.opensearch.index.query.QueryBuilders.termsQuery;
import static org.opensearch.index.query.QueryBuilders.wrapperQuery;
import static org.opensearch.search.SearchService.CLUSTER_CONCURRENT_SEGMENT_SEARCH_SETTING;
import static org.opensearch.test.hamcrest.OpenSearchAssertions.assertHitCount;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasItemInArray;
import static org.hamcrest.Matchers.hasKey;

public class MatchedQueriesIT extends ParameterizedStaticSettingsOpenSearchIntegTestCase {

    public MatchedQueriesIT(Settings staticSettings) {
        super(staticSettings);
    }

    @ParametersFactory
    public static Collection<Object[]> parameters() {
        return Arrays.asList(
            new Object[] { Settings.builder().put(CLUSTER_CONCURRENT_SEGMENT_SEARCH_SETTING.getKey(), false).build() },
            new Object[] { Settings.builder().put(CLUSTER_CONCURRENT_SEGMENT_SEARCH_SETTING.getKey(), true).build() }
        );
    }

    public void testSimpleMatchedQueryFromFilteredQuery() throws Exception {
        createIndex("test");
        ensureGreen();

        client().prepareIndex("test").setId("1").setSource("name", "test1", "number", 1).get();
        client().prepareIndex("test").setId("2").setSource("name", "test2", "number", 2).get();
        client().prepareIndex("test").setId("3").setSource("name", "test3", "number", 3).get();
        refresh();
        indexRandomForConcurrentSearch("test");

        SearchResponse searchResponse = client().prepareSearch()
            .setQuery(
                boolQuery().must(matchAllQuery())
                    .filter(
                        boolQuery().should(rangeQuery("number").lt(2).queryName("test1"))
                            .should(rangeQuery("number").gte(2).queryName("test2"))
                    )
            )
            .setIncludeNamedQueriesScore(true)
            .get();
        assertHitCount(searchResponse, 3L);
        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("3") || hit.getId().equals("2")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("test2"));
                assertThat(hit.getMatchedQueryScore("test2"), equalTo(1f));
            } else if (hit.getId().equals("1")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("test1"));
                assertThat(hit.getMatchedQueryScore("test1"), equalTo(1f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }

        searchResponse = client().prepareSearch()
            .setQuery(
                boolQuery().should(rangeQuery("number").lte(2).queryName("test1")).should(rangeQuery("number").gt(2).queryName("test2"))
            )
            .setIncludeNamedQueriesScore(true)
            .get();
        assertHitCount(searchResponse, 3L);
        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1") || hit.getId().equals("2")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("test1"));
                assertThat(hit.getMatchedQueryScore("test1"), equalTo(1f));
            } else if (hit.getId().equals("3")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("test2"));
                assertThat(hit.getMatchedQueryScore("test2"), equalTo(1f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }
    }

    public void testSimpleMatchedQueryFromTopLevelFilter() throws Exception {
        createIndex("test");
        ensureGreen();

        client().prepareIndex("test").setId("1").setSource("name", "test", "title", "title1").get();
        client().prepareIndex("test").setId("2").setSource("name", "test").get();
        client().prepareIndex("test").setId("3").setSource("name", "test").get();
        refresh();
        indexRandomForConcurrentSearch("test");

        SearchResponse searchResponse = client().prepareSearch()
            .setQuery(matchAllQuery())
            .setPostFilter(
                boolQuery().should(termQuery("name", "test").queryName("name")).should(termQuery("title", "title1").queryName("title"))
            )
            .get();
        assertHitCount(searchResponse, 3L);
        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(2));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("name"));
                assertThat(hit.getMatchedQueryScore("name"), greaterThan(0f));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("title"));
                assertThat(hit.getMatchedQueryScore("title"), greaterThan(0f));
            } else if (hit.getId().equals("2") || hit.getId().equals("3")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("name"));
                assertThat(hit.getMatchedQueryScore("name"), greaterThan(0f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }

        searchResponse = client().prepareSearch()
            .setQuery(matchAllQuery())
            .setPostFilter(
                boolQuery().should(termQuery("name", "test").queryName("name")).should(termQuery("title", "title1").queryName("title"))
            )
            .get();

        assertHitCount(searchResponse, 3L);
        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(2));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("name"));
                assertThat(hit.getMatchedQueryScore("name"), greaterThan(0f));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("title"));
                assertThat(hit.getMatchedQueryScore("title"), greaterThan(0f));
            } else if (hit.getId().equals("2") || hit.getId().equals("3")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("name"));
                assertThat(hit.getMatchedQueryScore("name"), greaterThan(0f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }
    }

    public void testSimpleMatchedQueryFromTopLevelFilterAndFilteredQuery() throws Exception {
        createIndex("test");
        ensureGreen();

        client().prepareIndex("test").setId("1").setSource("name", "test", "title", "title1").get();
        client().prepareIndex("test").setId("2").setSource("name", "test", "title", "title2").get();
        client().prepareIndex("test").setId("3").setSource("name", "test", "title", "title3").get();
        refresh();
        indexRandomForConcurrentSearch("test");

        SearchResponse searchResponse = client().prepareSearch()
            .setQuery(boolQuery().must(matchAllQuery()).filter(termsQuery("title", "title1", "title2", "title3").queryName("title")))
            .setPostFilter(termQuery("name", "test").queryName("name"))
            .get();
        assertHitCount(searchResponse, 3L);
        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1") || hit.getId().equals("2") || hit.getId().equals("3")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(2));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("name"));
                assertThat(hit.getMatchedQueryScore("name"), greaterThan(0f));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("title"));
                assertThat(hit.getMatchedQueryScore("title"), greaterThan(0f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }

        searchResponse = client().prepareSearch()
            .setQuery(termsQuery("title", "title1", "title2", "title3").queryName("title"))
            .setPostFilter(matchQuery("name", "test").queryName("name"))
            .get();
        assertHitCount(searchResponse, 3L);
        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1") || hit.getId().equals("2") || hit.getId().equals("3")) {
                assertThat(hit.getMatchedQueries().length, equalTo(2));
                assertThat(hit.getMatchedQueries(), hasItemInArray("name"));
                assertThat(hit.getMatchedQueries(), hasItemInArray("title"));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }
    }

    public void testRegExpQuerySupportsName() throws InterruptedException {
        createIndex("test1");
        ensureGreen();

        client().prepareIndex("test1").setId("1").setSource("title", "title1").get();
        refresh();
        indexRandomForConcurrentSearch("test1");

        SearchResponse searchResponse = client().prepareSearch()
            .setQuery(QueryBuilders.regexpQuery("title", "title1").queryName("regex"))
            .setIncludeNamedQueriesScore(true)
            .get();
        assertHitCount(searchResponse, 1L);

        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("regex"));
                assertThat(hit.getMatchedQueryScore("regex"), equalTo(1f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }
    }

    public void testPrefixQuerySupportsName() throws InterruptedException {
        createIndex("test1");
        ensureGreen();

        client().prepareIndex("test1").setId("1").setSource("title", "title1").get();
        refresh();
        indexRandomForConcurrentSearch("test1");

        var query = client().prepareSearch()
            .setQuery(QueryBuilders.prefixQuery("title", "title").queryName("prefix"))
            .setIncludeNamedQueriesScore(true);
        var searchResponse = query.get();
        assertHitCount(searchResponse, 1L);

        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("prefix"));
                assertThat(hit.getMatchedQueryScore("prefix"), equalTo(1f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }
    }

    public void testFuzzyQuerySupportsName() throws InterruptedException {
        createIndex("test1");
        ensureGreen();

        client().prepareIndex("test1").setId("1").setSource("title", "title1").get();
        refresh();
        indexRandomForConcurrentSearch("test1");

        SearchResponse searchResponse = client().prepareSearch()
            .setQuery(QueryBuilders.fuzzyQuery("title", "titel1").queryName("fuzzy"))
            .get();
        assertHitCount(searchResponse, 1L);

        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("fuzzy"));
                assertThat(hit.getMatchedQueryScore("fuzzy"), greaterThan(0f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }
    }

    public void testWildcardQuerySupportsName() throws InterruptedException {
        createIndex("test1");
        ensureGreen();

        client().prepareIndex("test1").setId("1").setSource("title", "title1").get();
        refresh();
        indexRandomForConcurrentSearch("test1");

        SearchResponse searchResponse = client().prepareSearch()
            .setQuery(QueryBuilders.wildcardQuery("title", "titl*").queryName("wildcard"))
            .setIncludeNamedQueriesScore(true)
            .get();
        assertHitCount(searchResponse, 1L);

        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("wildcard"));
                assertThat(hit.getMatchedQueryScore("wildcard"), equalTo(1f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }
    }

    public void testSpanFirstQuerySupportsName() throws InterruptedException {
        createIndex("test1");
        ensureGreen();

        client().prepareIndex("test1").setId("1").setSource("title", "title1 title2").get();
        refresh();
        indexRandomForConcurrentSearch("test1");

        SearchResponse searchResponse = client().prepareSearch()
            .setQuery(QueryBuilders.spanFirstQuery(QueryBuilders.spanTermQuery("title", "title1"), 10).queryName("span"))
            .get();
        assertHitCount(searchResponse, 1L);

        for (SearchHit hit : searchResponse.getHits()) {
            if (hit.getId().equals("1")) {
                assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                assertThat(hit.getMatchedQueriesAndScores(), hasKey("span"));
                assertThat(hit.getMatchedQueryScore("span"), greaterThan(0f));
            } else {
                fail("Unexpected document returned with id " + hit.getId());
            }
        }
    }

    /**
     * Test case for issue #4361: https://github.com/elastic/elasticsearch/issues/4361
     */
    public void testMatchedWithShould() throws Exception {
        createIndex("test");
        ensureGreen();

        client().prepareIndex("test").setId("1").setSource("content", "Lorem ipsum dolor sit amet").get();
        client().prepareIndex("test").setId("2").setSource("content", "consectetur adipisicing elit").get();
        refresh();
        indexRandomForConcurrentSearch("test");

        // Execute search at least two times to load it in cache
        int iter = scaledRandomIntBetween(2, 10);
        for (int i = 0; i < iter; i++) {
            SearchResponse searchResponse = client().prepareSearch()
                .setQuery(
                    boolQuery().minimumShouldMatch(1)
                        .should(queryStringQuery("dolor").queryName("dolor"))
                        .should(queryStringQuery("elit").queryName("elit"))
                )
                .setPreference("_primary")
                .get();

            assertHitCount(searchResponse, 2L);
            for (SearchHit hit : searchResponse.getHits()) {
                if (hit.getId().equals("1")) {
                    assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                    assertThat(hit.getMatchedQueriesAndScores(), hasKey("dolor"));
                    assertThat(hit.getMatchedQueryScore("dolor"), greaterThan(0f));
                } else if (hit.getId().equals("2")) {
                    assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
                    assertThat(hit.getMatchedQueriesAndScores(), hasKey("elit"));
                    assertThat(hit.getMatchedQueryScore("elit"), greaterThan(0f));
                } else {
                    fail("Unexpected document returned with id " + hit.getId());
                }
            }
        }
    }

    public void testMatchedWithWrapperQuery() throws Exception {
        createIndex("test");
        ensureGreen();

        client().prepareIndex("test").setId("1").setSource("content", "Lorem ipsum dolor sit amet").get();
        refresh();
        indexRandomForConcurrentSearch("test");

        MatchQueryBuilder matchQueryBuilder = matchQuery("content", "amet").queryName("abc");
        BytesReference matchBytes = XContentHelper.toXContent(matchQueryBuilder, MediaTypeRegistry.JSON, false);
        TermQueryBuilder termQueryBuilder = termQuery("content", "amet").queryName("abc");
        BytesReference termBytes = XContentHelper.toXContent(termQueryBuilder, MediaTypeRegistry.JSON, false);
        QueryBuilder[] queries = new QueryBuilder[] { wrapperQuery(matchBytes), constantScoreQuery(wrapperQuery(termBytes)) };
        for (QueryBuilder query : queries) {
            SearchResponse searchResponse = client().prepareSearch().setQuery(query).get();
            assertHitCount(searchResponse, 1L);
            SearchHit hit = searchResponse.getHits().getAt(0);
            assertThat(hit.getMatchedQueriesAndScores().size(), equalTo(1));
            assertThat(hit.getMatchedQueriesAndScores(), hasKey("abc"));
            assertThat(hit.getMatchedQueryScore("abc"), greaterThan(0f));
        }
    }
}
