/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

package org.opensearch.wlm.tracker;

import org.opensearch.core.tasks.resourcetracker.ResourceStats;
import org.opensearch.test.OpenSearchTestCase;
import org.opensearch.wlm.ResourceType;
import org.opensearch.wlm.WorkloadGroupTask;
import org.opensearch.wlm.tracker.ResourceUsageCalculatorTrackerServiceTests.TestClock;

import java.util.List;

import static org.opensearch.wlm.cancellation.WorkloadGroupTaskCancellationService.MIN_VALUE;
import static org.opensearch.wlm.tracker.CpuUsageCalculator.PROCESSOR_COUNT;
import static org.opensearch.wlm.tracker.MemoryUsageCalculator.HEAP_SIZE_BYTES;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class ResourceUsageCalculatorTests extends OpenSearchTestCase {

    public void testWorkloadGroupCpuUsage() {
        TestClock clock = new TestClock();
        long fastForwardTime = PROCESSOR_COUNT * 200L;
        clock.fastForwardBy(fastForwardTime);

        double expectedWorkloadGroupCpuUsage = 1.0 / PROCESSOR_COUNT;

        WorkloadGroupTask mockTask = createMockTaskWithResourceStats(WorkloadGroupTask.class, fastForwardTime, 200, 0, 123);
        when(mockTask.getElapsedTime()).thenReturn(fastForwardTime);
        double actualUsage = ResourceType.CPU.getResourceUsageCalculator().calculateResourceUsage(List.of(mockTask));
        assertEquals(expectedWorkloadGroupCpuUsage, actualUsage, MIN_VALUE);

        double taskResourceUsage = ResourceType.CPU.getResourceUsageCalculator().calculateTaskResourceUsage(mockTask);
        assertEquals(1.0, taskResourceUsage, MIN_VALUE);
    }

    public void testWorkloadGroupMemoryUsage() {
        WorkloadGroupTask mockTask = createMockTaskWithResourceStats(WorkloadGroupTask.class, 100, 200, 0, 123);
        double actualMemoryUsage = ResourceType.MEMORY.getResourceUsageCalculator().calculateResourceUsage(List.of(mockTask));
        double expectedMemoryUsage = 200.0 / HEAP_SIZE_BYTES;

        assertEquals(expectedMemoryUsage, actualMemoryUsage, MIN_VALUE);
        assertEquals(
            200.0 / HEAP_SIZE_BYTES,
            ResourceType.MEMORY.getResourceUsageCalculator().calculateTaskResourceUsage(mockTask),
            MIN_VALUE
        );
    }

    public static <T extends WorkloadGroupTask> T createMockTaskWithResourceStats(
        Class<T> type,
        long cpuUsage,
        long heapUsage,
        long startTimeNanos,
        long taskId
    ) {
        T task = mock(type);
        when(task.getTotalResourceUtilization(ResourceStats.CPU)).thenReturn(cpuUsage);
        when(task.getTotalResourceUtilization(ResourceStats.MEMORY)).thenReturn(heapUsage);
        when(task.getStartTimeNanos()).thenReturn(startTimeNanos);
        when(task.getId()).thenReturn(taskId);
        return task;
    }
}
