---
title: 'Claude Desktop'
description: 'Tutorial on connecting Pathway MCP Server to Claude Desktop for real-time data access'
date: '2025-08-19'
thumbnail: ''
tags: ['tutorial', 'engineering', 'claude']
keywords: ['MCP', 'Claude Desktop', 'real-time data', 'document store', 'statistics']
---

# Using Pathway MCP Server with Claude Desktop

## Introduction

The **[Model Context Protocol](https://modelcontextprotocol.io/docs/getting-started/intro) (MCP)** enables AI applications to interact with external data sources and tools in a standardized way.
By connecting **[Pathway MCP Server](/developers/user-guide/llm-xpack/pathway_mcp_server)** to **[Claude Desktop](https://claude.ai/download)**, you can provide Claude with all Pathway data processing capabilities, including real-time access to statistics and document stores.

This tutorial explains how to set up and use Pathway MCP Server with Claude Desktop, allowing Claude to retrieve real-time data, perform computations, and interact with your document store.

## Prerequisites

Before starting, ensure you have the following:

- **Claude Desktop** installed and updated to the latest version. You can find the latest version [here](https://claude.ai/download).
- **Python 3.9+** installed on your system.
- **Pathway** installed (`pip install pathway[xpack-llm]`, learn more [here](/developers/user-guide/introduction/installation)).
- **npm and npx** to be able to use the MCP server from Claude Desktop. You can find the installation guidelines [here](https://docs.npmjs.com/downloading-and-installing-node-js-and-npm).

::note
**Important**: Pathway MCP Server requires a Pathway license key. You can obtain a free license key [here](/get-license).
::

## Pathway MCP Servers

MCP servers act as intermediaries between AI applications (like Claude Desktop) and data sources and tools.
Pathway's MCP Server provides the real-time data processing capabilities of the Pathway engine to your AI applications, such as:

- Retrieving real-time statistics.
- Accessing and querying document stores.
- Performing computations on live data.

Each tool is explicit exposed, ensuring you maintain control over what Claude can access and modify.

You can learn more about Pathway MCP server in the [associated article](/developers/user-guide/llm-xpack/pathway_mcp_server).

## Setting Up Pathway MCP Server

### 1. Create a Basic MCP Server

Start by creating a simple MCP server that exposes a tool for Claude to interact with.
For example, let's expose a tool that returns a constant value.
Create the server in a Python file `pathway_mcp_server.py`:

```python [pathway_mcp_server.py]
import pathway as pw
from pathway.xpacks.llm.mcp_server import McpServable, McpServer, PathwayMcp

pw.set_license_key("demo-license-key-with-telemetry")

class EmptyRequestSchema(pw.Schema):
    pass

class ConstantValueTool(McpServable):
    def get_constant_value(self, input_from_client: pw.Table) -> pw.Table:
        return input_from_client.select(result=1)

    def register_mcp(self, server: McpServer):
        server.tool(
            "get_constant_value",
            request_handler=self.get_constant_value,
            schema=EmptyRequestSchema,
        )

function_to_serve = ConstantValueTool()
pathway_mcp_server = PathwayMcp(
    name="Pathway MCP Server",
    transport="streamable-http",
    host="localhost",
    port=8123,
    serve=[function_to_serve],
)
pw.run()
```

Other Pathway MCP server examples are available [here](/developers/user-guide/llm-xpack/pathway_mcp_server).

### 2. Run the MCP Server
Execute the script to start the MCP server:

```bash
python pathway_mcp_server.py
```

The server will be available at `http://localhost:8123/mcp/`.

## Configuring Claude Desktop

To be able to use your Pathway MCP Server in Claude Desktop do as follows:

- Open Claude Desktop.
- Click on the Claude menu in the top menu bar, **"File"**, and **"Settings..."**.

::article-img
---
src: '/assets/content/documentation/claude-desktop/claude-desktop-settings.png'
provider: 's3'
alt: 'Select File then Settings.'
class: 'mx-auto mt-16 mb-8'
---
::


- Navigate to the **"Developer"** tab, then click on **"Edit Config"**.


::article-img
---
src: '/assets/content/documentation/claude-desktop/claude-desktop-config.png'
provider: 's3'
alt: 'Select Developer then Edit Config'
class: 'mx-auto mt-16 mb-8'
---
::

- Select `claude_desktop_config.json` and edit it as follows:

```json
{
  "mcpServers": {
    "pathway": {
      "command": "npx",
      "args": [
        "mcp-remote",
        "http://localhost:8123/mcp/",
        "--transport",
        "http-first"
      ]
    }
  }
}
```

Reload Claude Desktop configuration with Ctrl+R and Claude Desktop will connect to your Pathway MCP Server.


If it doesn't work, check that Node.js appears in the "Detected Tools" section in "Extensions", "Advanced Settings".
If not, then you need to (re)install npm and npx.
You may need to restart your computer and Claude Desktop once it is done, especially on Windows.

::article-img
---
src: '/assets/content/documentation/claude-desktop/claude-desktop-advanced-settings.png'
provider: 's3'
alt: 'Go to advanced settings from Extensions'
class: 'mx-auto mt-16 mb-8'
---
::

::article-img
---
src: '/assets/content/documentation/claude-desktop/claude-desktop-nodejs.png'
provider: 's3'
alt: 'Check that Node.js is listed'
class: 'mx-auto mt-16 mb-8'
---
::

Once this is done, the server should appear below "Local MCP Server" when in the "Developer" section.


::article-img
---
src: '/assets/content/documentation/claude-desktop/claude-desktop-running-server.png'
provider: 's3'
alt: 'If everything went well, Pathway MCP server should be listed in the Developer section.'
class: 'mx-auto mt-16 mb-8'
---
::

## Using Pathway MCP Server with Claude
You can now ask Claude to use the tools exposed by the Pathway MCP Server. For example:

- **"What is the constant value?"** Claude will call the `get_constant_value` tool and return the result.
- **"Can you provide real-time statistics about the data?"** If you have exposed a statistics tool, Claude will retrieve and display the statistics.

Claude Desktop will ask you to validate the use of tools, ensuring that you control the usage.
It is useful for expensive tools, such as tools relying on LLMs.

Pathway should be listed as a tool in the chat:

::article-img
---
src: '/assets/content/documentation/claude-desktop/claude-desktop-pathway-server.png'
provider: 's3'
alt: 'Pathway should be listed as a tool in the chat.'
class: 'mx-auto mt-16 mb-8'
---
::


## Testing other tools

To use other tools, you can implement some presented in the [Pathway MCP article](/developers/user-guide/llm-xpack/pathway_mcp_server).
Restart your MCP server, and reload Claude Desktop: the configuration stay the same, you have nothing else to do!

## Conclusion
By connecting Pathway MCP Server to Claude Desktop, you enable Claude to access real-time data, perform computations, and interact with your Pathway tables.
You can easily set up a RAG pipeline by [exposing Pathway Document Store](/developers/user-guide/llm-xpack/pathway_mcp_server#exposing-pathway-document-store).
This integration enhances Claude's capabilities, allowing it to provide accurate, context-aware responses based on up-to-date information.
As real-time data processing becomes increasingly important, tools like Pathway MCP Server will be essential for effective data handling and decision-making in AI applications.
