# Copyright © 2026 Pathway

import datetime

import pandas as pd
import pytest

import pathway as pw
from pathway.debug import table_from_pandas
from pathway.tests.utils import assert_table_equality


@pytest.mark.parametrize(
    "timezone, dates",
    [
        ("Africa/Cairo", ["2010-04-29 23:59:59"]),
        (
            "Africa/Casablanca",
            ["2010-05-01 23:59:59", "2018-06-17 01:59:59", "2019-06-09 01:59:59"],
        ),
        (
            "Africa/Ceuta",
            ["2010-03-28 01:59:59", "2022-03-27 01:59:59", "2023-03-26 01:59:59"],
        ),
        ("Africa/Tripoli", ["2013-03-29 00:59:59"]),
        (
            "Africa/Windhoek",
            ["2010-09-05 01:59:59", "2016-09-04 01:59:59", "2017-09-03 01:59:59"],
        ),
        (
            "America/Adak",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Anchorage",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        ("America/Araguaina", ["2012-10-20 23:59:59"]),
        (
            "America/Asuncion",
            ["2010-10-02 23:59:59", "2022-10-01 23:59:59", "2023-09-30 23:59:59"],
        ),
        ("America/Bahia", ["2011-10-15 23:59:59"]),
        (
            "America/Bahia_Banderas",
            ["2010-04-04 01:59:59", "2021-04-04 01:59:59", "2022-04-03 01:59:59"],
        ),
        (
            "America/Boise",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Campo_Grande",
            ["2010-10-16 23:59:59", "2017-10-14 23:59:59", "2018-11-03 23:59:59"],
        ),
        (
            "America/Cancun",
            ["2010-04-04 01:59:59", "2014-04-06 01:59:59", "2015-02-01 01:59:59"],
        ),
        ("America/Caracas", ["2016-05-01 02:29:59"]),
        (
            "America/Chicago",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Chihuahua",
            ["2010-04-04 01:59:59", "2021-04-04 01:59:59", "2022-04-03 01:59:59"],
        ),
        (
            "America/Dawson",
            ["2010-03-14 01:59:59", "2019-03-10 01:59:59", "2020-03-08 01:59:59"],
        ),
        (
            "America/Detroit",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Ensenada",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Fort_Nelson",
            ["2010-03-14 01:59:59", "2014-03-09 01:59:59", "2015-03-08 01:59:59"],
        ),
        (
            "America/Glace_Bay",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Goose_Bay",
            ["2010-03-14 00:00:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Grand_Turk",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Havana",
            ["2010-03-13 23:59:59", "2022-03-12 23:59:59", "2023-03-11 23:59:59"],
        ),
        (
            "America/Merida",
            ["2010-04-04 01:59:59", "2021-04-04 01:59:59", "2022-04-03 01:59:59"],
        ),
        (
            "America/Metlakatla",
            ["2016-03-13 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Miquelon",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Montevideo",
            ["2010-10-03 01:59:59", "2013-10-06 01:59:59", "2014-10-05 01:59:59"],
        ),
        (
            "America/North_Dakota/Beulah",
            ["2010-03-14 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Ojinaga",
            ["2010-03-14 01:59:59", "2021-03-14 01:59:59", "2022-03-13 01:59:59"],
        ),
        (
            "America/Port-au-Prince",
            ["2012-03-11 01:59:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "America/Punta_Arenas",
            ["2010-10-09 23:59:59", "2014-09-06 23:59:59", "2016-08-13 23:59:59"],
        ),
        (
            "America/Santiago",
            ["2010-10-09 23:59:59", "2022-09-10 23:59:59", "2023-09-02 23:59:59"],
        ),
        (
            "America/Sao_Paulo",
            ["2010-10-16 23:59:59", "2017-10-14 23:59:59", "2018-11-03 23:59:59"],
        ),
        (
            "America/St_Johns",
            ["2010-03-14 00:00:59", "2022-03-13 01:59:59", "2023-03-12 01:59:59"],
        ),
        (
            "Antarctica/Casey",
            ["2011-10-28 01:59:59", "2019-10-04 02:59:59", "2020-10-04 00:00:59"],
        ),
        ("Antarctica/Davis", ["2010-03-11 00:59:59", "2012-02-22 00:59:59"]),
        (
            "Antarctica/Macquarie",
            ["2011-10-02 01:59:59", "2022-10-02 01:59:59", "2023-10-01 01:59:59"],
        ),
        (
            "Antarctica/McMurdo",
            ["2010-09-26 01:59:59", "2022-09-25 01:59:59", "2023-09-24 01:59:59"],
        ),
        (
            "Asia/Amman",
            ["2010-03-25 23:59:59", "2021-03-25 23:59:59", "2022-02-24 23:59:59"],
        ),
        ("Asia/Anadyr", ["2011-03-27 01:59:59"]),
        (
            "Asia/Baku",
            ["2010-03-28 03:59:59", "2014-03-30 03:59:59", "2015-03-29 03:59:59"],
        ),
        (
            "Asia/Barnaul",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2016-03-27 01:59:59"],
        ),
        (
            "Asia/Beirut",
            ["2010-03-27 23:59:59", "2022-03-26 23:59:59", "2023-03-25 23:59:59"],
        ),
        (
            "Asia/Chita",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2016-03-27 01:59:59"],
        ),
        ("Asia/Choibalsan", ["2015-03-28 01:59:59", "2016-03-26 01:59:59"]),
        (
            "Asia/Damascus",
            ["2010-04-01 23:59:59", "2021-03-25 23:59:59", "2022-03-24 23:59:59"],
        ),
        (
            "Asia/Famagusta",
            ["2010-03-28 02:59:59", "2022-03-27 02:59:59", "2023-03-26 02:59:59"],
        ),
        (
            "Asia/Gaza",
            ["2010-03-27 00:00:59", "2022-03-26 23:59:59", "2023-03-25 01:59:59"],
        ),
        (
            "Asia/Hebron",
            ["2010-03-25 23:59:59", "2022-03-26 23:59:59", "2023-03-25 01:59:59"],
        ),
        ("Asia/Hovd", ["2015-03-28 01:59:59", "2016-03-26 01:59:59"]),
        ("Asia/Irkutsk", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        (
            "Asia/Istanbul",
            ["2010-03-28 02:59:59", "2015-03-29 02:59:59", "2016-03-27 02:59:59"],
        ),
        (
            "Asia/Jerusalem",
            ["2010-03-26 01:59:59", "2022-03-25 01:59:59", "2023-03-24 01:59:59"],
        ),
        ("Asia/Khandyga", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        ("Asia/Krasnoyarsk", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        (
            "Asia/Magadan",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2016-04-24 01:59:59"],
        ),
        ("Asia/Novokuznetsk", ["2011-03-27 01:59:59"]),
        (
            "Asia/Novosibirsk",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2016-07-24 01:59:59"],
        ),
        ("Asia/Omsk", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        ("Asia/Pyongyang", ["2018-05-04 23:29:59"]),
        (
            "Asia/Sakhalin",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2016-03-27 01:59:59"],
        ),
        ("Asia/Srednekolymsk", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        (
            "Asia/Tehran",
            ["2010-03-21 23:59:59", "2021-03-21 23:59:59", "2022-03-21 23:59:59"],
        ),
        (
            "Asia/Tomsk",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2016-05-29 01:59:59"],
        ),
        ("Asia/Yakutsk", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        ("Asia/Yekaterinburg", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        ("Asia/Yerevan", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        ("Atlantic/Stanley", ["2010-09-05 01:59:59"]),
        (
            "Australia/ACT",
            ["2010-10-03 01:59:59", "2022-10-02 01:59:59", "2023-10-01 01:59:59"],
        ),
        (
            "Australia/Adelaide",
            ["2010-10-03 01:59:59", "2022-10-02 01:59:59", "2023-10-01 01:59:59"],
        ),
        (
            "Australia/LHI",
            ["2010-10-03 01:59:59", "2022-10-02 01:59:59", "2023-10-01 01:59:59"],
        ),
        (
            "Europe/Astrakhan",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2016-03-27 01:59:59"],
        ),
        (
            "Europe/Chisinau",
            ["2010-03-28 01:59:59", "2022-03-27 01:59:59", "2023-03-26 01:59:59"],
        ),
        ("Europe/Kaliningrad", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        ("Europe/Kirov", ["2010-03-28 01:59:59", "2011-03-27 01:59:59"]),
        ("Europe/Samara", ["2011-03-27 01:59:59"]),
        (
            "Europe/Saratov",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2016-12-04 01:59:59"],
        ),
        (
            "Europe/Simferopol",
            ["2010-03-28 02:59:59", "2013-03-31 02:59:59", "2014-03-30 01:59:59"],
        ),
        (
            "Europe/Volgograd",
            ["2010-03-28 01:59:59", "2011-03-27 01:59:59", "2018-10-28 01:59:59"],
        ),
        (
            "Pacific/Apia",
            ["2010-09-25 23:59:59", "2019-09-29 02:59:59", "2020-09-27 02:59:59"],
        ),
        ("Pacific/Bougainville", ["2014-12-28 01:59:59"]),
        (
            "Pacific/Fiji",
            ["2010-10-24 01:59:59", "2019-11-10 01:59:59", "2020-12-20 01:59:59"],
        ),
        (
            "Pacific/Norfolk",
            ["2019-10-06 01:59:59", "2022-10-02 01:59:59", "2023-10-01 01:59:59"],
        ),
        ("Pacific/Tongatapu", ["2016-11-06 01:59:59"]),
    ],
)
def test_non_existing_date_time_conversion(timezone: str, dates: list[str]) -> None:
    fmt = "%Y-%m-%d %H:%M:%S"
    df = pd.DataFrame({"date_string": dates})
    table = table_from_pandas(df)
    table_naive = table.select(t=table.date_string.dt.strptime(fmt))
    table_naive_non_existing_time = table_naive.select(
        t=pw.this.t + datetime.timedelta(seconds=1)
    )
    table_non_existing_to_utc = table_naive_non_existing_time.select(
        t=pw.this.t.dt.to_utc(timezone)
    )
    table_existing_utc = table_naive.select(
        t=pw.this.t.dt.to_utc(timezone) + datetime.timedelta(seconds=1)
    )
    try:
        assert_table_equality(table_non_existing_to_utc, table_existing_utc)
    except Exception:  # FIXME: make conversions work in all cases
        pytest.xfail(reason="Not all conversions working now")
