---
title: 'Sending alerts to Slack'
description: 'Tutorial on the connector for sending alerts to Slack'
date: '2024-02-16'
tags: ['tutorial', 'connectors']
keywords: ['connector', 'Slack', 'alerts', 'output', 'streaming']
tech_icons: ["mdi:slack"]
---

# Sending alerts to Slack

This tutorial will guide you through connecting Pathway to Slack and sending alerts to a specific channel.

If you need to send some alerts to Slack based on your Pathway pipeline, you should use a [`pw.io.slack.send_alerts`](/developers/api-docs/pathway-io/slack#pathway.io.slack.send_alerts) function. It allows for sending messages from a Pathway table to a specific channel in Slack. This tutorial will guide you on how to use it.

## Setting up Slack authentication

To send messages to Slack, Pathway uses the [Slack web API](https://api.slack.com/web). To use it, you need to have a token for authenticating to Slack. You can read how to get it in the [Slack documentation](https://api.slack.com/tutorials/tracks/getting-a-token). For `pw.io.slack.send_alerts`, your token needs to have a [`chat:write` scope](https://api.slack.com/scopes/chat:write).

## Connecting to Slack with Pathway

If you obtained your Slack token, you are ready to connect your Pathway application to Slack with [`pw.io.slack.send_alerts`](/developers/api-docs/pathway-io/slack#pathway.io.slack.send_alerts).

`pw.io.slack.send_alerts` takes 3 arguments:
- `alerts`, a [`ColumnReference`](/developers/api-docs/pathway#pathway.ColumnReference) with the content of the alerts you want to send to Slack.
- `slack_channel_id`, a string with the [ID of the channel](https://api.slack.com/methods/chat.postMessage#channels) you want to send the alerts to.
- `slack_token`, your token for authenticating to Slack.

To create a simple example of sending Slack alerts, you need a Table with a column with the content to be sent to Slack. This tutorial uses a simple table defined with [`pw.debug.table_from_markdown`](/developers/api-docs/debug#pathway.debug.table_from_markdown). In the following example, `slack_channel_id` and `slack_token` are obtained from environmental variables, respectively `SLACK_CHANNEL_ID` and `SLACK_TOKEN`. Make sure to set them before running the program.

```python
import os
import pathway as pw

slack_channel_id = os.environ["SLACK_CHANNEL_ID"]
slack_token = os.environ["SLACK_TOKEN"]

t = pw.debug.table_from_markdown("""
messages
Hello_Slack
This_is_Slack_alert
""")
pw.io.slack.send_alerts(t.messages, slack_channel_id, slack_token)
pw.run()
```

When you run it, you will get the alerts in your Slack channel.

::article-img
---
src: '/assets/content/documentation/slack-alerts-tutorial.png'
alt: 'Alerts sent to Slack channel'
height: '96'
width: '269'
quality: '100'
class: 'mx-auto'
---
::


Note, that the values of the column `messages` in the above example do not have spaces. It is a restriction of `pw.debug.table_from_markdown` which uses spaces to separate columns. Any regular string works with the other connectors.

If you want to see more examples with `pw.io.slack.send_alerts` you can check the [`drive_alert`](https://github.com/pathwaycom/llm-app/tree/main/templates/drive_alert) example in the llm-app.
