"""Tests for the MCP (Model Context Protocol) server implementation."""

from __future__ import annotations

import base64
import re
from datetime import timezone
from pathlib import Path
from typing import Any
from unittest.mock import AsyncMock, patch

import pytest
from inline_snapshot import snapshot

from pydantic_ai import (
    BinaryContent,
    BinaryImage,
    ModelRequest,
    ModelResponse,
    RetryPromptPart,
    TextPart,
    ThinkingPart,
    ToolCallPart,
    ToolReturnPart,
    UserPromptPart,
)
from pydantic_ai.agent import Agent
from pydantic_ai.exceptions import (
    ModelRetry,
    UnexpectedModelBehavior,
    UserError,
)
from pydantic_ai.mcp import (
    MCPError,
    MCPServerStreamableHTTP,
    Resource,
    ResourceAnnotations,
    ResourceTemplate,
    ServerCapabilities,
    load_mcp_servers,
)
from pydantic_ai.models import Model, cached_async_http_client
from pydantic_ai.models.test import TestModel
from pydantic_ai.tools import RunContext
from pydantic_ai.usage import RequestUsage, RunUsage

from .conftest import IsDatetime, IsInstance, IsNow, IsStr, try_import

with try_import() as imports_successful:
    from mcp import ErrorData, McpError, SamplingMessage
    from mcp.client.session import ClientSession
    from mcp.shared.context import RequestContext
    from mcp.types import (
        CreateMessageRequestParams,
        ElicitRequestParams,
        ElicitResult,
        ImageContent,
        Implementation,
        TextContent,
    )

    from pydantic_ai._mcp import map_from_mcp_params, map_from_model_response, map_from_pai_messages
    from pydantic_ai.mcp import CallToolFunc, MCPServerSSE, MCPServerStdio, ToolResult
    from pydantic_ai.models.google import GoogleModel
    from pydantic_ai.models.openai import OpenAIChatModel
    from pydantic_ai.providers.google import GoogleProvider
    from pydantic_ai.providers.openai import OpenAIProvider

pytestmark = [
    pytest.mark.skipif(not imports_successful(), reason='mcp and openai not installed'),
    pytest.mark.anyio,
    pytest.mark.vcr,
]


@pytest.fixture
def mcp_server() -> MCPServerStdio:
    return MCPServerStdio('python', ['-m', 'tests.mcp_server'])


@pytest.fixture
def model(openai_api_key: str) -> Model:
    return OpenAIChatModel('gpt-4o', provider=OpenAIProvider(api_key=openai_api_key))


@pytest.fixture
def agent(model: Model, mcp_server: MCPServerStdio) -> Agent:
    return Agent(model, toolsets=[mcp_server])


@pytest.fixture
def run_context(model: Model) -> RunContext[int]:
    return RunContext(deps=0, model=model, usage=RunUsage())


async def test_stdio_server(run_context: RunContext[int]):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        tools = [tool.tool_def for tool in (await server.get_tools(run_context)).values()]
        assert len(tools) == snapshot(20)
        assert tools[0].name == 'celsius_to_fahrenheit'
        assert isinstance(tools[0].description, str)
        assert tools[0].description.startswith('Convert Celsius to Fahrenheit.')

        # Test calling the temperature conversion tool
        result = await server.direct_call_tool('celsius_to_fahrenheit', {'celsius': 0})
        assert result == snapshot(32.0)


async def test_reentrant_context_manager():
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        async with server:
            pass


async def test_context_manager_initialization_error() -> None:
    """Test if streams are closed if client fails to initialize."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    from mcp.client.session import ClientSession

    with patch.object(ClientSession, 'initialize', side_effect=Exception):
        with pytest.raises(Exception):
            async with server:
                pass

    assert server._read_stream._closed  # pyright: ignore[reportPrivateUsage]
    assert server._write_stream._closed  # pyright: ignore[reportPrivateUsage]


async def test_aexit_called_more_times_than_aenter():
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])

    with pytest.raises(ValueError, match='MCPServer.__aexit__ called more times than __aenter__'):
        await server.__aexit__(None, None, None)

    async with server:
        pass  # This will call __aenter__ and __aexit__ once each

    with pytest.raises(ValueError, match='MCPServer.__aexit__ called more times than __aenter__'):
        await server.__aexit__(None, None, None)


async def test_stdio_server_with_tool_prefix(run_context: RunContext[int]):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], tool_prefix='foo')
    async with server:
        tools = await server.get_tools(run_context)
        assert all(name.startswith('foo_') for name in tools.keys())

        result = await server.call_tool(
            'foo_celsius_to_fahrenheit', {'celsius': 0}, run_context, tools['foo_celsius_to_fahrenheit']
        )
        assert result == snapshot(32.0)


async def test_stdio_server_with_cwd(run_context: RunContext[int]):
    test_dir = Path(__file__).parent
    server = MCPServerStdio('python', ['mcp_server.py'], cwd=test_dir)
    async with server:
        tools = await server.get_tools(run_context)
        assert len(tools) == snapshot(20)


async def test_process_tool_call(run_context: RunContext[int]) -> int:
    called: bool = False

    async def process_tool_call(
        ctx: RunContext[int],
        call_tool: CallToolFunc,
        name: str,
        tool_args: dict[str, Any],
    ) -> ToolResult:
        """A process_tool_call that sets a flag and sends deps as metadata."""
        nonlocal called
        called = True
        return await call_tool(name, tool_args, {'deps': ctx.deps})

    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], process_tool_call=process_tool_call)
    async with server:
        agent = Agent(deps_type=int, model=TestModel(call_tools=['echo_deps']), toolsets=[server])
        result = await agent.run('Echo with deps set to 42', deps=42)
        assert result.output == snapshot('{"echo_deps":{"echo":"This is an echo message","deps":42}}')
        assert called, 'process_tool_call should have been called'


def test_sse_server():
    sse_server = MCPServerSSE(url='http://localhost:8000/sse')
    assert sse_server.url == 'http://localhost:8000/sse'
    assert sse_server.log_level is None


def test_sse_server_with_header_and_timeout():
    with pytest.warns(DeprecationWarning, match="'sse_read_timeout' is deprecated, use 'read_timeout' instead."):
        sse_server = MCPServerSSE(
            url='http://localhost:8000/sse',
            headers={'my-custom-header': 'my-header-value'},
            timeout=10,
            sse_read_timeout=100,
            log_level='info',
        )
    assert sse_server.url == 'http://localhost:8000/sse'
    assert sse_server.headers is not None and sse_server.headers['my-custom-header'] == 'my-header-value'
    assert sse_server.timeout == 10
    assert sse_server.read_timeout == 100
    assert sse_server.log_level == 'info'


def test_sse_server_conflicting_timeout_params():
    with pytest.raises(TypeError, match="'read_timeout' and 'sse_read_timeout' cannot be set at the same time."):
        MCPServerSSE(
            url='http://localhost:8000/sse',
            read_timeout=50,
            sse_read_timeout=100,
        )


async def test_agent_with_stdio_server(allow_model_requests: None, agent: Agent):
    async with agent:
        result = await agent.run('What is 0 degrees Celsius in Fahrenheit?')
        assert result.output == snapshot('0 degrees Celsius is equal to 32 degrees Fahrenheit.')
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='What is 0 degrees Celsius in Fahrenheit?',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='celsius_to_fahrenheit',
                            args='{"celsius":0}',
                            tool_call_id='call_QssdxTGkPblTYHmyVES1tKBj',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=195,
                        output_tokens=19,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRlnvvqIPFofAtKqtQKMWZkgXhzlT',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='celsius_to_fahrenheit',
                            content=32.0,
                            tool_call_id='call_QssdxTGkPblTYHmyVES1tKBj',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[TextPart(content='0 degrees Celsius is equal to 32 degrees Fahrenheit.')],
                    usage=RequestUsage(
                        input_tokens=227,
                        output_tokens=13,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRlnyjUo5wlyqvdNdM5I8vIWjo1qF',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_agent_with_conflict_tool_name(agent: Agent):
    @agent.tool_plain
    def get_none() -> None:  # pragma: no cover
        """Return nothing"""
        return None

    async with agent:
        with pytest.raises(
            UserError,
            match=re.escape(
                "MCPServerStdio(command='python', args=['-m', 'tests.mcp_server']) defines a tool whose name conflicts with existing tool from the agent: 'get_none'. Set the `tool_prefix` attribute to avoid name conflicts."
            ),
        ):
            await agent.run('Get me a conflict')


async def test_agent_with_prefix_tool_name(openai_api_key: str):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], tool_prefix='foo')
    model = OpenAIChatModel('gpt-4o', provider=OpenAIProvider(api_key=openai_api_key))
    agent = Agent(
        model,
        toolsets=[server],
    )

    @agent.tool_plain
    def get_none() -> None:  # pragma: no cover
        """Return nothing"""
        return None

    async with agent:
        # This means that we passed the _prepare_request_parameters check and there is no conflict in the tool name
        with pytest.raises(RuntimeError, match='Model requests are not allowed, since ALLOW_MODEL_REQUESTS is False'):
            await agent.run('No conflict')


async def test_agent_with_server_not_running(agent: Agent, allow_model_requests: None):
    result = await agent.run('What is 0 degrees Celsius in Fahrenheit?')
    assert result.output == snapshot('0 degrees Celsius is 32.0 degrees Fahrenheit.')


async def test_log_level_unset(run_context: RunContext[int]):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    assert server.log_level is None
    async with server:
        result = await server.direct_call_tool('get_log_level', {})
        assert result == snapshot('unset')


async def test_stdio_server_list_resources(run_context: RunContext[int]):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        resources = await server.list_resources()
        assert resources == snapshot(
            [
                Resource(name='kiwi_resource', description='', mime_type='image/jpeg', uri='resource://kiwi.jpg'),
                Resource(name='marcelo_resource', description='', mime_type='audio/mpeg', uri='resource://marcelo.mp3'),
                Resource(
                    name='product_name_resource',
                    description='',
                    mime_type='text/plain',
                    annotations=ResourceAnnotations(audience=['user', 'assistant'], priority=0.5),
                    uri='resource://product_name.txt',
                ),
            ]
        )


async def test_stdio_server_list_resource_templates(run_context: RunContext[int]):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        resource_templates = await server.list_resource_templates()
        assert resource_templates == snapshot(
            [
                ResourceTemplate(
                    name='greeting_resource_template',
                    description='Dynamic greeting resource template.',
                    mime_type='text/plain',
                    uri_template='resource://greeting/{name}',
                )
            ]
        )


async def test_log_level_set(run_context: RunContext[int]):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], log_level='info')
    assert server.log_level == 'info'
    async with server:
        result = await server.direct_call_tool('get_log_level', {})
        assert result == snapshot('info')


async def test_tool_returning_str(allow_model_requests: None, agent: Agent):
    async with agent:
        result = await agent.run('What is the weather in Mexico City?')
        assert result.output == snapshot(
            'The weather in Mexico City is currently sunny with a temperature of 26 degrees Celsius.'
        )
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='What is the weather in Mexico City?',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_weather_forecast',
                            args='{"location":"Mexico City"}',
                            tool_call_id='call_m9goNwaHBbU926w47V7RtWPt',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=194,
                        output_tokens=18,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRlo3e1Ud2lnvkddMilmwC7LAemiy',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_weather_forecast',
                            content='The weather in Mexico City is sunny and 26 degrees Celsius.',
                            tool_call_id='call_m9goNwaHBbU926w47V7RtWPt',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        TextPart(
                            content='The weather in Mexico City is currently sunny with a temperature of 26 degrees Celsius.'
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=234,
                        output_tokens=19,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRlo41LxqBYgGKWgGrQn67fQacOLp',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_text_resource(allow_model_requests: None, agent: Agent):
    async with agent:
        result = await agent.run('Get me the product name')
        assert result.output == snapshot('The product name is "Pydantic AI".')
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me the product name',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_product_name',
                            args='{}',
                            tool_call_id='call_LaiWltzI39sdquflqeuF0EyE',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=200,
                        output_tokens=12,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRmhyweJVYonarb7s9ckIMSHf2vHo',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_product_name',
                            content='Pydantic AI',
                            tool_call_id='call_LaiWltzI39sdquflqeuF0EyE',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[TextPart(content='The product name is "Pydantic AI".')],
                    usage=RequestUsage(
                        input_tokens=224,
                        output_tokens=12,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRmhzqXFObpYwSzREMpJvX9kbDikR',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_text_resource_link(allow_model_requests: None, agent: Agent):
    async with agent:
        result = await agent.run('Get me the product name via get_product_name_link')
        assert result.output == snapshot('The product name is "Pydantic AI".')
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me the product name via get_product_name_link',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_product_name_link',
                            args='{}',
                            tool_call_id='call_qi5GtBeIEyT7Y3yJvVFIi062',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=305,
                        output_tokens=12,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BwdHSFe0EykAOpf0LWZzsWAodIQzb',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_product_name_link',
                            content='Pydantic AI\n',
                            tool_call_id='call_qi5GtBeIEyT7Y3yJvVFIi062',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[TextPart(content='The product name is "Pydantic AI".')],
                    usage=RequestUsage(
                        input_tokens=332,
                        output_tokens=11,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BwdHTIlBZWzXJPBR8VTOdC4O57ZQA',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_image_resource(allow_model_requests: None, agent: Agent, image_content: BinaryContent):
    async with agent:
        result = await agent.run('Get me the image resource')
        assert result.output == snapshot(
            'This is an image of a sliced kiwi with a vibrant green interior and black seeds.'
        )
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me the image resource',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_image_resource',
                            args='{}',
                            tool_call_id='call_nFsDHYDZigO0rOHqmChZ3pmt',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=191,
                        output_tokens=12,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRlo7KYJVXuNZ5lLLdYcKZDsX2CHb',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_image_resource',
                            content='See file 241a70',
                            tool_call_id='call_nFsDHYDZigO0rOHqmChZ3pmt',
                            timestamp=IsDatetime(),
                        ),
                        UserPromptPart(content=['This is file 241a70:', image_content], timestamp=IsDatetime()),
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        TextPart(
                            content='This is an image of a sliced kiwi with a vibrant green interior and black seeds.'
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=1332,
                        output_tokens=19,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRloBGHh27w3fQKwxq4fX2cPuZJa9',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_image_resource_link(
    allow_model_requests: None, agent: Agent, image_content: BinaryContent
):
    async with agent:
        result = await agent.run('Get me the image resource via get_image_resource_link')
        assert result.output == snapshot(
            'This is an image of a sliced kiwi fruit. It shows the green, seed-speckled interior with fuzzy brown skin around the edges.'
        )
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me the image resource via get_image_resource_link',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_image_resource_link',
                            args='{}',
                            tool_call_id='call_eVFgn54V9Nuh8Y4zvuzkYjUp',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=305,
                        output_tokens=12,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BwdHygYePH1mZgHo2Xxzib0Y7sId7',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_image_resource_link',
                            content='See file 241a70',
                            tool_call_id='call_eVFgn54V9Nuh8Y4zvuzkYjUp',
                            timestamp=IsDatetime(),
                        ),
                        UserPromptPart(content=['This is file 241a70:', image_content], timestamp=IsDatetime()),
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        TextPart(
                            content='This is an image of a sliced kiwi fruit. It shows the green, seed-speckled interior with fuzzy brown skin around the edges.'
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=1452,
                        output_tokens=29,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BwdI2D2r9dvqq3pbsA0qgwKDEdTtD',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_audio_resource(
    allow_model_requests: None, agent: Agent, audio_content: BinaryContent, gemini_api_key: str
):
    model = GoogleModel('gemini-2.5-pro-preview-03-25', provider=GoogleProvider(api_key=gemini_api_key))
    async with agent:
        result = await agent.run("What's the content of the audio resource?", model=model)
        assert result.output == snapshot('The audio resource contains a voice saying "Hello, my name is Marcelo."')
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[UserPromptPart(content="What's the content of the audio resource?", timestamp=IsDatetime())],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[ToolCallPart(tool_name='get_audio_resource', args={}, tool_call_id=IsStr())],
                    usage=RequestUsage(
                        input_tokens=383, output_tokens=137, details={'thoughts_tokens': 125, 'text_prompt_tokens': 383}
                    ),
                    model_name='models/gemini-2.5-pro-preview-05-06',
                    timestamp=IsDatetime(),
                    provider_name='google-gla',
                    provider_url='https://generativelanguage.googleapis.com/',
                    provider_details={'finish_reason': 'STOP'},
                    provider_response_id=IsStr(),
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_audio_resource',
                            content='See file 2d36ae',
                            tool_call_id=IsStr(),
                            timestamp=IsDatetime(),
                        ),
                        UserPromptPart(content=['This is file 2d36ae:', audio_content], timestamp=IsDatetime()),
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[TextPart(content='The audio resource contains a voice saying "Hello, my name is Marcelo."')],
                    usage=RequestUsage(
                        input_tokens=575,
                        output_tokens=15,
                        input_audio_tokens=144,
                        details={'text_prompt_tokens': 431, 'audio_prompt_tokens': 144},
                    ),
                    model_name='models/gemini-2.5-pro-preview-05-06',
                    timestamp=IsDatetime(),
                    provider_name='google-gla',
                    provider_url='https://generativelanguage.googleapis.com/',
                    provider_details={'finish_reason': 'STOP'},
                    provider_response_id=IsStr(),
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_audio_resource_link(
    allow_model_requests: None, agent: Agent, audio_content: BinaryContent, gemini_api_key: str
):
    model = GoogleModel('gemini-2.5-pro', provider=GoogleProvider(api_key=gemini_api_key))
    async with agent:
        result = await agent.run("What's the content of the audio resource via get_audio_resource_link?", model=model)
        assert result.output == snapshot('00:05')
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content="What's the content of the audio resource via get_audio_resource_link?",
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_audio_resource_link',
                            args={},
                            tool_call_id=IsStr(),
                            provider_details={'thought_signature': IsStr()},
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=605, output_tokens=168, details={'thoughts_tokens': 154, 'text_prompt_tokens': 605}
                    ),
                    model_name='gemini-2.5-pro',
                    timestamp=IsDatetime(),
                    provider_name='google-gla',
                    provider_url='https://generativelanguage.googleapis.com/',
                    provider_details={'finish_reason': 'STOP'},
                    provider_response_id='Pe_BaJGqOKSdz7IP0NqogA8',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_audio_resource_link',
                            content='See file 2d36ae',
                            tool_call_id=IsStr(),
                            timestamp=IsDatetime(),
                        ),
                        UserPromptPart(
                            content=[
                                'This is file 2d36ae:',
                                audio_content,
                            ],
                            timestamp=IsDatetime(),
                        ),
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[TextPart(content='00:05')],
                    usage=RequestUsage(
                        input_tokens=801,
                        output_tokens=5,
                        input_audio_tokens=144,
                        details={'text_prompt_tokens': 657, 'audio_prompt_tokens': 144},
                    ),
                    model_name='gemini-2.5-pro',
                    timestamp=IsDatetime(),
                    provider_name='google-gla',
                    provider_url='https://generativelanguage.googleapis.com/',
                    provider_details={'finish_reason': 'STOP'},
                    provider_response_id='QO_BaLC6AozQz7IPh5Kj4Q4',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_image(allow_model_requests: None, agent: Agent, image_content: BinaryContent):
    async with agent:
        result = await agent.run('Get me an image')
        assert result.output == snapshot(
            "Here's an image of a kiwi fruit cut in half. The vibrant green flesh contrasts with the tiny black seeds, making it visually appealing. If you need more images or a different kind, just let me know!"
        )
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me an image',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_image_resource',
                            args='{}',
                            tool_call_id='call_KL2BXptkWmKifse91X727M7y',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=393,
                        output_tokens=11,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-CpxEaVAApbQvDQSTnqrFd0mxu7Cs3',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_image_resource',
                            content='See file 241a70',
                            tool_call_id='call_KL2BXptkWmKifse91X727M7y',
                            timestamp=IsDatetime(),
                        ),
                        UserPromptPart(
                            content=['This is file 241a70:', IsInstance(BinaryImage)],
                            timestamp=IsDatetime(),
                        ),
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        TextPart(
                            content="Here's an image of a kiwi fruit cut in half. The vibrant green flesh contrasts with the tiny black seeds, making it visually appealing. If you need more images or a different kind, just let me know!"
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=1196,
                        output_tokens=43,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-CpxEdNdePHwVHTJLjmaHWzNUfpoNo',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_dict(allow_model_requests: None, agent: Agent):
    async with agent:
        result = await agent.run('Get me a dict, respond on one line')
        assert result.output == snapshot('{"foo":"bar","baz":123}')
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me a dict, respond on one line',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[ToolCallPart(tool_name='get_dict', args='{}', tool_call_id='call_oqKviITBj8PwpQjGyUu4Zu5x')],
                    usage=RequestUsage(
                        input_tokens=195,
                        output_tokens=11,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRloOs7Bb2tq8wJyy9Rv7SQ7L65a7',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_dict',
                            content={'foo': 'bar', 'baz': 123},
                            tool_call_id='call_oqKviITBj8PwpQjGyUu4Zu5x',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[TextPart(content='{"foo":"bar","baz":123}')],
                    usage=RequestUsage(
                        input_tokens=222,
                        output_tokens=11,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRloPczU1HSCWnreyo21DdNtdOM7L',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_unstructured_dict(allow_model_requests: None, agent: Agent):
    async with agent:
        result = await agent.run('Get me an unstructured dict, respond on one line')
        assert result.output == snapshot('{"foo":"bar","baz":123}')
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me an unstructured dict, respond on one line',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_unstructured_dict', args='{}', tool_call_id='call_R0n2R7S9vL2aZOX25T9jahTd'
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=343,
                        output_tokens=12,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-CLbP82ODQMEznhobUKdq6Rjn9Aa12',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_unstructured_dict',
                            content={'foo': 'bar', 'baz': 123},
                            tool_call_id='call_R0n2R7S9vL2aZOX25T9jahTd',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[TextPart(content='{"foo":"bar","baz":123}')],
                    usage=RequestUsage(
                        input_tokens=374,
                        output_tokens=10,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-CLbPAOYN3jPYdvYeD8JNOOXF5N554',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_error(allow_model_requests: None, agent: Agent):
    async with agent:
        result = await agent.run('Get me an error, pass False as a value, unless the tool tells you otherwise')
        assert result.output == snapshot(
            'I called the tool with the correct parameter, and it returned: "This is not an error."'
        )
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me an error, pass False as a value, unless the tool tells you otherwise',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_error',
                            args='{"value":false}',
                            tool_call_id='call_rETXZWddAGZSHyVHAxptPGgc',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=203,
                        output_tokens=15,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRloSNg7aGSp1rXDkhInjMIUHKd7A',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        RetryPromptPart(
                            content='Error executing tool get_error: This is an error. Call the tool with True instead',
                            tool_name='get_error',
                            tool_call_id='call_rETXZWddAGZSHyVHAxptPGgc',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_error',
                            args='{"value":true}',
                            tool_call_id='call_4xGyvdghYKHN8x19KWkRtA5N',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=250,
                        output_tokens=15,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRloTvSkFeX4DZKQLqfH9KbQkWlpt',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_error',
                            content='This is not an error',
                            tool_call_id='call_4xGyvdghYKHN8x19KWkRtA5N',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        TextPart(
                            content='I called the tool with the correct parameter, and it returned: "This is not an error."'
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=277,
                        output_tokens=22,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRloU3MhnqNEqujs28a3ofRbs7VPF',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_none(allow_model_requests: None, agent: Agent):
    async with agent:
        result = await agent.run('Call the none tool and say Hello')
        assert result.output == snapshot('Hello! How can I assist you today?')
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Call the none tool and say Hello',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[ToolCallPart(tool_name='get_none', args='{}', tool_call_id='call_mJTuQ2Cl5SaHPTJbIILEUhJC')],
                    usage=RequestUsage(
                        input_tokens=193,
                        output_tokens=11,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRloX2RokWc9j9PAXAuNXGR73WNqY',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_none',
                            content=[],
                            tool_call_id='call_mJTuQ2Cl5SaHPTJbIILEUhJC',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[TextPart(content='Hello! How can I assist you today?')],
                    usage=RequestUsage(
                        input_tokens=212,
                        output_tokens=11,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-BRloYWGujk8yE94gfVSsM1T1Ol2Ej',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_returning_multiple_items(allow_model_requests: None, agent: Agent, image_content: BinaryContent):
    async with agent:
        result = await agent.run('Get me multiple items and summarize in one sentence')
        assert result.output == snapshot(
            'The content includes two strings, a dictionary with keys "foo" and "baz," and an image of a kiwi fruit.'
        )
        assert result.all_messages() == snapshot(
            [
                ModelRequest(
                    parts=[
                        UserPromptPart(
                            content='Get me multiple items and summarize in one sentence',
                            timestamp=IsDatetime(),
                        )
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        ToolCallPart(
                            tool_name='get_multiple_items',
                            args='{}',
                            tool_call_id='call_pyHWn85cReaMKhKpY5J4cGev',
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=398,
                        output_tokens=11,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'tool_calls',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-CpzU5Mhbq7bf8kaSOksp2KUTqG4u0',
                    finish_reason='tool_call',
                    run_id=IsStr(),
                ),
                ModelRequest(
                    parts=[
                        ToolReturnPart(
                            tool_name='get_multiple_items',
                            content=[
                                'This is a string',
                                'Another string',
                                {'foo': 'bar', 'baz': 123},
                                'See file 241a70',
                            ],
                            tool_call_id='call_pyHWn85cReaMKhKpY5J4cGev',
                            timestamp=IsDatetime(),
                        ),
                        UserPromptPart(
                            content=['This is file 241a70:', IsInstance(BinaryImage)], timestamp=IsDatetime()
                        ),
                    ],
                    timestamp=IsDatetime(),
                    run_id=IsStr(),
                ),
                ModelResponse(
                    parts=[
                        TextPart(
                            content='The content includes two strings, a dictionary with keys "foo" and "baz," and an image of a kiwi fruit.'
                        )
                    ],
                    usage=RequestUsage(
                        input_tokens=1220,
                        output_tokens=26,
                        details={
                            'accepted_prediction_tokens': 0,
                            'audio_tokens': 0,
                            'reasoning_tokens': 0,
                            'rejected_prediction_tokens': 0,
                        },
                    ),
                    model_name='gpt-4o-2024-08-06',
                    timestamp=IsDatetime(),
                    provider_name='openai',
                    provider_url='https://api.openai.com/v1/',
                    provider_details={
                        'finish_reason': 'stop',
                        'timestamp': IsDatetime(),
                    },
                    provider_response_id='chatcmpl-CpzU6VAYJTRYnthvYDAolptinBLkS',
                    finish_reason='stop',
                    run_id=IsStr(),
                ),
            ]
        )


async def test_tool_metadata_extraction():
    """Test that MCP tool metadata is properly extracted into ToolDefinition."""

    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        ctx = RunContext(deps=None, model=TestModel(), usage=RunUsage())
        tools = [tool.tool_def for tool in (await server.get_tools(ctx)).values()]
        # find `celsius_to_fahrenheit`
        celsius_to_fahrenheit = next(tool for tool in tools if tool.name == 'celsius_to_fahrenheit')
        assert celsius_to_fahrenheit.metadata is not None
        assert celsius_to_fahrenheit.metadata.get('annotations') is not None
        assert celsius_to_fahrenheit.metadata.get('annotations', {}).get('title', None) == 'Celsius to Fahrenheit'
        assert celsius_to_fahrenheit.metadata.get('output_schema') is not None
        assert celsius_to_fahrenheit.metadata.get('output_schema', {}).get('type', None) == 'object'


async def test_client_sampling(run_context: RunContext[int]):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    server.sampling_model = TestModel(custom_output_text='sampling model response')
    async with server:
        result = await server.direct_call_tool('use_sampling', {'foo': 'bar'})
        assert result == snapshot(
            {
                '_meta': None,
                'role': 'assistant',
                'content': {'type': 'text', 'text': 'sampling model response', 'annotations': None, '_meta': None},
                'model': 'test',
                'stopReason': None,
            }
        )


async def test_client_sampling_disabled(run_context: RunContext[int]):
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], allow_sampling=False)
    server.sampling_model = TestModel(custom_output_text='sampling model response')
    async with server:
        with pytest.raises(ModelRetry, match='Error executing tool use_sampling: Sampling not supported'):
            await server.direct_call_tool('use_sampling', {'foo': 'bar'})


async def test_mcp_server_raises_mcp_error(
    allow_model_requests: None, mcp_server: MCPServerStdio, agent: Agent, run_context: RunContext[int]
) -> None:
    mcp_error = McpError(error=ErrorData(code=400, message='Test MCP error conversion'))

    async with agent:
        with patch.object(
            mcp_server._client,  # pyright: ignore[reportPrivateUsage]
            'send_request',
            new=AsyncMock(side_effect=mcp_error),
        ):
            with pytest.raises(ModelRetry, match='Test MCP error conversion'):
                await mcp_server.direct_call_tool('test_tool', {})


def test_map_from_mcp_params_model_request():
    params = CreateMessageRequestParams(
        messages=[
            SamplingMessage(role='user', content=TextContent(type='text', text='xx')),
            SamplingMessage(
                role='user',
                content=ImageContent(type='image', data=base64.b64encode(b'img').decode(), mimeType='image/png'),
            ),
        ],
        maxTokens=8,
    )
    pai_messages = map_from_mcp_params(params)
    assert pai_messages == snapshot(
        [
            ModelRequest(
                parts=[
                    UserPromptPart(content='xx', timestamp=IsNow(tz=timezone.utc)),
                    UserPromptPart(
                        content=[BinaryContent(data=b'img', media_type='image/png', identifier='978ea7')],
                        timestamp=IsNow(tz=timezone.utc),
                    ),
                ]
            )
        ]
    )


def test_map_from_mcp_params_model_response():
    params = CreateMessageRequestParams(
        messages=[
            SamplingMessage(role='assistant', content=TextContent(type='text', text='xx')),
        ],
        maxTokens=8,
    )
    pai_messages = map_from_mcp_params(params)
    assert pai_messages == snapshot(
        [
            ModelResponse(
                parts=[TextPart(content='xx')],
                timestamp=IsNow(tz=timezone.utc),
            )
        ]
    )


def test_map_from_pai_messages_with_binary_content():
    """Test that map_from_pai_messages correctly converts image and audio content to MCP format.

    Note: `data` in this case are base64-encoded bytes (e.g., base64.b64encode(b'raw')).
    map_from_pai_messages decodes this to get the base64 string for MCP.
    """

    message = ModelRequest(
        parts=[
            UserPromptPart(content='text message'),
            UserPromptPart(content=[BinaryContent(data=b'raw_image_bytes', media_type='image/png')]),
            # TODO uncomment when audio content is supported
            # UserPromptPart(content=[BinaryContent(data=b'raw_audio_bytes', media_type='audio/wav'), 'text after audio']),
        ]
    )
    system_prompt, sampling_msgs = map_from_pai_messages([message])
    assert system_prompt == ''
    assert [m.model_dump(by_alias=True) for m in sampling_msgs] == snapshot(
        [
            {'role': 'user', 'content': {'type': 'text', 'text': 'text message', 'annotations': None, '_meta': None}},
            {
                'role': 'user',
                'content': {
                    'type': 'image',
                    'data': 'cmF3X2ltYWdlX2J5dGVz',
                    'mimeType': 'image/png',
                    'annotations': None,
                    '_meta': None,
                },
            },
        ]
    )

    # Unsupported content type raises NotImplementedError
    message_with_video = ModelRequest(
        parts=[UserPromptPart(content=[BinaryContent(data=b'raw_video_bytes', media_type='video/mp4')])]
    )
    with pytest.raises(
        NotImplementedError, match="Unsupported content type: <class 'pydantic_ai.messages.BinaryContent'>"
    ):
        map_from_pai_messages([message_with_video])


def test_map_from_model_response_unexpected_part_raises_error():
    with pytest.raises(UnexpectedModelBehavior, match='Unexpected part type: ToolCallPart, expected TextPart'):
        map_from_model_response(ModelResponse(parts=[ToolCallPart(tool_name='test-tool')]))


def test_map_from_model_response_mixed_parts():
    result = map_from_model_response(
        ModelResponse(
            parts=[
                TextPart(content='Hello '),
                ThinkingPart(content='Should I say world?'),
                TextPart(content='world!'),
                ThinkingPart(content='That sounded good.'),
            ]
        )
    )
    assert result.model_dump(by_alias=True) == snapshot(
        {'type': 'text', 'text': 'Hello world!', 'annotations': None, '_meta': None}
    )


async def test_elicitation_callback_functionality(run_context: RunContext[int]):
    """Test that elicitation callback is actually called and works."""
    # Track callback execution
    callback_called = False
    callback_message = None
    callback_response = 'Yes, proceed with the action'

    async def mock_elicitation_callback(
        context: RequestContext[ClientSession, Any, Any], params: ElicitRequestParams
    ) -> ElicitResult:
        nonlocal callback_called, callback_message
        callback_called = True
        callback_message = params.message
        return ElicitResult(action='accept', content={'response': callback_response})

    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], elicitation_callback=mock_elicitation_callback)

    async with server:
        # Call the tool that uses elicitation
        result = await server.direct_call_tool('use_elicitation', {'question': 'Should I continue?'})

        # Verify the callback was called
        assert callback_called, 'Elicitation callback should have been called'
        assert callback_message == 'Should I continue?', 'Callback should receive the question'
        assert result == f'User responded: {callback_response}', 'Tool should return the callback response'


async def test_elicitation_callback_not_set(run_context: RunContext[int]):
    """Test that elicitation fails when no callback is set."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])

    async with server:
        # Should raise an error when elicitation is attempted without callback
        with pytest.raises(ModelRetry, match='Elicitation not supported'):
            await server.direct_call_tool('use_elicitation', {'question': 'Should I continue?'})


async def test_read_text_resource(run_context: RunContext[int]):
    """Test reading a text resource (converted to string)."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        # Test reading by URI string
        content = await server.read_resource('resource://product_name.txt')
        assert isinstance(content, str)
        assert content == snapshot('Pydantic AI\n')

        # Test reading by Resource object
        resource = Resource(uri='resource://product_name.txt', name='product_name_resource')
        content_from_resource = await server.read_resource(resource)
        assert isinstance(content_from_resource, str)
        assert content_from_resource == snapshot('Pydantic AI\n')


async def test_read_blob_resource(run_context: RunContext[int]):
    """Test reading a binary resource (converted to BinaryContent)."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        content = await server.read_resource('resource://kiwi.jpg')
        assert isinstance(content, BinaryContent)
        assert content.media_type == snapshot('image/jpeg')
        # Verify it's JPEG data (starts with JPEG magic bytes)
        assert content.data.startswith(bytes.fromhex('ffd8ffe0'))  # JPEG magic bytes


async def test_read_resource_template(run_context: RunContext[int]):
    """Test reading a resource template with parameters (converted to string)."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        content = await server.read_resource('resource://greeting/Alice')
        assert isinstance(content, str)
        assert content == snapshot('Hello, Alice!')


async def test_read_resource_error(mcp_server: MCPServerStdio) -> None:
    """Test that read_resource converts McpError to MCPError for generic errors."""
    mcp_error = McpError(
        error=ErrorData(code=-32603, message='Failed to read resource', data={'details': 'disk error'})
    )

    async with mcp_server:
        with patch.object(
            mcp_server._client,  # pyright: ignore[reportPrivateUsage]
            'read_resource',
            new=AsyncMock(side_effect=mcp_error),
        ):
            with pytest.raises(MCPError, match='Failed to read resource') as exc_info:
                await mcp_server.read_resource('resource://error')

            # Verify the exception has the expected attributes
            assert exc_info.value.code == -32603
            assert exc_info.value.message == 'Failed to read resource'
            assert exc_info.value.data == {'details': 'disk error'}


async def test_read_resource_empty_contents(mcp_server: MCPServerStdio) -> None:
    """Test that read_resource returns empty list when server returns empty contents."""
    from mcp.types import ReadResourceResult

    # Mock a result with empty contents
    empty_result = ReadResourceResult(contents=[])

    async with mcp_server:
        with patch.object(
            mcp_server._client,  # pyright: ignore[reportPrivateUsage]
            'read_resource',
            new=AsyncMock(return_value=empty_result),
        ):
            result = await mcp_server.read_resource('resource://empty')
            assert result == []


async def test_list_resources_error(mcp_server: MCPServerStdio) -> None:
    """Test that list_resources converts McpError to MCPError."""
    mcp_error = McpError(
        error=ErrorData(code=-32603, message='Failed to list resources', data={'details': 'server overloaded'})
    )

    async with mcp_server:
        with patch.object(
            mcp_server._client,  # pyright: ignore[reportPrivateUsage]
            'list_resources',
            new=AsyncMock(side_effect=mcp_error),
        ):
            with pytest.raises(MCPError, match='Failed to list resources') as exc_info:
                await mcp_server.list_resources()

            # Verify the exception has the expected attributes
            assert exc_info.value.code == -32603
            assert exc_info.value.message == 'Failed to list resources'
            assert exc_info.value.data == {'details': 'server overloaded'}
            assert (
                str(exc_info.value) == "Failed to list resources (code: -32603, data: {'details': 'server overloaded'})"
            )


async def test_list_resource_templates_error(mcp_server: MCPServerStdio) -> None:
    """Test that list_resource_templates converts McpError to MCPError."""
    mcp_error = McpError(error=ErrorData(code=-32001, message='Service unavailable'))

    async with mcp_server:
        with patch.object(
            mcp_server._client,  # pyright: ignore[reportPrivateUsage]
            'list_resource_templates',
            new=AsyncMock(side_effect=mcp_error),
        ):
            with pytest.raises(MCPError, match='Service unavailable') as exc_info:
                await mcp_server.list_resource_templates()

            # Verify the exception has the expected attributes
            assert exc_info.value.code == -32001
            assert exc_info.value.message == 'Service unavailable'


def test_load_mcp_servers(tmp_path: Path):
    config = tmp_path / 'mcp.json'

    config.write_text('{"mcpServers": {"potato": {"url": "https://example.com/mcp"}}}', encoding='utf-8')
    server = load_mcp_servers(config)[0]
    assert server == MCPServerStreamableHTTP(url='https://example.com/mcp', id='potato', tool_prefix='potato')

    config.write_text(
        '{"mcpServers": {"potato": {"command": "python", "args": ["-m", "tests.mcp_server"]}}}', encoding='utf-8'
    )
    server = load_mcp_servers(config)[0]
    assert server == MCPServerStdio(
        command='python', args=['-m', 'tests.mcp_server'], id='potato', tool_prefix='potato'
    )

    config.write_text('{"mcpServers": {"potato": {"url": "https://example.com/sse"}}}', encoding='utf-8')
    server = load_mcp_servers(config)[0]
    assert server == MCPServerSSE(url='https://example.com/sse', id='potato', tool_prefix='potato')

    with pytest.raises(FileNotFoundError):
        load_mcp_servers(tmp_path / 'does_not_exist.json')


def test_load_mcp_servers_with_env_vars(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test environment variable expansion in config files."""
    config = tmp_path / 'mcp.json'

    # Test with environment variables in command
    monkeypatch.setenv('PYTHON_CMD', 'python3')
    monkeypatch.setenv('MCP_MODULE', 'tests.mcp_server')
    config.write_text(
        '{"mcpServers": {"my_server": {"command": "${PYTHON_CMD}", "args": ["-m", "${MCP_MODULE}"]}}}', encoding='utf-8'
    )

    servers = load_mcp_servers(config)

    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.command == 'python3'
    assert server.args == ['-m', 'tests.mcp_server']
    assert server.id == 'my_server'
    assert server.tool_prefix == 'my_server'


def test_load_mcp_servers_env_var_in_env_dict(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test environment variable expansion in env dict."""
    config = tmp_path / 'mcp.json'

    # Test with environment variables in env dict
    monkeypatch.setenv('API_KEY', 'secret123')
    config.write_text(
        '{"mcpServers": {"my_server": {"command": "python", "args": ["-m", "tests.mcp_server"], '
        '"env": {"API_KEY": "${API_KEY}"}}}}',
        encoding='utf-8',
    )

    servers = load_mcp_servers(config)

    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.env == {'API_KEY': 'secret123'}


def test_load_mcp_servers_env_var_expansion_url(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test environment variable expansion in URL."""
    config = tmp_path / 'mcp.json'

    # Test with environment variables in URL
    monkeypatch.setenv('SERVER_HOST', 'example.com')
    monkeypatch.setenv('SERVER_PORT', '8080')
    config.write_text(
        '{"mcpServers": {"web_server": {"url": "https://${SERVER_HOST}:${SERVER_PORT}/mcp"}}}', encoding='utf-8'
    )

    servers = load_mcp_servers(config)

    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStreamableHTTP)
    assert server.url == 'https://example.com:8080/mcp'


def test_load_mcp_servers_undefined_env_var(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test that undefined environment variables raise an error."""
    config = tmp_path / 'mcp.json'

    # Make sure the environment variable is not set
    monkeypatch.delenv('UNDEFINED_VAR', raising=False)

    config.write_text('{"mcpServers": {"my_server": {"command": "${UNDEFINED_VAR}", "args": []}}}', encoding='utf-8')

    with pytest.raises(ValueError, match='Environment variable \\$\\{UNDEFINED_VAR\\} is not defined'):
        load_mcp_servers(config)


def test_load_mcp_servers_partial_env_vars(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test environment variables in partial strings."""
    config = tmp_path / 'mcp.json'

    monkeypatch.setenv('HOST', 'example.com')
    monkeypatch.setenv('PATH_SUFFIX', 'mcp')
    config.write_text('{"mcpServers": {"server": {"url": "https://${HOST}/api/${PATH_SUFFIX}"}}}', encoding='utf-8')

    servers = load_mcp_servers(config)

    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStreamableHTTP)
    assert server.url == 'https://example.com/api/mcp'


def test_load_mcp_servers_with_non_string_values(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test that non-string primitive values (int, bool, null) in nested structures are passed through unchanged."""
    config = tmp_path / 'mcp.json'

    # Create a config with environment variables and extra fields containing primitives
    # The extra fields will be ignored during validation but go through _expand_env_vars
    monkeypatch.setenv('PYTHON_CMD', 'python')
    config.write_text(
        '{"mcpServers": {"my_server": {"command": "${PYTHON_CMD}", "args": ["-m", "tests.mcp_server"], '
        '"metadata": {"count": 42, "enabled": true, "value": null}}}}',
        encoding='utf-8',
    )

    # This should successfully expand env vars and ignore the metadata field
    servers = load_mcp_servers(config)

    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.command == 'python'


def test_load_mcp_servers_with_default_values(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test ${VAR:-default} syntax for environment variable expansion."""
    config = tmp_path / 'mcp.json'

    # Test with undefined variable using default
    monkeypatch.delenv('UNDEFINED_VAR', raising=False)
    config.write_text(
        '{"mcpServers": {"server": {"command": "${UNDEFINED_VAR:-python3}", "args": []}}}', encoding='utf-8'
    )

    servers = load_mcp_servers(config)
    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.command == 'python3'

    # Test with defined variable (should use actual value, not default)
    monkeypatch.setenv('DEFINED_VAR', 'actual_value')
    config.write_text(
        '{"mcpServers": {"server": {"command": "${DEFINED_VAR:-default_value}", "args": []}}}', encoding='utf-8'
    )

    servers = load_mcp_servers(config)
    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.command == 'actual_value'

    # Test with empty string as default
    monkeypatch.delenv('UNDEFINED_VAR', raising=False)
    config.write_text('{"mcpServers": {"server": {"command": "${UNDEFINED_VAR:-}", "args": []}}}', encoding='utf-8')

    servers = load_mcp_servers(config)
    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.command == ''


def test_load_mcp_servers_with_default_values_in_url(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test ${VAR:-default} syntax in URLs."""
    config = tmp_path / 'mcp.json'

    # Test with default values in URL
    monkeypatch.delenv('HOST', raising=False)
    monkeypatch.setenv('PROTOCOL', 'https')
    config.write_text(
        '{"mcpServers": {"server": {"url": "${PROTOCOL:-http}://${HOST:-localhost}:${PORT:-8080}/mcp"}}}',
        encoding='utf-8',
    )

    servers = load_mcp_servers(config)
    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStreamableHTTP)
    assert server.url == 'https://localhost:8080/mcp'


def test_load_mcp_servers_with_default_values_in_env_dict(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test ${VAR:-default} syntax in env dictionary."""
    config = tmp_path / 'mcp.json'

    monkeypatch.delenv('API_KEY', raising=False)
    monkeypatch.setenv('CUSTOM_VAR', 'custom_value')
    config.write_text(
        '{"mcpServers": {"server": {"command": "python", "args": [], '
        '"env": {"API_KEY": "${API_KEY:-default_key}", "CUSTOM": "${CUSTOM_VAR:-fallback}"}}}}',
        encoding='utf-8',
    )

    servers = load_mcp_servers(config)
    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.env == {'API_KEY': 'default_key', 'CUSTOM': 'custom_value'}


def test_load_mcp_servers_with_complex_default_values(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test ${VAR:-default} syntax with special characters in default."""
    config = tmp_path / 'mcp.json'

    monkeypatch.delenv('PATH_VAR', raising=False)
    # Test default with slashes, dots, and dashes
    config.write_text(
        '{"mcpServers": {"server": {"command": "${PATH_VAR:-/usr/local/bin/python-3.10}", "args": []}}}',
        encoding='utf-8',
    )

    servers = load_mcp_servers(config)
    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.command == '/usr/local/bin/python-3.10'


def test_load_mcp_servers_with_mixed_syntax(tmp_path: Path, monkeypatch: pytest.MonkeyPatch):
    """Test mixing ${VAR} and ${VAR:-default} syntax in the same config."""
    config = tmp_path / 'mcp.json'

    monkeypatch.setenv('REQUIRED_VAR', 'required_value')
    monkeypatch.delenv('OPTIONAL_VAR', raising=False)
    config.write_text(
        '{"mcpServers": {"server": {"command": "${REQUIRED_VAR}", "args": ["${OPTIONAL_VAR:-default_arg}"]}}}',
        encoding='utf-8',
    )

    servers = load_mcp_servers(config)
    assert len(servers) == 1
    server = servers[0]
    assert isinstance(server, MCPServerStdio)
    assert server.command == 'required_value'
    assert server.args == ['default_arg']


async def test_server_info(mcp_server: MCPServerStdio) -> None:
    with pytest.raises(
        AttributeError, match='The `MCPServerStdio.server_info` is only instantiated after initialization.'
    ):
        mcp_server.server_info
    async with mcp_server:
        assert mcp_server.server_info is not None
        assert mcp_server.server_info.name == 'Pydantic AI MCP Server'


async def test_capabilities(mcp_server: MCPServerStdio) -> None:
    with pytest.raises(
        AttributeError, match='The `MCPServerStdio.capabilities` is only instantiated after initialization.'
    ):
        mcp_server.capabilities
    async with mcp_server:
        assert mcp_server.capabilities is not None
        assert mcp_server.capabilities.resources is True
        assert mcp_server.capabilities.tools is True
        assert mcp_server.capabilities.prompts is True
        assert mcp_server.capabilities.logging is True
        assert mcp_server.capabilities.completions is False
        assert mcp_server.capabilities.experimental is None


async def test_resource_methods_without_capability(mcp_server: MCPServerStdio) -> None:
    """Test that resource list methods return empty values when resources capability is not available."""
    async with mcp_server:
        # Mock the capabilities to not support resources
        mock_capabilities = ServerCapabilities(resources=False)
        with patch.object(mcp_server, '_server_capabilities', mock_capabilities):
            # list_resources should return empty list
            result = await mcp_server.list_resources()
            assert result == []

            # list_resource_templates should return empty list
            result = await mcp_server.list_resource_templates()
            assert result == []


async def test_instructions(mcp_server: MCPServerStdio) -> None:
    with pytest.raises(
        AttributeError, match='The `MCPServerStdio.instructions` is only available after initialization.'
    ):
        mcp_server.instructions
    async with mcp_server:
        assert mcp_server.instructions == 'Be a helpful assistant.'


async def test_client_info_passed_to_session() -> None:
    """Test that provided client_info is passed unchanged to ClientSession."""
    implementation = Implementation(
        name='MyCustomClient',
        version='2.5.3',
        title='Custom MCP client',
        websiteUrl='https://example.com/client',
    )
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], client_info=implementation)

    async with server:
        result = await server.direct_call_tool('get_client_info', {})
        assert result == {
            'name': 'MyCustomClient',
            'version': '2.5.3',
            'title': 'Custom MCP client',
            'websiteUrl': 'https://example.com/client',
        }


async def test_client_info_not_set() -> None:
    """Test that when client_info is not set, the default MCP client info is used."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])

    async with server:
        result = await server.direct_call_tool('get_client_info', {})
        # When client_info is not set, the MCP library provides default client info
        assert result is not None
        assert isinstance(result, dict)
        assert result['name'] == 'mcp'


async def test_agent_run_stream_with_mcp_server_http(allow_model_requests: None, model: Model):
    server = MCPServerStreamableHTTP(url='https://mcp.deepwiki.com/mcp', timeout=30)
    agent = Agent(model, toolsets=[server], instructions='Be concise.')

    # This should not raise an error.
    # See https://github.com/pydantic/pydantic-ai/issues/2818#issuecomment-3476480829
    async with agent.run_stream('Summarize the pydantic/pydantic-ai repo in one sentence') as result:
        output = await result.get_output()
    assert output == snapshot(
        'The `pydantic/pydantic-ai` repository is a Python agent framework designed to facilitate the development of production-grade Generative AI applications and workflows with a focus on type-safety and an ergonomic developer experience.'
    )


async def test_custom_http_client_not_closed():
    custom_http_client = cached_async_http_client()

    assert not custom_http_client.is_closed

    my_mcp_server = MCPServerStreamableHTTP(
        url='https://mcp.deepwiki.com/mcp', http_client=custom_http_client, timeout=30
    )

    tools = await my_mcp_server.list_tools()
    assert len(tools) > 0

    assert not custom_http_client.is_closed


# ============================================================================
# Tool and Resource Caching Tests
# ============================================================================


async def test_tools_caching_enabled_by_default() -> None:
    """Test that list_tools() caches results by default."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        # First call - should fetch from server and cache
        tools1 = await server.list_tools()
        assert len(tools1) > 0
        assert server._cached_tools is not None  # pyright: ignore[reportPrivateUsage]

        # Second call - should return cached value (cache is still populated)
        tools2 = await server.list_tools()
        assert tools2 == tools1
        assert server._cached_tools is not None  # pyright: ignore[reportPrivateUsage]


async def test_tools_no_caching_when_disabled() -> None:
    """Test that list_tools() does not cache when cache_tools=False."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], cache_tools=False)
    async with server:
        # First call - should not populate cache
        tools1 = await server.list_tools()
        assert len(tools1) > 0
        assert server._cached_tools is None  # pyright: ignore[reportPrivateUsage]

        # Second call - cache should still be None
        tools2 = await server.list_tools()
        assert tools2 == tools1
        assert server._cached_tools is None  # pyright: ignore[reportPrivateUsage]


async def test_tools_cache_invalidation_on_notification() -> None:
    """Test that tools cache is invalidated when ToolListChangedNotification is received."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        # Get initial tools - hidden_tool should NOT be present (it's disabled at startup)
        tools1 = await server.list_tools()
        tool_names1 = [t.name for t in tools1]
        assert 'hidden_tool' not in tool_names1
        assert 'enable_hidden_tool' in tool_names1

        # Enable the hidden tool (server sends ToolListChangedNotification)
        await server.direct_call_tool('enable_hidden_tool', {})

        # Get tools again - hidden_tool should now be present (cache was invalidated)
        tools2 = await server.list_tools()
        tool_names2 = [t.name for t in tools2]
        assert 'hidden_tool' in tool_names2


async def test_resources_caching_enabled_by_default() -> None:
    """Test that list_resources() caches results by default."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        assert server.capabilities.resources

        # First call - should fetch from server and cache
        resources1 = await server.list_resources()
        assert server._cached_resources is not None  # pyright: ignore[reportPrivateUsage]

        # Second call - should return cached value (cache is still populated)
        resources2 = await server.list_resources()
        assert resources2 == resources1
        assert server._cached_resources is not None  # pyright: ignore[reportPrivateUsage]


async def test_resources_no_caching_when_disabled() -> None:
    """Test that list_resources() does not cache when cache_resources=False."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'], cache_resources=False)
    async with server:
        assert server.capabilities.resources

        # First call - should not populate cache
        resources1 = await server.list_resources()
        assert server._cached_resources is None  # pyright: ignore[reportPrivateUsage]

        # Second call - cache should still be None
        resources2 = await server.list_resources()
        assert resources2 == resources1
        assert server._cached_resources is None  # pyright: ignore[reportPrivateUsage]


async def test_resources_cache_invalidation_on_notification() -> None:
    """Test that resources cache is invalidated when ResourceListChangedNotification is received."""
    from mcp.types import ResourceListChangedNotification, ServerNotification

    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        assert server.capabilities.resources

        # Populate cache
        await server.list_resources()
        assert server._cached_resources is not None  # pyright: ignore[reportPrivateUsage]

        # Simulate receiving a resource list changed notification
        notification = ServerNotification(ResourceListChangedNotification())
        await server._handle_notification(notification)  # pyright: ignore[reportPrivateUsage]

        # Cache should be invalidated
        assert server._cached_resources is None  # pyright: ignore[reportPrivateUsage]


async def test_cache_cleared_on_connection_close() -> None:
    """Test that caches are cleared when the connection is closed."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])

    # First connection
    async with server:
        await server.list_tools()
        assert server._cached_tools is not None  # pyright: ignore[reportPrivateUsage]

    # After exiting, cache should be cleared by __aexit__
    assert server._cached_tools is None  # pyright: ignore[reportPrivateUsage]

    # Reconnect and verify cache starts empty
    async with server:
        assert server._cached_tools is None  # pyright: ignore[reportPrivateUsage]
        # Fetch again to populate
        await server.list_tools()
        assert server._cached_tools is not None  # pyright: ignore[reportPrivateUsage]


async def test_server_capabilities_list_changed_fields() -> None:
    """Test that ServerCapabilities correctly parses listChanged fields."""
    server = MCPServerStdio('python', ['-m', 'tests.mcp_server'])
    async with server:
        caps = server.capabilities
        assert isinstance(caps.prompts_list_changed, bool)
        assert isinstance(caps.tools_list_changed, bool)
        assert isinstance(caps.resources_list_changed, bool)
