#!/usr/bin/env bats

load test_helper

@test "empty rehash" {
  assert [ ! -d "${PYENV_ROOT}/shims" ]
  run pyenv-rehash
  assert_success ""
  assert [ -d "${PYENV_ROOT}/shims" ]
  rmdir "${PYENV_ROOT}/shims"
}

@test "non-writable shims directory" {
  mkdir -p "${PYENV_ROOT}/shims"
  chmod -w "${PYENV_ROOT}/shims"
  run pyenv-rehash
  assert_failure "pyenv: cannot rehash: ${PYENV_ROOT}/shims isn't writable"
}

@test "rehash in progress" {
  export PYENV_REHASH_TIMEOUT=1
  mkdir -p "${PYENV_ROOT}/shims"
  touch "${PYENV_ROOT}/shims/.pyenv-shim"
  #avoid failure due to a localized error message
  LANG=C run pyenv-rehash
  assert_failure <<!
pyenv: cannot rehash: couldn't acquire lock ${PYENV_ROOT}/shims/.pyenv-shim for 1 seconds. Last error message:
*/pyenv-rehash: line *: ${PYENV_ROOT}/shims/.pyenv-shim: cannot overwrite existing file
!
}

@test "wait until lock acquisition" {
  export PYENV_REHASH_TIMEOUT=5
  mkdir -p "${PYENV_ROOT}/shims"
  touch "${PYENV_ROOT}/shims/.pyenv-shim"
  bash -c "sleep 1 && rm -f ${PYENV_ROOT}/shims/.pyenv-shim" &
  run pyenv-rehash
  assert_success
}

@test "creates shims" {
  create_alt_executable_in_version "2.7" "python"
  create_alt_executable_in_version "2.7" "fab"
  create_alt_executable_in_version "3.4" "python"
  create_alt_executable_in_version "3.4" "py.test"

  assert [ ! -e "${PYENV_ROOT}/shims/fab" ]
  assert [ ! -e "${PYENV_ROOT}/shims/python" ]
  assert [ ! -e "${PYENV_ROOT}/shims/py.test" ]

  run pyenv-rehash
  assert_success ""

  run ls "${PYENV_ROOT}/shims"
  assert_success
  assert_output <<OUT
fab
py.test
python
OUT
}

@test "removes stale shims" {
  mkdir -p "${PYENV_ROOT}/shims"
  touch "${PYENV_ROOT}/shims/oldshim1"
  chmod +x "${PYENV_ROOT}/shims/oldshim1"

  create_alt_executable_in_version "3.4" "fab"
  create_alt_executable_in_version "3.4" "python"

  run pyenv-rehash
  assert_success ""

  assert [ ! -e "${PYENV_ROOT}/shims/oldshim1" ]
}

@test "binary install locations containing spaces" {
  create_alt_executable_in_version "dirname1 p247" "python"
  create_alt_executable_in_version "dirname2 preview1" "py.test"

  assert [ ! -e "${PYENV_ROOT}/shims/python" ]
  assert [ ! -e "${PYENV_ROOT}/shims/py.test" ]

  run pyenv-rehash
  assert_success ""

  run ls "${PYENV_ROOT}/shims"
  assert_success
  assert_output <<OUT
py.test
python
OUT
}

@test "carries original IFS within hooks" {
  create_hook rehash hello.bash <<SH
hellos=(\$(printf "hello\\tugly world\\nagain"))
echo HELLO="\$(printf ":%s" "\${hellos[@]}")"
exit
SH

  IFS=$' \t\n' run pyenv-rehash
  assert_success
  assert_output "HELLO=:hello:ugly:world:again"
}

@test "sh-rehash in bash" {
  create_alt_executable_in_version "3.4" "python"
  PYENV_SHELL=bash run pyenv-sh-rehash
  assert_success "command pyenv rehash
hash -r 2>/dev/null || true"
}

@test "sh-rehash in bash (integration)" {
  create_alt_executable_in_version "3.4" "python"
  PYENV_SHELL=bash run eval "$(pyenv-sh-rehash)"
  assert_success
  assert [ -x "${PYENV_ROOT}/shims/python" ]
}

@test "sh-rehash in fish" {
  create_alt_executable_in_version "3.4" "python"
  PYENV_SHELL=fish run pyenv-sh-rehash
  assert_success "command pyenv rehash"
}

@test "sh-rehash in fish (integration)" {
  command -v fish >/dev/null || skip "-- fish not installed" 
  create_alt_executable_in_version "3.4" "python"
  PYENV_SHELL=fish fish -Nc "source (pyenv-sh-rehash | psub)"
  assert_success
  assert [ -x "${PYENV_ROOT}/shims/python" ]
}

@test "sh-rehash in pwsh" {
  create_alt_executable_in_version "3.4" "python"
  PYENV_SHELL=pwsh run pyenv-sh-rehash
  assert_success "& (get-command pyenv -commandtype application) rehash"
}

@test "sh-rehash in pwsh (integration)" {
  command -v pwsh >/dev/null || skip "-- pwsh not installed" 
  assert [ ! -x "${PYENV_ROOT}/shims/python" ]
  create_alt_executable_in_version "3.4" "python"
  run pwsh -nop -c -<<'!'
$ErrorActionPreference = "Stop"
$PSNativeCommandUseErrorActionPreference = $true
iex ((& pyenv init - pwsh) -join "`n")
& pyenv rehash
!
  assert_success ""
  assert [ -x "${PYENV_ROOT}/shims/python" ]
}

@test "shim sets _PYENV_SHIM_PATH when linked from elsewhere" {
  export PYENV_VERSION="custom"
  create_alt_executable python3
  #must stub pyenv before rehash 'cuz the path is hardcoded into shims
  create_stub pyenv <<!
[[ \$1 == 'exec' ]] && \
echo _PYENV_SHIM_PATH="\$_PYENV_SHIM_PATH"
!
  pyenv-rehash
  mkdir -p "${PYENV_TEST_DIR}/alt-shim"
  ln -s "${PYENV_ROOT}/shims/python3" "${PYENV_TEST_DIR}/alt-shim/python3"
  
  run "${PYENV_TEST_DIR}/alt-shim/python3"
  assert_success
  assert_output <<!
_PYENV_SHIM_PATH=${PYENV_TEST_DIR}/alt-shim
!

  run python3
  assert_success
  assert_output <<!
_PYENV_SHIM_PATH=
!
}
