import os
import unittest

import pyexcel as pe

from .nose_tools import eq_


class TestIrregularities(unittest.TestCase):
    def setUp(self):
        self.testfile = "test.xlsm"
        self.content = [
            [1, "", "", "", "", ""],
            [1, 2, "", "", "", ""],
            [1, 2, 3, "", "", ""],
            [1, 2, 3, 4, "", ""],
            [1, 2, 3, 4, 5, ""],
            [1, 2, 3, 4, 5, 6],
            [1, "", "", "", "", ""],
            [1, 2, "", "", "", ""],
            [1, 2, 3, "", "", ""],
            [1, 2, 3, 4, "", ""],
            [1, 2, 3, 4, 5, ""],
        ]
        pe.save_as(dest_file_name=self.testfile, array=self.content)

    def test_irregular_data_file(self):
        r = pe.get_sheet(file_name=self.testfile)
        result = list(r.rows())
        assert result == self.content

    def tearDown(self):
        if os.path.exists(self.testfile):
            os.unlink(self.testfile)


class TestIrregularities2(unittest.TestCase):
    def setUp(self):
        self.testfile = "test.xlsm"
        self.content = {
            "Series_6": ["", "", "", "", "", 6.0],
            "Series_5": ["", "", "", "", "", 5.0, 5.0, "", "", ""],
            "Series_4": ["", "", "", 4.0, 4.0, 4.0, "", "", "", 4.0, 4.0],
            "Series_3": ["", "", 3.0, 3.0, 3.0, 3.0, "", "", 3.0, 3.0, 3.0],
            "Series_2": ["", 2.0, 2.0, 2.0, 2.0, 2.0],
            "Series_1": [1.0],
        }
        pe.save_as(
            dest_file_name=self.testfile,
            adict=self.content,
            skip_empty_rows=False,
        )

    def test_write_irregular_dict_to_file(self):
        r = pe.get_sheet(file_name=self.testfile, skip_empty_rows=False)
        r.name_columns_by_row(0)
        actual = [
            [
                "Series_1",
                "Series_2",
                "Series_3",
                "Series_4",
                "Series_5",
                "Series_6",
            ],
            [1, "", "", "", "", ""],
            ["", 2, "", "", "", ""],
            ["", 2, 3, "", "", ""],
            ["", 2, 3, 4, "", ""],
            ["", 2, 3, 4, "", ""],
            ["", 2, 3, 4, 5, 6],
            ["", "", "", "", 5, ""],
            ["", "", "", "", "", ""],
            ["", "", 3, "", "", ""],
            ["", "", 3, 4, "", ""],
            ["", "", 3, 4, "", ""],
        ]
        print(r.array)
        print(actual)
        for r1, r2 in zip(actual, r.array):
            eq_(r1, r2)

    def tearDown(self):
        if os.path.exists(self.testfile):
            os.unlink(self.testfile)


class TestIrregularities10(unittest.TestCase):
    def setUp(self):
        self.testfile = "test.xlsm"
        self.content = [
            [1.0],
            ["", 2.0, 2.0, 2.0, 2.0, 2.0],
            ["", "", 3.0, 3.0, 3.0, 3.0, "", "", 3.0, 3.0, 3.0],
            ["", "", "", 4.0, 4.0, 4.0, "", "", "", 4.0, 4.0],
            ["", "", "", "", "", 5.0, 5.0, "", "", ""],
            ["", "", "", "", "", 6.0],
        ]
        sheet = pe.get_sheet(array=self.content)
        sheet.transpose()
        sheet.save_as(self.testfile)

    def test_write_irregular_arrays_to_file(self):
        r = pe.get_sheet(file_name=self.testfile, skip_empty_rows=False)
        actual = [
            [1, "", "", "", "", ""],
            ["", 2, "", "", "", ""],
            ["", 2, 3, "", "", ""],
            ["", 2, 3, 4, "", ""],
            ["", 2, 3, 4, "", ""],
            ["", 2, 3, 4, 5, 6],
            ["", "", "", "", 5, ""],
            ["", "", "", "", "", ""],
            ["", "", 3, "", "", ""],
            ["", "", 3, 4, "", ""],
            ["", "", 3, 4, "", ""],
        ]
        eq_(actual, r.array)

    def tearDown(self):
        if os.path.exists(self.testfile):
            os.unlink(self.testfile)


class TestIrregularities3(unittest.TestCase):
    def setUp(self):
        self.testfile = "test.csv"
        self.content = []
        pe.save_as(dest_file_name=self.testfile, array=self.content)

    def test_file_there(self):
        if os.path.exists(self.testfile):
            assert 1 == 1

    def test_file_content(self):
        r = pe.get_sheet(file_name=self.testfile)
        result = list(r)
        assert not result

    def tearDown(self):
        if os.path.exists(self.testfile):
            os.unlink(self.testfile)


class TestMultiSheetReader(unittest.TestCase):
    def setUp(self):
        self.testfile = "file_with_an_empty_sheet.xlsx"

    def test_reader_with_correct_sheets(self):
        r = pe.get_book(
            file_name=os.path.join("tests", "fixtures", self.testfile),
        )
        assert r.number_of_sheets() == 3


class TestEmptyCSVFile(unittest.TestCase):
    def setUp(self):
        self.testfile = os.path.join("tests", "fixtures", "empty.csv")

    def test_number_of_columns(self):
        r = pe.get_sheet(file_name=self.testfile)
        assert r.number_of_columns() == 0


class TestNonUniformCSVFile(unittest.TestCase):
    def setUp(self):
        self.testfile = os.path.join(
            "tests",
            "fixtures",
            "non-uniform-rows.csv",
        )

    def test_number_of_columns(self):
        r = pe.get_sheet(file_name=self.testfile)
        assert r.number_of_columns() == 8
        result = [
            [1, "", "", "", "", "", "", ""],
            [1, 2, "", "", "", "", "", ""],
            [1, 2, 3, "", "", "", "", ""],
            [1, 2, 3, 4, "", "", "", ""],
            [1, 2, 3, 4, 5, "", "", ""],
            [1, 2, 3, 4, 5, 6, "", ""],
            [1, 2, 3, 4, 5, 6, 7, 8],
        ]
        eq_(result, r.array)


class TestMergeFiles(unittest.TestCase):
    def test_merge_two_csv_files(self):
        file1 = os.path.join(
            "tests",
            "fixtures",
            "test_cook_book_merge_numbers.csv",
        )
        file2 = os.path.join(
            "tests",
            "fixtures",
            "test_cook_book_merge_alphabets.csv",
        )
        file_list = [file1, file2]
        pe.cookbook.merge_files(file_list)
        r = pe.get_sheet(file_name="pyexcel_merged.csv")
        actual = [
            [1, 2, 3, "a", "b", "c"],
            [4, 5, 6, "d", "e", "f"],
            [7, 8, 9, "g", "h", "I"],
        ]
        eq_(actual, r.array)

    def test_merge_two_csv_files_2(self):
        file1 = os.path.join(
            "tests",
            "fixtures",
            "test_cook_book_merge_numbers_2.csv",
        )
        file2 = os.path.join(
            "tests",
            "fixtures",
            "test_cook_book_merge_alphabets_2.csv",
        )
        file_list = [file1, file2]
        pe.cookbook.merge_files(file_list)
        r = pe.get_sheet(file_name="pyexcel_merged.csv")
        actual = [
            [1, 2, 3, "", "", "", "", "a", "b", "c", "", ""],
            [4, 5, 6, 10, 11, 12, "", "d", "e", "f", "m", "q"],
            [7, 8, 9, "", "", "", 9, "g", "h", "I", "n", "p"],
        ]
        eq_(actual, r.array)

    def tearDown(self):
        if os.path.exists("pyexcel_merged.csv"):
            os.unlink("pyexcel_merged.csv")
