// Copyright 2021 The Ray Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "config_internal.h"

#include <boost/dll/runtime_symbol_info.hpp>
#include <charconv>
#include <filesystem>

#include "absl/flags/flag.h"
#include "absl/flags/parse.h"
#include "absl/strings/str_split.h"
#include "nlohmann/json.hpp"
#include "ray/common/id.h"
#include "ray/util/network_util.h"

ABSL_FLAG(std::string, ray_address, "", "The address of the Ray cluster to connect to.");

/// absl::flags does not provide a IsDefaultValue method, so use a non-empty dummy default
/// value to support an empty Redis username.
ABSL_FLAG(std::string,
          ray_redis_username,
          "absl::flags dummy default value",
          "Prevents external clients without the username from connecting to Redis "
          "if provided.");

/// absl::flags does not provide a IsDefaultValue method, so use a non-empty dummy default
/// value to support empty redis password.
ABSL_FLAG(std::string,
          ray_redis_password,
          "absl::flags dummy default value",
          "Prevents external clients without the password from connecting to Redis "
          "if provided.");

ABSL_FLAG(std::string,
          ray_code_search_path,
          "",
          "A list of directories or files of dynamic libraries that specify the "
          "search path for user code. Only searching the top level under a directory. "
          "':' is used as the separator.");

ABSL_FLAG(std::string, ray_job_id, "", "Assigned job id.");

ABSL_FLAG(int32_t, ray_node_manager_port, 0, "The port to use for the node manager.");

ABSL_FLAG(std::string,
          ray_raylet_socket_name,
          "",
          "It will specify the socket name used by the raylet if provided.");

ABSL_FLAG(std::string,
          ray_plasma_store_socket_name,
          "",
          "It will specify the socket name used by the plasma store if provided.");

ABSL_FLAG(std::string, ray_session_dir, "", "The path of this session.");

ABSL_FLAG(std::string, ray_logs_dir, "", "Logs dir for workers.");

ABSL_FLAG(std::string, ray_node_ip_address, "", "The ip address for this node.");

ABSL_FLAG(std::string,
          ray_head_args,
          "",
          "The command line args to be appended as parameters of the `ray start` "
          "command. It takes effect only if Ray head is started by a driver. Run `ray "
          "start --help` for details.");

ABSL_FLAG(std::string,
          ray_worker_id,
          "",
          "The worker ID assigned to this worker process by the raylet (hex string).");

ABSL_FLAG(std::string,
          ray_default_actor_lifetime,
          "",
          "The default actor lifetime type, `detached` or `non_detached`.");

ABSL_FLAG(std::string, ray_runtime_env, "", "The serialized runtime env.");

ABSL_FLAG(int,
          ray_runtime_env_hash,
          -1,
          "The computed hash of the runtime env for this worker.");

ABSL_FLAG(std::string,
          ray_job_namespace,
          "",
          "The namespace of job. If not set,"
          " a unique value will be randomly generated.");

using json = nlohmann::json;

namespace ray {
namespace internal {

rpc::JobConfig_ActorLifetime ParseDefaultActorLifetimeType(
    const std::string &default_actor_lifetime_origin) {
  std::string default_actor_lifetime;
  default_actor_lifetime.resize(default_actor_lifetime_origin.size());
  transform(default_actor_lifetime_origin.begin(),
            default_actor_lifetime_origin.end(),
            default_actor_lifetime.begin(),
            ::tolower);
  RAY_CHECK(default_actor_lifetime == "non_detached" ||
            default_actor_lifetime == "detached")
      << "The default_actor_lifetime_string config must be `detached` or `non_detached`.";
  return default_actor_lifetime == "non_detached"
             ? rpc::JobConfig_ActorLifetime_NON_DETACHED
             : rpc::JobConfig_ActorLifetime_DETACHED;
}

void ConfigInternal::Init(RayConfig &config, int argc, char **argv) {
  if (!config.address.empty()) {
    SetBootstrapAddress(config.address);
  }
  run_mode = config.local_mode ? RunMode::SINGLE_PROCESS : RunMode::CLUSTER;
  if (!config.code_search_path.empty()) {
    code_search_path = config.code_search_path;
  }
  if (config.redis_username_) {
    redis_username = *config.redis_username_;
  }
  if (config.redis_password_) {
    redis_password = *config.redis_password_;
  }
  if (!config.head_args.empty()) {
    head_args = config.head_args;
  }
  if (config.default_actor_lifetime == ActorLifetime::DETACHED) {
    default_actor_lifetime = rpc::JobConfig_ActorLifetime_DETACHED;
  }
  if (config.runtime_env) {
    runtime_env = config.runtime_env;
  }

  if (argc != 0 && argv != nullptr) {
    // Parse config from command line.
    absl::ParseCommandLine(argc, argv);

    if (!FLAGS_ray_code_search_path.CurrentValue().empty()) {
      // Code search path like this "/path1/xxx.so:/path2".
      RAY_LOG(DEBUG) << "The code search path is "
                     << FLAGS_ray_code_search_path.CurrentValue();
      code_search_path = absl::StrSplit(
          FLAGS_ray_code_search_path.CurrentValue(), ':', absl::SkipEmpty());
    }
    if (!FLAGS_ray_address.CurrentValue().empty()) {
      SetBootstrapAddress(FLAGS_ray_address.CurrentValue());
    }
    // Don't rewrite `ray_redis_username` when it is not set in the command line.
    if (FLAGS_ray_redis_username.CurrentValue() !=
        FLAGS_ray_redis_username.DefaultValue()) {
      redis_username = FLAGS_ray_redis_username.CurrentValue();
    }
    // Don't rewrite `ray_redis_password` when it is not set in the command line.
    if (FLAGS_ray_redis_password.CurrentValue() !=
        FLAGS_ray_redis_password.DefaultValue()) {
      redis_password = FLAGS_ray_redis_password.CurrentValue();
    }
    if (!FLAGS_ray_job_id.CurrentValue().empty()) {
      job_id = FLAGS_ray_job_id.CurrentValue();
    }
    node_manager_port = absl::GetFlag<int32_t>(FLAGS_ray_node_manager_port);
    if (!FLAGS_ray_raylet_socket_name.CurrentValue().empty()) {
      raylet_socket_name = FLAGS_ray_raylet_socket_name.CurrentValue();
    }
    if (!FLAGS_ray_plasma_store_socket_name.CurrentValue().empty()) {
      plasma_store_socket_name = FLAGS_ray_plasma_store_socket_name.CurrentValue();
    }
    if (!FLAGS_ray_session_dir.CurrentValue().empty()) {
      session_dir = FLAGS_ray_session_dir.CurrentValue();
    }
    if (!FLAGS_ray_logs_dir.CurrentValue().empty()) {
      logs_dir = FLAGS_ray_logs_dir.CurrentValue();
    }
    if (!FLAGS_ray_node_ip_address.CurrentValue().empty()) {
      node_ip_address = FLAGS_ray_node_ip_address.CurrentValue();
    }
    if (!FLAGS_ray_head_args.CurrentValue().empty()) {
      std::vector<std::string> args =
          absl::StrSplit(FLAGS_ray_head_args.CurrentValue(), ' ', absl::SkipEmpty());
      head_args.insert(head_args.end(), args.begin(), args.end());
    }
    worker_id = absl::GetFlag<std::string>(FLAGS_ray_worker_id);
    if (!FLAGS_ray_default_actor_lifetime.CurrentValue().empty()) {
      default_actor_lifetime =
          ParseDefaultActorLifetimeType(FLAGS_ray_default_actor_lifetime.CurrentValue());
    }
    if (!FLAGS_ray_runtime_env.CurrentValue().empty()) {
      runtime_env = RuntimeEnv::Deserialize(FLAGS_ray_runtime_env.CurrentValue());
    }
    runtime_env_hash = absl::GetFlag<int>(FLAGS_ray_runtime_env_hash);
  }
  worker_type = config.is_worker_ ? WorkerType::WORKER : WorkerType::DRIVER;
  if (worker_type == WorkerType::DRIVER && run_mode == RunMode::CLUSTER) {
    if (bootstrap_ip.empty()) {
      auto ray_address_env = std::getenv("RAY_ADDRESS");
      if (ray_address_env) {
        RAY_LOG(DEBUG) << "Initialize Ray cluster address to \"" << ray_address_env
                       << "\" from environment variable \"RAY_ADDRESS\".";
        SetBootstrapAddress(ray_address_env);
      }
    }
    if (code_search_path.empty()) {
      auto program_path = boost::dll::program_location().parent_path();
      RAY_LOG(INFO) << "No code search path found yet. "
                    << "The program location path " << program_path
                    << " will be added for searching dynamic libraries by default."
                    << " And you can add some search paths by '--ray_code_search_path'";
      code_search_path.emplace_back(program_path.string());
    } else {
      // Convert all the paths to absolute path to support configuring relative paths in
      // driver.
      std::vector<std::string> absolute_path;
      for (const auto &path : code_search_path) {
        absolute_path.emplace_back(std::filesystem::absolute(path).string());
      }
      code_search_path = absolute_path;
    }
  }
  if (worker_type == WorkerType::DRIVER) {
    ray_namespace = config.ray_namespace;
    if (!FLAGS_ray_job_namespace.CurrentValue().empty()) {
      ray_namespace = FLAGS_ray_job_namespace.CurrentValue();
    }
    if (ray_namespace.empty()) {
      ray_namespace = UniqueID::FromRandom().Hex();
    }
  }

  auto job_config_json_string = std::getenv("RAY_JOB_CONFIG_JSON_ENV_VAR");
  if (job_config_json_string) {
    json job_config_json = json::parse(job_config_json_string);
    runtime_env = RuntimeEnv::Deserialize(job_config_json.at("runtime_env").dump());
    job_config_metadata = job_config_json.at("metadata")
                              .get<std::unordered_map<std::string, std::string>>();
    RAY_CHECK(job_config_json.size() == 2);
  }
};

void ConfigInternal::SetBootstrapAddress(std::string_view bootstrap_address) {
  auto ip_and_port = ParseAddress(std::string(bootstrap_address));
  RAY_CHECK(ip_and_port.has_value());

  bootstrap_ip = (*ip_and_port)[0];
  auto ret = std::from_chars((*ip_and_port)[1].data(),
                             (*ip_and_port)[1].data() + (*ip_and_port)[1].size(),
                             bootstrap_port);
  RAY_CHECK(ret.ec == std::errc());
}

void ConfigInternal::UpdateSessionDir(const std::string dir) {
  if (session_dir.empty()) {
    session_dir = dir;
  }
  if (logs_dir.empty()) {
    logs_dir = session_dir + "/logs";
  }
}
}  // namespace internal
}  // namespace ray
