// Copyright 2017 The Ray Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "ray/common/id.h"

#include <gtest/gtest.h>

#include <utility>

#include "absl/container/flat_hash_set.h"

namespace ray {

void TestFromIndexObjectId(const TaskID &task_id, int64_t index) {
  // Round trip test for computing the object ID for an object created by a task, either
  // via an object put or by being a return object for the task.
  ObjectID obj_id = ObjectID::FromIndex(task_id, index);
  ASSERT_EQ(obj_id.TaskId(), task_id);
  ASSERT_EQ(obj_id.ObjectIndex(), index);
}

void TestRandomObjectId() {
  // Round trip test for computing the object ID from random.
  const ObjectID random_object_id = ObjectID::FromRandom();
  ASSERT_FALSE(random_object_id.TaskId().IsNil());
  ASSERT_EQ(random_object_id.ObjectIndex(), 0);
}

static const JobID kDefaultJobId = JobID::FromInt(199);

static const TaskID kDefaultDriverTaskId = TaskID::ForDriverTask(kDefaultJobId);

TEST(JobIDTest, TestJobID) {
  uint32_t id = 100;
  JobID job_id = JobID::FromInt(id);
  ASSERT_EQ(job_id.ToInt(), id);
}

TEST(ActorIDTest, TestActorID) {
  {
    // test from binary
    const ActorID actor_id_1 = ActorID::Of(kDefaultJobId, kDefaultDriverTaskId, 1);
    const auto actor_id_1_binary = actor_id_1.Binary();
    const auto actor_id_2 = ActorID::FromBinary(actor_id_1_binary);
    ASSERT_EQ(actor_id_1, actor_id_2);
  }

  {
    // test get job id
    const ActorID actor_id = ActorID::Of(kDefaultJobId, kDefaultDriverTaskId, 1);
    ASSERT_EQ(kDefaultJobId, actor_id.JobId());
  }
}

TEST(TaskIDTest, TestTaskID) {
  // Round trip test for task ID.
  {
    const ActorID actor_id = ActorID::Of(kDefaultJobId, kDefaultDriverTaskId, 1);
    const TaskID task_id_1 =
        TaskID::ForActorTask(kDefaultJobId, kDefaultDriverTaskId, 1, actor_id);
    ASSERT_EQ(actor_id, task_id_1.ActorId());
    ASSERT_FALSE(task_id_1.IsForActorCreationTask());

    auto actor_creation_task_id = TaskID::ForActorCreationTask(actor_id);
    ASSERT_TRUE(actor_creation_task_id.IsForActorCreationTask());

    ASSERT_FALSE(TaskID::Nil().IsForActorCreationTask());
    ASSERT_FALSE(TaskID::FromRandom(kDefaultJobId).IsForActorCreationTask());
  }
}

TEST(TaskIDTest, TestTaskIDForExecution) {
  const TaskID task_id = TaskID::FromRandom(kDefaultJobId);
  ASSERT_NE(task_id, TaskID::ForExecutionAttempt(task_id, 0));
  ASSERT_NE(task_id, TaskID::ForExecutionAttempt(task_id, 1));
  ASSERT_NE(TaskID::ForExecutionAttempt(task_id, 0),
            TaskID::ForExecutionAttempt(task_id, 1));
  ASSERT_EQ(TaskID::ForExecutionAttempt(task_id, 0),
            TaskID::ForExecutionAttempt(task_id, 0));
  ASSERT_EQ(TaskID::ForExecutionAttempt(task_id, 1),
            TaskID::ForExecutionAttempt(task_id, 1));
  // Check for overflow.
  ASSERT_NE(TaskID::ForExecutionAttempt(task_id, 0),
            TaskID::ForExecutionAttempt(task_id, 256));
  ASSERT_NE(TaskID::ForExecutionAttempt(task_id, 0),
            TaskID::ForExecutionAttempt(task_id, 256 * 256));

  // Check that child tasks generated by different execution attempts of the
  // same task do not collide.
  TaskID child_task_1 =
      TaskID::ForNormalTask(kDefaultJobId, TaskID::ForExecutionAttempt(task_id, 0), 0);
  TaskID child_task_2 =
      TaskID::ForNormalTask(kDefaultJobId, TaskID::ForExecutionAttempt(task_id, 1), 0);
  ASSERT_NE(child_task_1, child_task_2);
}

TEST(ObjectIDTest, TestObjectID) {
  static const ActorID default_actor_id =
      ActorID::Of(kDefaultJobId, kDefaultDriverTaskId, 1);
  static const TaskID default_task_id =
      TaskID::ForActorTask(kDefaultJobId, kDefaultDriverTaskId, 1, default_actor_id);

  {
    // test from index
    TestFromIndexObjectId(default_task_id, 1);
    TestFromIndexObjectId(default_task_id, 2);
    TestFromIndexObjectId(default_task_id, ObjectID::kMaxObjectIndex);
  }

  {
    // test random object id
    TestRandomObjectId();
  }
}

TEST(NilTest, TestIsNil) {
  ASSERT_TRUE(TaskID().IsNil());
  ASSERT_TRUE(TaskID::Nil().IsNil());
  ASSERT_TRUE(ObjectID().IsNil());
  ASSERT_TRUE(ObjectID::Nil().IsNil());
}

TEST(HashTest, TestNilHash) {
  // Manually trigger the hash calculation of the static global nil ID.
  auto nil_hash = ObjectID::Nil().Hash();
  ObjectID id1 = ObjectID::FromRandom();
  ASSERT_NE(nil_hash, id1.Hash());
  ObjectID id2 = ObjectID::FromBinary(ObjectID::FromRandom().Binary());
  ASSERT_NE(nil_hash, id2.Hash());
  ASSERT_NE(id1.Hash(), id2.Hash());
}

TEST(HashTest, TestIdHash) {
  absl::flat_hash_set<ObjectID> oids;
  ObjectID cur_oid = ObjectID::FromRandom();
  oids.emplace(cur_oid);

  // Lookup with the same id shows exists.
  EXPECT_TRUE(oids.contains(cur_oid));
  // Lookup with constant reference shows exists.
  EXPECT_TRUE(oids.contains(static_cast<const ObjectID &>(cur_oid)));
  // Insert with rvalue reference show exists.
  auto [_, is_new] = oids.emplace(std::move(cur_oid));
  EXPECT_FALSE(is_new);
}

TEST(PlacementGroupIDTest, TestPlacementGroup) {
  {
    // test from binary
    PlacementGroupID placement_group_id_1 = PlacementGroupID::Of(JobID::FromInt(1));
    const auto placement_group_id_1_binary = placement_group_id_1.Binary();
    const auto placement_group_id_2 =
        PlacementGroupID::FromBinary(placement_group_id_1_binary);
    ASSERT_EQ(placement_group_id_1, placement_group_id_2);
    const auto placement_group_id_1_hex = placement_group_id_1.Hex();
    const auto placement_group_id_3 = PlacementGroupID::FromHex(placement_group_id_1_hex);
    ASSERT_EQ(placement_group_id_1, placement_group_id_3);
  }

  {
    // test get job id
    auto job_id = JobID::FromInt(1);
    const PlacementGroupID placement_group_id = PlacementGroupID::Of(job_id);
    ASSERT_EQ(job_id, placement_group_id.JobId());
  }
}

TEST(LeaseIDTest, TestLeaseID) {
  // Test basic LeaseID creation, size, and worker extraction
  const WorkerID worker_id = WorkerID::FromRandom();
  const LeaseID lease_id = LeaseID::FromWorker(worker_id, 2);
  const size_t lease_id_size = 32;
  ASSERT_FALSE(lease_id.IsNil());
  ASSERT_EQ(lease_id.WorkerId(), worker_id);
  ASSERT_EQ(LeaseID::Size(), lease_id_size);
  ASSERT_EQ(lease_id.Binary().size(), lease_id_size);

  const LeaseID random_lease = LeaseID::FromRandom();
  const LeaseID another_lease = LeaseID::FromWorker(worker_id, 1);

  ASSERT_FALSE(random_lease.IsNil());
  ASSERT_NE(lease_id, another_lease);
  ASSERT_NE(lease_id, random_lease);
  ASSERT_EQ(lease_id.WorkerId(), another_lease.WorkerId());

  // Test serialization roundtrip
  const LeaseID from_hex = LeaseID::FromHex(lease_id.Hex());
  const LeaseID from_binary = LeaseID::FromBinary(lease_id.Binary());

  ASSERT_EQ(lease_id, from_hex);
  ASSERT_EQ(lease_id, from_binary);
  ASSERT_EQ(lease_id.WorkerId(), from_hex.WorkerId());
}

}  // namespace ray
