// Copyright 2020-2021 The Ray Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <grpcpp/grpcpp.h>

#include <boost/asio.hpp>
#include <boost/chrono.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/thread.hpp>
#include <cstdlib>
#include <memory>
#include <unordered_map>
#include <unordered_set>
#include <vector>

using namespace boost;            // NOLINT
using namespace boost::asio;      // NOLINT
using namespace boost::asio::ip;  // NOLINT

#include <ray/rpc/grpc_server.h>

#include <chrono>
#include <thread>

#include "gtest/gtest.h"
#include "ray/gcs/gcs_health_check_manager.h"
#include "ray/observability/fake_metric.h"
#include "ray/util/network_util.h"

int GetFreePort() {
  io_context io_service;
  tcp::acceptor acceptor(io_service);
  tcp::endpoint endpoint;

  // try to bind to port 0 to find a free port
  acceptor.open(tcp::v4());
  acceptor.bind(tcp::endpoint(tcp::v4(), 0));
  endpoint = acceptor.local_endpoint();
  auto port = endpoint.port();
  acceptor.close();
  return port;
}

using namespace ray;                             // NOLINT
using namespace std::literals::chrono_literals;  // NOLINT

class GcsHealthCheckManagerTest : public ::testing::Test {
 protected:
  GcsHealthCheckManagerTest() = default;
  ~GcsHealthCheckManagerTest() = default;
  void SetUp() override {
    grpc::EnableDefaultHealthCheckService(true);

    health_check = gcs::GcsHealthCheckManager::Create(
        io_service,
        [this](const NodeID &id) { dead_nodes.insert(id); },
        fake_health_check_rpc_latency_ms_histogram_,
        initial_delay_ms,
        timeout_ms,
        period_ms,
        failure_threshold);
  }

  void TearDown() override {
    io_service.poll();
    io_service.stop();

    // Stop the servers.
    for (auto [_, server] : servers) {
      server->Shutdown();
    }

    // Allow gRPC to cleanup.
    boost::this_thread::sleep_for(boost::chrono::seconds(2));
  }

  NodeID AddServer(bool alive = true) {
    std::promise<int> port_promise;
    auto node_id = NodeID::FromRandom();
    auto port = GetFreePort();
    RAY_LOG(INFO) << "Get port " << port;
    auto server = std::make_shared<rpc::GrpcServer>(node_id.Hex(), port, true);

    auto channel = grpc::CreateChannel(BuildAddress("localhost", port),
                                       grpc::InsecureChannelCredentials());
    server->Run();
    if (alive) {
      server->GetServer().GetHealthCheckService()->SetServingStatus(node_id.Hex(), true);
    }
    servers.emplace(node_id, server);
    health_check->AddNode(node_id, channel);
    return node_id;
  }

  void StopServing(const NodeID &id) {
    auto iter = servers.find(id);
    RAY_CHECK(iter != servers.end());
    iter->second->GetServer().GetHealthCheckService()->SetServingStatus(false);
  }

  void StartServing(const NodeID &id) {
    auto iter = servers.find(id);
    RAY_CHECK(iter != servers.end());
    iter->second->GetServer().GetHealthCheckService()->SetServingStatus(true);
  }

  void DeleteServer(const NodeID &id) {
    auto iter = servers.find(id);
    if (iter != servers.end()) {
      iter->second->Shutdown();
      servers.erase(iter);
    }
  }

  void Run(size_t n = 1) {
    // If n == 0 it mean we just run it and return.
    if (n == 0) {
      io_service.run();
      io_service.restart();
      return;
    }

    while (n) {
      auto i = io_service.run_one();
      n -= i;
      io_service.restart();
    }
  }

  instrumented_io_context io_service;
  std::shared_ptr<gcs::GcsHealthCheckManager> health_check;
  std::unordered_map<NodeID, std::shared_ptr<rpc::GrpcServer>> servers;
  std::unordered_set<NodeID> dead_nodes;
  ray::observability::FakeHistogram fake_health_check_rpc_latency_ms_histogram_;
  const int64_t initial_delay_ms = 100;
  const int64_t timeout_ms = 10;
  const int64_t period_ms = 10;
  const int64_t failure_threshold = 5;
};

TEST_F(GcsHealthCheckManagerTest, TestBasic) {
  auto node_id = AddServer();
  Run(0);  // Initial run
  ASSERT_TRUE(dead_nodes.empty());

  // Run the first health check
  Run();
  ASSERT_TRUE(dead_nodes.empty());

  Run(2);  // One for starting RPC and one for the RPC callback.
  ASSERT_TRUE(dead_nodes.empty());
  StopServing(node_id);

  for (auto i = 0; i < failure_threshold; ++i) {
    Run(2);  // One for starting RPC and one for the RPC callback.
  }

  ASSERT_EQ(1, dead_nodes.size());
  ASSERT_TRUE(dead_nodes.count(node_id));
}

TEST_F(GcsHealthCheckManagerTest, MarkHealthAndSkipCheck) {
  auto node_id = AddServer();
  Run(0);  // Initial run
  ASSERT_TRUE(dead_nodes.empty());

  // Run the first health check: even we mark node down, health check is skipped due to
  // fresh enough information.
  StopServing(node_id);
  health_check->MarkNodeHealthy(node_id);
  Run(0);
  ASSERT_TRUE(dead_nodes.empty());
}

TEST_F(GcsHealthCheckManagerTest, StoppedAndResume) {
  auto node_id = AddServer();
  Run(0);  // Initial run
  ASSERT_TRUE(dead_nodes.empty());

  // Run the first health check
  Run();
  ASSERT_TRUE(dead_nodes.empty());

  Run(2);  // One for starting RPC and one for the RPC callback.
  ASSERT_TRUE(dead_nodes.empty());
  StopServing(node_id);

  for (auto i = 0; i < failure_threshold; ++i) {
    Run(2);  // One for starting RPC and one for the RPC callback.
    if (i == failure_threshold / 2) {
      StartServing(node_id);
    }
  }

  ASSERT_EQ(0, dead_nodes.size());
}

TEST_F(GcsHealthCheckManagerTest, Crashed) {
  auto node_id = AddServer();
  Run(0);  // Initial run
  ASSERT_TRUE(dead_nodes.empty());

  // Run the first health check
  Run();
  ASSERT_TRUE(dead_nodes.empty());

  Run(2);  // One for starting RPC and one for the RPC callback.
  ASSERT_TRUE(dead_nodes.empty());

  // Check it again
  Run(2);  // One for starting RPC and one for the RPC callback.
  ASSERT_TRUE(dead_nodes.empty());

  DeleteServer(node_id);

  for (auto i = 0; i < failure_threshold; ++i) {
    Run(2);  // One for starting RPC and one for the RPC callback.
  }

  ASSERT_EQ(1, dead_nodes.size());
  ASSERT_TRUE(dead_nodes.count(node_id));
}

TEST_F(GcsHealthCheckManagerTest, NodeRemoved) {
  auto node_id = AddServer();
  Run(0);  // Initial run
  ASSERT_TRUE(dead_nodes.empty());

  // Run the first health check
  Run();
  ASSERT_TRUE(dead_nodes.empty());

  Run(2);  // One for starting RPC and one for the RPC callback.
  ASSERT_TRUE(dead_nodes.empty());
  health_check->RemoveNode(node_id);

  // Make sure it's not monitored any more
  for (auto i = 0; i < failure_threshold; ++i) {
    io_service.poll();
  }

  ASSERT_EQ(0, dead_nodes.size());
  ASSERT_EQ(0, health_check->GetAllNodes().size());
}

TEST_F(GcsHealthCheckManagerTest, NoRegister) {
  auto node_id = AddServer(false);
  for (auto i = 0; i < failure_threshold; ++i) {
    Run(2);  // One for starting RPC and one for the RPC callback.
  }

  Run(1);
  ASSERT_EQ(1, dead_nodes.size());
  ASSERT_TRUE(dead_nodes.count(node_id));
}

TEST_F(GcsHealthCheckManagerTest, StressTest) {
#ifdef _RAY_TSAN_BUILD
  GTEST_SKIP() << "Disabled in tsan because of performance";
#endif
  boost::asio::executor_work_guard<boost::asio::io_context::executor_type> work(
      io_service.get_executor());
  std::srand(std::time(nullptr));
  auto t = std::make_unique<std::thread>([this]() { this->io_service.run(); });

  std::vector<NodeID> alive_nodes;

  for (int i = 0; i < 200; ++i) {
    alive_nodes.emplace_back(AddServer(true));
    std::this_thread::sleep_for(10ms);
  }

  for (size_t i = 0; i < 20000UL; ++i) {
    RAY_LOG(INFO) << "Progress: " << i << "/20000";
    auto iter = alive_nodes.begin() + std::rand() % alive_nodes.size();
    health_check->RemoveNode(*iter);
    DeleteServer(*iter);
    alive_nodes.erase(iter);
    alive_nodes.emplace_back(AddServer(true));
  }
  RAY_LOG(INFO) << "Finished!";
  io_service.stop();
  t->join();
}
