// Copyright 2017 The Ray Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "ray/gcs/gcs_placement_group_manager.h"

#include <gtest/gtest.h>

#include <memory>
#include <string>
#include <vector>

#include "mock/ray/gcs/gcs_node_manager.h"
#include "mock/ray/pubsub/publisher.h"
#include "ray/common/asio/instrumented_io_context.h"
#include "ray/common/test_utils.h"
#include "ray/gcs/store_client/in_memory_store_client.h"
#include "ray/observability/fake_metric.h"
#include "ray/raylet/scheduling/cluster_resource_manager.h"
#include "ray/util/counter_map.h"

namespace ray {
namespace gcs {

using ::testing::_;

class MockPlacementGroupScheduler : public gcs::GcsPlacementGroupSchedulerInterface {
 public:
  MockPlacementGroupScheduler() = default;

  void ScheduleUnplacedBundles(const SchedulePgRequest &request) override {
    placement_groups_.push_back(request.placement_group);
  }

  MOCK_METHOD1(DestroyPlacementGroupBundleResourcesIfExists,
               void(const PlacementGroupID &placement_group_id));

  MOCK_METHOD1(MarkScheduleCancelled, void(const PlacementGroupID &placement_group_id));

  MOCK_METHOD1(
      ReleaseUnusedBundles,
      void(const absl::flat_hash_map<NodeID, std::vector<rpc::Bundle>> &node_to_bundles));

  MOCK_METHOD2(
      Initialize,
      void(const absl::flat_hash_map<PlacementGroupID,
                                     std::vector<std::shared_ptr<BundleSpecification>>>
               &group_to_bundles,
           const std::vector<SchedulePgRequest> &prepared_pgs));

  MOCK_METHOD((absl::flat_hash_map<PlacementGroupID, std::vector<int64_t>>),
              GetBundlesOnNode,
              (const NodeID &node_id),
              (const, override));

  absl::flat_hash_map<PlacementGroupID, std::vector<int64_t>> GetAndRemoveBundlesOnNode(
      const NodeID &node_id) override {
    absl::flat_hash_map<PlacementGroupID, std::vector<int64_t>> bundles;
    bundles[group_on_dead_node_] = bundles_on_dead_node_;
    return bundles;
  }

  int GetPlacementGroupCount() { return placement_groups_.size(); }

  PlacementGroupID group_on_dead_node_;
  std::vector<int64_t> bundles_on_dead_node_;
  std::vector<std::shared_ptr<gcs::GcsPlacementGroup>> placement_groups_;
};

class GcsPlacementGroupManagerTest : public ::testing::Test {
 public:
  GcsPlacementGroupManagerTest()
      : mock_placement_group_scheduler_(new MockPlacementGroupScheduler()),
        cluster_resource_manager_(io_service_) {
    gcs_publisher_ = std::make_shared<pubsub::GcsPublisher>(
        std::make_unique<ray::pubsub::MockPublisher>());
    gcs_table_storage_ =
        std::make_unique<gcs::GcsTableStorage>(std::make_unique<InMemoryStoreClient>());
    gcs_node_manager_ = std::make_shared<gcs::MockGcsNodeManager>();
    gcs_resource_manager_ = std::make_shared<gcs::GcsResourceManager>(
        io_service_, cluster_resource_manager_, *gcs_node_manager_, NodeID::FromRandom());
    gcs_placement_group_manager_.reset(new gcs::GcsPlacementGroupManager(
        io_service_,
        mock_placement_group_scheduler_.get(),
        gcs_table_storage_.get(),
        *gcs_resource_manager_,
        [this](const JobID &job_id) { return job_namespace_table_[job_id]; },
        fake_placement_group_gauge_,
        fake_placement_group_creation_latency_in_ms_histogram_,
        fake_placement_group_scheduling_latency_in_ms_histogram_,
        fake_placement_group_count_gauge_));
    counter_.reset(new CounterMap<rpc::PlacementGroupTableData::PlacementGroupState>());
    for (int i = 1; i <= 10; i++) {
      auto job_id = JobID::FromInt(i);
      job_namespace_table_[job_id] = "";
    }
  }

  void SetUp() override { io_service_.restart(); }

  void TearDown() override { io_service_.stop(); }

  // Make placement group registration sync.
  void RegisterPlacementGroup(const ray::rpc::CreatePlacementGroupRequest &request,
                              rpc::StatusCallback callback) {
    std::promise<void> promise;
    JobID job_id = JobID::FromBinary(request.placement_group_spec().creator_job_id());
    std::string ray_namespace = job_namespace_table_[job_id];
    gcs_placement_group_manager_->RegisterPlacementGroup(
        std::make_shared<gcs::GcsPlacementGroup>(request, ray_namespace, counter_),
        [&callback, &promise](Status status) {
          RAY_CHECK_OK(status);
          callback(status);
          promise.set_value();
        });
    RunIOService();
    promise.get_future().get();
  }

  // Mock receiving prepare request for a placement group and update the committed
  // resources for each bundle
  void MockReceivePrepareRequest(
      const std::shared_ptr<gcs::GcsPlacementGroup> &placement_group) {
    int bundles_size = placement_group->GetPlacementGroupTableData().bundles_size();
    for (int bundle_index = 0; bundle_index < bundles_size; bundle_index++) {
      placement_group->GetMutableBundle(bundle_index)
          ->set_node_id(NodeID::FromRandom().Binary());
    }
  }

  // Mock receiving prepare request for specific bundle in a placement group
  // and update the committed resources for the specific bundles
  void MockReceivePrepareRequestWithBundleIndexes(
      const std::shared_ptr<gcs::GcsPlacementGroup> &placement_group,
      const std::vector<int> &bundle_indices) {
    for (const auto &bundle_index : bundle_indices) {
      placement_group->GetMutableBundle(bundle_index)
          ->set_node_id(NodeID::FromRandom().Binary());
    }
  }

  // Mock prepare resource bundles for a placement group
  void PrepareBundleResources(
      const std::shared_ptr<gcs::GcsPlacementGroup> &placement_group) {
    // mock all bundles of pg have prepared and committed resource.
    MockReceivePrepareRequest(placement_group);

    // A placement group must first become PREPARED then it can become CREATED.
    // Normally transition to PREPARED is performed by
    // GcsPlacementGroupScheduler::OnAllBundlePrepareRequestReturned.
    placement_group->UpdateState(rpc::PlacementGroupTableData::PREPARED);
  }

  // Mock prepare resource bundles for a placement group with specific bundle indexes
  void PrepareBundleResourcesWithIndex(
      const std::shared_ptr<gcs::GcsPlacementGroup> &placement_group,
      const std::vector<int> &bundle_indices) {
    // mock prepare resource bundles with committed resource for specific bundle indexes
    MockReceivePrepareRequestWithBundleIndexes(placement_group, bundle_indices);

    // A placement group must first become PREPARED then it can become CREATED.
    // Normally transition to PREPARED is performed by
    // GcsPlacementGroupScheduler::OnAllBundlePrepareRequestReturned.
    placement_group->UpdateState(rpc::PlacementGroupTableData::PREPARED);
  }

  // Mock committing resource bundles for a placement group
  void CommitBundleResources(
      const std::shared_ptr<gcs::GcsPlacementGroup> &placement_group) {
    gcs_placement_group_manager_->OnPlacementGroupCreationSuccess(placement_group);
    RunIOService();
  }

  // We need this to ensure that `MarkSchedulingDone` and `SchedulePendingPlacementGroups`
  // was already invoked when we have invoked `OnPlacementGroupCreationSuccess`.
  // Mock creating a placement group
  void OnPlacementGroupCreationSuccess(
      const std::shared_ptr<gcs::GcsPlacementGroup> &placement_group) {
    std::promise<void> promise;
    gcs_placement_group_manager_->WaitPlacementGroup(
        placement_group->GetPlacementGroupID(), [&promise](Status status) {
          RAY_CHECK_OK(status);
          promise.set_value();
        });

    PrepareBundleResources(placement_group);
    CommitBundleResources(placement_group);
    promise.get_future().get();
  }

  std::shared_ptr<GcsInitData> LoadDataFromDataStorage() {
    auto gcs_init_data = std::make_shared<GcsInitData>(*gcs_table_storage_);
    std::promise<void> promise;
    gcs_init_data->AsyncLoad({[&promise] { promise.set_value(); }, io_service_});
    RunIOService();
    promise.get_future().get();
    return gcs_init_data;
  }

  void RunIOService() { io_service_.poll(); }

  ExponentialBackoff GetExpBackOff() { return ExponentialBackoff(0, 1); }

  std::shared_ptr<MockPlacementGroupScheduler> mock_placement_group_scheduler_;
  std::unique_ptr<gcs::GcsPlacementGroupManager> gcs_placement_group_manager_;
  absl::flat_hash_map<JobID, std::string> job_namespace_table_;
  std::shared_ptr<CounterMap<rpc::PlacementGroupTableData::PlacementGroupState>> counter_;

 protected:
  std::unique_ptr<gcs::GcsTableStorage> gcs_table_storage_;
  instrumented_io_context io_service_;
  ray::observability::FakeGauge fake_placement_group_gauge_;
  ray::observability::FakeHistogram
      fake_placement_group_creation_latency_in_ms_histogram_;
  ray::observability::FakeHistogram
      fake_placement_group_scheduling_latency_in_ms_histogram_;
  ray::observability::FakeGauge fake_placement_group_count_gauge_;

 private:
  ClusterResourceManager cluster_resource_manager_;
  std::shared_ptr<gcs::GcsNodeManager> gcs_node_manager_;
  std::shared_ptr<gcs::GcsResourceManager> gcs_resource_manager_;
  std::shared_ptr<pubsub::GcsPublisher> gcs_publisher_;
};

TEST_F(GcsPlacementGroupManagerTest, TestPlacementGroupBundleCache) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request,
                         [&registered_placement_group_count](const Status &status) {
                           ++registered_placement_group_count;
                         });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  ASSERT_TRUE(placement_group->cached_bundle_specs_.empty());
  // Fill the cache and verify it.
  const auto &bundle_specs = placement_group->GetBundles();
  ASSERT_EQ(placement_group->cached_bundle_specs_, bundle_specs);
  ASSERT_FALSE(placement_group->cached_bundle_specs_.empty());
  // Invalidate the cache and verify it.
  RAY_UNUSED(placement_group->GetMutableBundle(0));
  ASSERT_TRUE(placement_group->cached_bundle_specs_.empty());
}

TEST_F(GcsPlacementGroupManagerTest, TestBasic) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request,
                         [&registered_placement_group_count](const Status &status) {
                           ++registered_placement_group_count;
                         });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::PENDING), 1);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::CREATED), 0);
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::PENDING), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::CREATED), 1);

  gcs_placement_group_manager_->SetUsageStatsClient(nullptr);
  gcs_placement_group_manager_->RecordMetrics();
  auto counter_tag_to_value = fake_placement_group_count_gauge_.GetTagToValue();
  // 3 states: PENDING, REGISTERED, INFEASIBLE
  ASSERT_EQ(counter_tag_to_value.size(), 3);
  for (auto &[key, value] : counter_tag_to_value) {
    if (key.at("State") == "Registered") {
      ASSERT_EQ(value, 1);
    } else if (key.at("State") == "Infeasible") {
      ASSERT_EQ(value, 0);
    } else if (key.at("State") == "Pending") {
      ASSERT_EQ(value, 0);
    }
  }
  auto creation_latency_tag_to_value =
      fake_placement_group_creation_latency_in_ms_histogram_.GetTagToValue();
  ASSERT_EQ(creation_latency_tag_to_value.size(), 1);
  auto scheduling_latency_tag_to_value =
      fake_placement_group_scheduling_latency_in_ms_histogram_.GetTagToValue();
  ASSERT_EQ(scheduling_latency_tag_to_value.size(), 1);
}

TEST_F(GcsPlacementGroupManagerTest, TestSchedulingFailed) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request,
                         [&registered_placement_group_count](const Status &status) {
                           ++registered_placement_group_count;
                         });

  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.clear();

  ASSERT_EQ(placement_group->GetStats().scheduling_attempt(), 1);
  gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
      placement_group, GetExpBackOff(), true);
  RunIOService();
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::PENDING), 1);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::CREATED), 0);

  gcs_placement_group_manager_->SchedulePendingPlacementGroups();
  RunIOService();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 1);
  ASSERT_EQ(placement_group->GetStats().scheduling_attempt(), 2);
  mock_placement_group_scheduler_->placement_groups_.clear();

  // Check that the placement_group is in state `CREATED`.
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::PENDING), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::CREATED), 1);
}

TEST_F(GcsPlacementGroupManagerTest, TestGetPlacementGroupIDByName) {
  auto request = GenCreatePlacementGroupRequest("test_name");
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });

  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();

  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  ASSERT_EQ(
      gcs_placement_group_manager_->GetPlacementGroupIDByName("test_name", ""),
      PlacementGroupID::FromBinary(request.placement_group_spec().placement_group_id()));
}

TEST_F(GcsPlacementGroupManagerTest, TestRemoveNamedPlacementGroup) {
  auto request = GenCreatePlacementGroupRequest("test_name");
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request,
                         [&registered_placement_group_count](const Status &status) {
                           ++registered_placement_group_count;
                         });

  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();

  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  // Remove the named placement group.
  gcs_placement_group_manager_->RemovePlacementGroup(
      placement_group->GetPlacementGroupID(),
      [](const Status &status) { ASSERT_TRUE(status.ok()); });
  RunIOService();
  ASSERT_EQ(gcs_placement_group_manager_->GetPlacementGroupIDByName("test_name", ""),
            PlacementGroupID::Nil());
}

TEST_F(GcsPlacementGroupManagerTest, TestRemovedPlacementGroupNotReportedAsLoad) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.clear();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::PENDING);

  // Placement group is in leasing state.
  const auto &placement_group_id = placement_group->GetPlacementGroupID();
  EXPECT_CALL(*mock_placement_group_scheduler_, MarkScheduleCancelled(placement_group_id))
      .Times(1);
  gcs_placement_group_manager_->RemovePlacementGroup(placement_group_id,
                                                     [](const Status &status) {});
  RunIOService();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::REMOVED);
  gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
      placement_group, GetExpBackOff(), true);
  RunIOService();

  auto load = gcs_placement_group_manager_->GetPlacementGroupLoad();
  ASSERT_EQ(load->placement_group_data_size(), 0);
}

TEST_F(GcsPlacementGroupManagerTest, TestRescheduleWhenNodeAdd) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();

  // If the creation of placement group fails, it will be rescheduled after a short time.
  gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
      placement_group, GetExpBackOff(), true);
  ASSERT_TRUE(WaitForCondition(
      [this]() {
        RunIOService();
        return mock_placement_group_scheduler_->GetPlacementGroupCount() == 1;
      },
      10 * 1000));
}

TEST_F(GcsPlacementGroupManagerTest, TestRemovingPendingPlacementGroup) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.clear();

  gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
      placement_group, GetExpBackOff(), true);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::PENDING);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::PENDING), 1);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::CREATED), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::REMOVED), 0);
  const auto &placement_group_id = placement_group->GetPlacementGroupID();
  gcs_placement_group_manager_->RemovePlacementGroup(placement_group_id,
                                                     [](const Status &status) {});
  RunIOService();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::REMOVED);
  ASSERT_EQ(placement_group->GetStats().scheduling_state(),
            rpc::PlacementGroupStats::REMOVED);

  // Make sure it is not rescheduled
  gcs_placement_group_manager_->SchedulePendingPlacementGroups();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 0);
  mock_placement_group_scheduler_->placement_groups_.clear();

  // Make sure we can re-remove again.
  gcs_placement_group_manager_->RemovePlacementGroup(
      placement_group_id, [](const Status &status) { ASSERT_TRUE(status.ok()); });
  RunIOService();
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::PENDING), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::CREATED), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::REMOVED), 1);
}

TEST_F(GcsPlacementGroupManagerTest, TestRemovingLeasingPlacementGroup) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.clear();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::PENDING);

  // Placement group is in leasing state.
  const auto &placement_group_id = placement_group->GetPlacementGroupID();
  EXPECT_CALL(*mock_placement_group_scheduler_, MarkScheduleCancelled(placement_group_id))
      .Times(1);
  gcs_placement_group_manager_->RemovePlacementGroup(placement_group_id,
                                                     [](const Status &status) {});
  RunIOService();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::REMOVED);
  gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
      placement_group, GetExpBackOff(), true);

  // Make sure it is not rescheduled
  gcs_placement_group_manager_->SchedulePendingPlacementGroups();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 0);
  mock_placement_group_scheduler_->placement_groups_.clear();

  // Make sure we can re-remove again.
  gcs_placement_group_manager_->RemovePlacementGroup(
      placement_group_id, [](const Status &status) { ASSERT_TRUE(status.ok()); });
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::PENDING), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::CREATED), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::REMOVED), 1);
}

TEST_F(GcsPlacementGroupManagerTest, TestRemovingCreatedPlacementGroup) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();

  // We have ensured that this operation is synchronized.
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);

  const auto &placement_group_id = placement_group->GetPlacementGroupID();
  EXPECT_CALL(*mock_placement_group_scheduler_,
              DestroyPlacementGroupBundleResourcesIfExists(placement_group_id))
      .Times(1);
  EXPECT_CALL(*mock_placement_group_scheduler_, MarkScheduleCancelled(placement_group_id))
      .Times(0);
  gcs_placement_group_manager_->RemovePlacementGroup(placement_group_id,
                                                     [](const Status &status) {});
  RunIOService();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::REMOVED);

  // Make sure it is not rescheduled
  gcs_placement_group_manager_->SchedulePendingPlacementGroups();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 0);
  mock_placement_group_scheduler_->placement_groups_.clear();

  // Make sure we can re-remove again.
  gcs_placement_group_manager_->RemovePlacementGroup(
      placement_group_id, [](const Status &status) { ASSERT_TRUE(status.ok()); });
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::PENDING), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::CREATED), 0);
  ASSERT_EQ(counter_->Get(rpc::PlacementGroupTableData::REMOVED), 1);
}

TEST_F(GcsPlacementGroupManagerTest, TestReschedulingRetry) {
  ///
  /// Test when the rescheduling is failed, the scheduling is retried.
  /// pg scheduled -> pg created -> node dead ->
  /// pg rescheduled -> rescheduling failed -> retry.
  ///
  auto request1 = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request1, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  OnPlacementGroupCreationSuccess(placement_group);

  // Placement group is now rescheduled because bundles are killed.
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(0);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  const auto &bundles =
      mock_placement_group_scheduler_->placement_groups_[0]->GetBundles();
  EXPECT_TRUE(NodeID::FromBinary(bundles[0]->GetMessage().node_id()).IsNil());
  EXPECT_FALSE(NodeID::FromBinary(bundles[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // Rescheduling failed. It should be retried.
  placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
      placement_group, GetExpBackOff(), true);
  ASSERT_TRUE(WaitForCondition(
      [this]() {
        RunIOService();
        return mock_placement_group_scheduler_->GetPlacementGroupCount() == 1;
      },
      10 * 1000));
  // Verify the pg scheduling is retried when its state is RESCHEDULING.
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_[0]->GetPlacementGroupID(),
            placement_group->GetPlacementGroupID());
}

TEST_F(GcsPlacementGroupManagerTest, TestRescheduleWhenNodeDead) {
  ///
  /// Test the basic case.
  /// pg scheduled -> pg created -> node dead -> pg rescheduled.
  ///
  auto request1 = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request1, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  OnPlacementGroupCreationSuccess(placement_group);

  // If a node dies, we will set the bundles above it to be unplaced and reschedule the
  // placement group. The placement group state is set to `RESCHEDULING`
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(0);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_[0]->GetPlacementGroupID(),
            placement_group->GetPlacementGroupID());
  const auto &bundles = placement_group->GetBundles();
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  EXPECT_TRUE(NodeID::FromBinary(bundles[0]->GetMessage().node_id()).IsNil());
  EXPECT_FALSE(NodeID::FromBinary(bundles[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // Test placement group rescheduling success.
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
}

TEST_F(GcsPlacementGroupManagerTest, TestNodeDeadBeforePlacementGroupCreated) {
  ///
  /// Test the case where a node dies before the placement group is created.
  /// pg scheduled -> node dead -> pg created -> pg rescheduled.
  ///
  auto request1 = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request1, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  PrepareBundleResources(placement_group);

  // Node dies before the placement group is created.
  // Expect the placement group state continues to be PREPARED.
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(0);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  const auto &bundles = placement_group->GetBundles();
  EXPECT_TRUE(NodeID::FromBinary(bundles[0]->GetMessage().node_id()).IsNil());
  EXPECT_FALSE(NodeID::FromBinary(bundles[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::PREPARED);

  // Test placement group rescheduling success.
  CommitBundleResources(placement_group);
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_[0]->GetPlacementGroupID(),
            placement_group->GetPlacementGroupID());
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
}

TEST_F(GcsPlacementGroupManagerTest, TestTwoNodesWithBundlesFromSamePlacementGroupDie1) {
  ///
  /// Test the first scenario of the case where two nodes with bundles from the same
  /// placement group die consecutively.
  /// pg scheduled -> pg created -> node1 dead -> pg rescheduled
  /// -> bundles on node1 prepared -> node2 dead -> pg still in prepared state ->
  /// -> bundles on node1 created -> pg rescheduled (for bundles on node2) -> pg created
  ///

  auto request1 = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request1, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  OnPlacementGroupCreationSuccess(placement_group);

  // Node 1 dies. Assuming Node 1 has bundle 0. Node 2 has bundle 1.
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(0);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_[0]->GetPlacementGroupID(),
            placement_group->GetPlacementGroupID());
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  const auto &bundles1 = placement_group->GetBundles();
  EXPECT_TRUE(NodeID::FromBinary(bundles1[0]->GetMessage().node_id()).IsNil());
  EXPECT_FALSE(NodeID::FromBinary(bundles1[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // Bundles on node1 are prepared.
  PrepareBundleResourcesWithIndex(placement_group, {0});
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::PREPARED);

  // Node 2 dies.
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.pop_back();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(1);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  const auto &bundles2 = placement_group->GetBundles();
  EXPECT_FALSE(NodeID::FromBinary(bundles2[0]->GetMessage().node_id()).IsNil());
  EXPECT_TRUE(NodeID::FromBinary(bundles2[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::PREPARED);

  // Complete the placement group creation for bundles in node1
  // Placement group state should be set to RESCHEDULING to reschedule bundles on node2
  CommitBundleResources(placement_group);
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_[0]->GetPlacementGroupID(),
            placement_group->GetPlacementGroupID());
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // Complete the placement group creation for bundles in node2
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
}

TEST_F(GcsPlacementGroupManagerTest, TestTwoNodesWithBundlesFromSamePlacementGroupDie2) {
  ///
  /// Test the second scenario of the case where two nodes with bundles from the same
  /// placement group die consecutively.
  /// pg scheduled -> pg created -> node1 dead -> pg rescheduled
  /// -> all prepare requests returned -> node2 dead -> pg still in rescheduled state
  /// -> pg prepared -> bundles on node1 created -> pg rescheduled (for bundles on node2)
  /// -> pg created
  ///
  auto request1 = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request1, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  OnPlacementGroupCreationSuccess(placement_group);

  // Node 1 dies. Assuming Node 1 has bundle 0. Node 2 has bundle 1.
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(0);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_[0]->GetPlacementGroupID(),
            placement_group->GetPlacementGroupID());
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  const auto &bundles1 = placement_group->GetBundles();
  EXPECT_TRUE(NodeID::FromBinary(bundles1[0]->GetMessage().node_id()).IsNil());
  EXPECT_FALSE(NodeID::FromBinary(bundles1[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // All prepare requests returned.
  MockReceivePrepareRequestWithBundleIndexes(placement_group, {0});

  // Node 2 dies.
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.pop_back();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(1);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 0);
  const auto &bundles2 = placement_group->GetBundles();
  EXPECT_FALSE(NodeID::FromBinary(bundles2[0]->GetMessage().node_id()).IsNil());
  EXPECT_TRUE(NodeID::FromBinary(bundles2[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // Complete the placement group creation for bundles in Node 1
  placement_group->UpdateState(rpc::PlacementGroupTableData::PREPARED);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::PREPARED);
  CommitBundleResources(placement_group);
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_[0]->GetPlacementGroupID(),
            placement_group->GetPlacementGroupID());
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // Complete the placement group creation for bundles in Node 2
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
}

TEST_F(GcsPlacementGroupManagerTest, TestTwoNodesWithBundlesFromSamePlacementGroupDie3) {
  ///
  /// Test the third scenario of the case where two nodes with bundles from the same
  /// placement group die consecutively.
  /// pg scheduled -> pg created -> node1 dead -> pg rescheduled -> node2 dead
  /// -> pg still in rescheduled state -> pg prepared -> pg created
  ///
  auto request1 = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request1, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  OnPlacementGroupCreationSuccess(placement_group);

  // Node 1 dies. Assuming Node 1 has bundle 0. Node 2 has bundle 1.
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(0);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_[0]->GetPlacementGroupID(),
            placement_group->GetPlacementGroupID());
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  const auto &bundles1 = placement_group->GetBundles();
  EXPECT_TRUE(NodeID::FromBinary(bundles1[0]->GetMessage().node_id()).IsNil());
  EXPECT_FALSE(NodeID::FromBinary(bundles1[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // Node 2 dies.
  mock_placement_group_scheduler_->group_on_dead_node_ =
      placement_group->GetPlacementGroupID();
  mock_placement_group_scheduler_->bundles_on_dead_node_.pop_back();
  mock_placement_group_scheduler_->bundles_on_dead_node_.push_back(1);
  gcs_placement_group_manager_->OnNodeDead(NodeID::FromRandom());
  RunIOService();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 0);
  const auto &bundles2 = placement_group->GetBundles();
  EXPECT_TRUE(NodeID::FromBinary(bundles2[0]->GetMessage().node_id()).IsNil());
  EXPECT_TRUE(NodeID::FromBinary(bundles2[1]->GetMessage().node_id()).IsNil());
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::RESCHEDULING);

  // Complete the placement group creation for both bundles
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
}

/// TODO(sang): Currently code is badly structured that it is difficult to test
/// the following scenarios. We should rewrite some APIs and handle them.
/// 1. Node is dead before finishing to create a pg
/// (in this case, we should cancel the in-flight scheduling
/// and prioritze rescheduling to avoid partially allocated pg,
/// 2. While doing rescheduling, an additional node is dead.
/// relevant: https://github.com/ray-project/ray/pull/24875

TEST_F(GcsPlacementGroupManagerTest, TestSchedulerReinitializeAfterGcsRestart) {
  // Create a placement group and make sure it has been created successfully.
  auto job_id = JobID::FromInt(1);
  auto request = GenCreatePlacementGroupRequest(
      /* name */ "",
      rpc::PlacementStrategy::SPREAD,
      /* bundles_count */ 2,
      /* cpu_num */ 1.0,
      /* job_id */ job_id);
  auto job_table_data = GenJobTableData(job_id);
  gcs_table_storage_->JobTable().Put(job_id, *job_table_data, {[](auto) {}, io_service_});
  std::atomic<int> registered_placement_group_count{0};
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);

  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  placement_group->GetMutableBundle(0)->set_node_id(NodeID::FromRandom().Binary());
  placement_group->GetMutableBundle(1)->set_node_id(NodeID::FromRandom().Binary());
  mock_placement_group_scheduler_->placement_groups_.pop_back();
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  // Reinitialize the placement group manager and test the node dead case.
  auto gcs_init_data = LoadDataFromDataStorage();
  ASSERT_EQ(1, gcs_init_data->PlacementGroups().size());
  EXPECT_TRUE(
      gcs_init_data->PlacementGroups().find(placement_group->GetPlacementGroupID()) !=
      gcs_init_data->PlacementGroups().end());
  EXPECT_CALL(*mock_placement_group_scheduler_, ReleaseUnusedBundles(_)).Times(1);
  EXPECT_CALL(
      *mock_placement_group_scheduler_,
      Initialize(testing::Contains(testing::Key(placement_group->GetPlacementGroupID())),
                 /*prepared_pgs=*/testing::IsEmpty()))
      .Times(1);
  gcs_placement_group_manager_->Initialize(*gcs_init_data);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
}

TEST_F(GcsPlacementGroupManagerTest, TestAutomaticCleanupWhenActorDeadAndJobDead) {
  // Test the scenario where actor dead -> job dead.
  const auto job_id = JobID::FromInt(1);
  const auto actor_id = ActorID::Of(job_id, TaskID::Nil(), 0);
  auto request = GenCreatePlacementGroupRequest(
      /* name */ "",
      rpc::PlacementStrategy::SPREAD,
      /* bundles_count */ 2,
      /* cpu_num */ 1.0,
      /* job_id */ job_id,
      /* actor_id */ actor_id);
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  auto placement_group_id = placement_group->GetPlacementGroupID();
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  // When both job and actor is dead, placement group should be destroyed.
  EXPECT_CALL(*mock_placement_group_scheduler_,
              DestroyPlacementGroupBundleResourcesIfExists(placement_group_id))
      .Times(0);
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenActorDead(actor_id);
  RunIOService();
  // Placement group shouldn't be cleaned when only an actor is killed.
  // When both job and actor is dead, placement group should be destroyed.
  EXPECT_CALL(*mock_placement_group_scheduler_,
              DestroyPlacementGroupBundleResourcesIfExists(placement_group_id))
      .Times(1);
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenJobDead(job_id);
  RunIOService();
}

TEST_F(GcsPlacementGroupManagerTest, TestAutomaticCleanupWhenActorAndJobDead) {
  // Test the scenario where job dead -> actor dead.
  const auto job_id = JobID::FromInt(1);
  const auto actor_id = ActorID::Of(job_id, TaskID::Nil(), 0);
  auto request = GenCreatePlacementGroupRequest(
      /* name */ "",
      rpc::PlacementStrategy::SPREAD,
      /* bundles_count */ 2,
      /* cpu_num */ 1.0,
      /* job_id */ job_id,
      /* actor_id */ actor_id);
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  auto placement_group_id = placement_group->GetPlacementGroupID();
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  EXPECT_CALL(*mock_placement_group_scheduler_,
              DestroyPlacementGroupBundleResourcesIfExists(placement_group_id))
      .Times(0);
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenJobDead(job_id);
  RunIOService();
  // Placement group shouldn't be cleaned when only an actor is killed.
  EXPECT_CALL(*mock_placement_group_scheduler_,
              DestroyPlacementGroupBundleResourcesIfExists(placement_group_id))
      .Times(1);
  // This method should ensure idempotency.
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenActorDead(actor_id);
  RunIOService();
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenActorDead(actor_id);
  RunIOService();
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenActorDead(actor_id);
  RunIOService();
}

TEST_F(GcsPlacementGroupManagerTest, TestAutomaticCleanupWhenOnlyJobDead) {
  // Test placement group is cleaned when both actor & job are dead.
  const auto job_id = JobID::FromInt(1);
  auto request = GenCreatePlacementGroupRequest(
      /* name */ "",
      rpc::PlacementStrategy::SPREAD,
      /* bundles_count */ 2,
      /* cpu_num */ 1.0,
      /* job_id */ job_id,
      /* actor_id */ ActorID::Nil());
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  auto placement_group_id = placement_group->GetPlacementGroupID();
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  EXPECT_CALL(*mock_placement_group_scheduler_,
              DestroyPlacementGroupBundleResourcesIfExists(placement_group_id))
      .Times(1);
  // This method should ensure idempotency.
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenJobDead(job_id);
  RunIOService();
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenJobDead(job_id);
  RunIOService();
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenJobDead(job_id);
  RunIOService();
}

TEST_F(GcsPlacementGroupManagerTest,
       TestAutomaticCleanupDoNothingWhenDifferentJobIsDead) {
  // Test placement group is cleaned when both actor & job are dead.
  const auto job_id = JobID::FromInt(1);
  const auto different_job_id = JobID::FromInt(3);
  auto request = GenCreatePlacementGroupRequest(
      /* name */ "",
      rpc::PlacementStrategy::SPREAD,
      /* bundles_count */ 2,
      /* cpu_num */ 1.0,
      /* job_id */ job_id,
      /* actor_id */ ActorID::Nil());
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  auto placement_group_id = placement_group->GetPlacementGroupID();
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  // This shouldn't have been called.
  EXPECT_CALL(*mock_placement_group_scheduler_,
              DestroyPlacementGroupBundleResourcesIfExists(placement_group_id))
      .Times(0);
  // This method should ensure idempotency.
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenJobDead(different_job_id);
  RunIOService();
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenJobDead(different_job_id);
  RunIOService();
  gcs_placement_group_manager_->CleanPlacementGroupIfNeededWhenJobDead(different_job_id);
  RunIOService();
}

TEST_F(GcsPlacementGroupManagerTest, TestSchedulingCanceledWhenPgIsInfeasible) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request,
                         [&registered_placement_group_count](const Status &status) {
                           ++registered_placement_group_count;
                         });

  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.clear();

  // Mark it non-retryable.
  gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
      placement_group, GetExpBackOff(), false);
  ASSERT_EQ(placement_group->GetStats().scheduling_state(),
            rpc::PlacementGroupStats::INFEASIBLE);

  // Schedule twice to make sure it will not be scheduled afterward.
  gcs_placement_group_manager_->SchedulePendingPlacementGroups();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 0);
  gcs_placement_group_manager_->SchedulePendingPlacementGroups();
  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 0);

  // Add a node and make sure it will reschedule the infeasible placement group.
  const auto &node_id = NodeID::FromRandom();
  gcs_placement_group_manager_->OnNodeAdd(node_id);
  RunIOService();

  ASSERT_EQ(mock_placement_group_scheduler_->placement_groups_.size(), 1);
  mock_placement_group_scheduler_->placement_groups_.clear();

  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  ASSERT_EQ(placement_group->GetStats().scheduling_state(),
            rpc::PlacementGroupStats::FINISHED);
}

TEST_F(GcsPlacementGroupManagerTest, TestRayNamespace) {
  auto request1 = GenCreatePlacementGroupRequest("test_name");
  job_namespace_table_[JobID::FromInt(11)] = "another_namespace";
  auto request2 = GenCreatePlacementGroupRequest(
      "test_name", rpc::PlacementStrategy::SPREAD, 2, 1.0, JobID::FromInt(11));
  auto request3 = GenCreatePlacementGroupRequest("test_name");
  {  // Create a placement group in the empty namespace.
    std::atomic<int> registered_placement_group_count(0);
    RegisterPlacementGroup(request1, [&registered_placement_group_count](Status status) {
      ++registered_placement_group_count;
    });

    ASSERT_EQ(registered_placement_group_count, 1);
    ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
    auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
    mock_placement_group_scheduler_->placement_groups_.pop_back();

    OnPlacementGroupCreationSuccess(placement_group);
    ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
    ASSERT_EQ(gcs_placement_group_manager_->GetPlacementGroupIDByName("test_name", ""),
              PlacementGroupID::FromBinary(
                  request1.placement_group_spec().placement_group_id()));
  }
  {  // Create a placement group in the empty namespace.
    job_namespace_table_[JobID::FromInt(11)] = "another_namespace";
    std::atomic<int> registered_placement_group_count(0);
    RegisterPlacementGroup(request2, [&registered_placement_group_count](Status status) {
      ++registered_placement_group_count;
    });

    ASSERT_EQ(registered_placement_group_count, 1);
    ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
    auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
    mock_placement_group_scheduler_->placement_groups_.pop_back();

    OnPlacementGroupCreationSuccess(placement_group);
    ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
    ASSERT_EQ(gcs_placement_group_manager_->GetPlacementGroupIDByName(
                  "test_name", "another_namespace"),
              PlacementGroupID::FromBinary(
                  request2.placement_group_spec().placement_group_id()));
    ASSERT_NE(gcs_placement_group_manager_->GetPlacementGroupIDByName(
                  "test_name", "another_namespace"),
              PlacementGroupID::FromBinary(
                  request1.placement_group_spec().placement_group_id()));
  }
  {  // Placement groups with the same namespace, different jobs should still collide.
    std::promise<void> promise;
    gcs_placement_group_manager_->RegisterPlacementGroup(
        std::make_shared<gcs::GcsPlacementGroup>(request3, "", counter_),
        [&promise](Status status) {
          ASSERT_FALSE(status.ok());
          promise.set_value();
        });
    RunIOService();
    promise.get_future().get();

    ASSERT_EQ(gcs_placement_group_manager_->GetPlacementGroupIDByName("test_name", ""),
              PlacementGroupID::FromBinary(
                  request1.placement_group_spec().placement_group_id()));
  }
}

TEST_F(GcsPlacementGroupManagerTest, TestStats) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  RegisterPlacementGroup(request,
                         [&registered_placement_group_count](const Status &status) {
                           ++registered_placement_group_count;
                         });

  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.clear();

  /// Feasible, but still failing.
  {
    ASSERT_EQ(placement_group->GetStats().scheduling_attempt(), 1);
    ASSERT_EQ(placement_group->GetStats().scheduling_state(),
              rpc::PlacementGroupStats::QUEUED);
    gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
        placement_group, GetExpBackOff(), /*is_feasible*/ true);
    ASSERT_TRUE(WaitForCondition(
        [this]() {
          RunIOService();
          return mock_placement_group_scheduler_->GetPlacementGroupCount() == 1;
        },
        10 * 1000));
    auto last_placement_group = mock_placement_group_scheduler_->placement_groups_.back();
    mock_placement_group_scheduler_->placement_groups_.clear();
    ASSERT_EQ(last_placement_group->GetStats().scheduling_state(),
              rpc::PlacementGroupStats::NO_RESOURCES);
    ASSERT_EQ(last_placement_group->GetStats().scheduling_attempt(), 2);
  }

  /// Feasible, but failed to commit resources.
  {
    placement_group->UpdateState(rpc::PlacementGroupTableData::RESCHEDULING);
    gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
        placement_group, GetExpBackOff(), /*is_feasible*/ true);
    ASSERT_TRUE(WaitForCondition(
        [this]() {
          RunIOService();
          return mock_placement_group_scheduler_->GetPlacementGroupCount() == 1;
        },
        10 * 1000));
    auto last_placement_group = mock_placement_group_scheduler_->placement_groups_.back();
    mock_placement_group_scheduler_->placement_groups_.clear();
    ASSERT_EQ(last_placement_group->GetStats().scheduling_state(),
              rpc::PlacementGroupStats::FAILED_TO_COMMIT_RESOURCES);
    ASSERT_EQ(last_placement_group->GetStats().scheduling_attempt(), 3);
  }

  // Check that the placement_group scheduling state is `FINISHED`.
  {
    OnPlacementGroupCreationSuccess(placement_group);
    ASSERT_EQ(placement_group->GetStats().scheduling_state(),
              rpc::PlacementGroupStats::FINISHED);
    ASSERT_EQ(placement_group->GetStats().scheduling_attempt(), 3);
  }
}

TEST_F(GcsPlacementGroupManagerTest, TestStatsCreationTime) {
  auto request = GenCreatePlacementGroupRequest();
  std::atomic<int> registered_placement_group_count(0);
  auto request_received_ns = absl::GetCurrentTimeNanos();
  RegisterPlacementGroup(request,
                         [&registered_placement_group_count](const Status &status) {
                           ++registered_placement_group_count;
                         });
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);
  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  mock_placement_group_scheduler_->placement_groups_.clear();

  /// Failed to create a pg.
  gcs_placement_group_manager_->OnPlacementGroupCreationFailed(
      placement_group, GetExpBackOff(), /*is_feasible*/ true);
  auto scheduling_started_ns = absl::GetCurrentTimeNanos();
  ASSERT_TRUE(WaitForCondition(
      [this]() {
        RunIOService();
        return mock_placement_group_scheduler_->GetPlacementGroupCount() == 1;
      },
      10 * 1000));

  OnPlacementGroupCreationSuccess(placement_group);
  auto scheduling_done_ns = absl::GetCurrentTimeNanos();

  /// Make sure the creation time is correctly recorded.
  ASSERT_NE(placement_group->GetStats().scheduling_latency_us(), 0);
  ASSERT_NE(placement_group->GetStats().end_to_end_creation_latency_us(), 0);
  // The way to measure latency is a little brittle now. Alternatively, we can mock
  // the absl::GetCurrentNanos() to a callback method and have more accurate test.
  auto scheduling_latency_us =
      absl::Nanoseconds(scheduling_done_ns - scheduling_started_ns) /
      absl::Microseconds(1);
  auto end_to_end_creation_latency_us =
      absl::Nanoseconds(scheduling_done_ns - request_received_ns) / absl::Microseconds(1);
  ASSERT_TRUE(placement_group->GetStats().scheduling_latency_us() <
              scheduling_latency_us);
  ASSERT_TRUE(placement_group->GetStats().end_to_end_creation_latency_us() <
              end_to_end_creation_latency_us);
}

TEST_F(GcsPlacementGroupManagerTest, TestGetAllPlacementGroupInfoLimit) {
  auto num_pgs = 3;
  std::atomic<int> registered_placement_group_count(0);
  for (int i = 0; i < num_pgs; i++) {
    auto request = GenCreatePlacementGroupRequest();
    RegisterPlacementGroup(request,
                           [&registered_placement_group_count](const Status &status) {
                             ++registered_placement_group_count;
                           });
  }
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);

  {
    rpc::GetAllPlacementGroupRequest request;
    rpc::GetAllPlacementGroupReply reply;
    std::promise<void> promise;
    auto callback = [&promise](Status status,
                               std::function<void()> success,
                               std::function<void()> failure) { promise.set_value(); };
    gcs_placement_group_manager_->HandleGetAllPlacementGroup(request, &reply, callback);
    RunIOService();
    promise.get_future().get();
    ASSERT_EQ(reply.placement_group_table_data().size(), 3);
    ASSERT_EQ(reply.total(), 3);
  }
  {
    rpc::GetAllPlacementGroupRequest request;
    rpc::GetAllPlacementGroupReply reply;
    request.set_limit(2);
    std::promise<void> promise;
    auto callback = [&promise](Status status,
                               std::function<void()> success,
                               std::function<void()> failure) { promise.set_value(); };
    gcs_placement_group_manager_->HandleGetAllPlacementGroup(request, &reply, callback);
    RunIOService();
    promise.get_future().get();
    ASSERT_EQ(reply.placement_group_table_data().size(), 2);
    ASSERT_EQ(reply.total(), 3);
  }
}

TEST_F(GcsPlacementGroupManagerTest, TestCheckCreatorJobIsDeadWhenGcsRestart) {
  auto job_id = JobID::FromInt(1);
  auto request = GenCreatePlacementGroupRequest(
      /* name */ "",
      rpc::PlacementStrategy::SPREAD,
      /* bundles_count */ 2,
      /* cpu_num */ 1.0,
      /* job_id */ job_id);
  auto job_table_data = GenJobTableData(job_id);
  job_table_data->set_is_dead(true);
  gcs_table_storage_->JobTable().Put(job_id, *job_table_data, {[](auto) {}, io_service_});
  std::atomic<int> registered_placement_group_count{0};
  RegisterPlacementGroup(request, [&registered_placement_group_count](Status status) {
    ++registered_placement_group_count;
  });
  ASSERT_EQ(registered_placement_group_count, 1);
  ASSERT_EQ(mock_placement_group_scheduler_->GetPlacementGroupCount(), 1);

  auto placement_group = mock_placement_group_scheduler_->placement_groups_.back();
  OnPlacementGroupCreationSuccess(placement_group);
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::CREATED);
  // Reinitialize the placement group manager and the job is dead.
  auto gcs_init_data = LoadDataFromDataStorage();
  ASSERT_EQ(1, gcs_init_data->PlacementGroups().size());
  EXPECT_TRUE(
      gcs_init_data->PlacementGroups().find(placement_group->GetPlacementGroupID()) !=
      gcs_init_data->PlacementGroups().end());
  EXPECT_CALL(
      *mock_placement_group_scheduler_,
      Initialize(testing::Contains(testing::Key(placement_group->GetPlacementGroupID())),
                 /*prepared_pgs=*/testing::IsEmpty()))
      .Times(1);
  gcs_placement_group_manager_->Initialize(*gcs_init_data);
  // Make sure placement group is removed after gcs restart for the creator job is dead
  ASSERT_EQ(placement_group->GetState(), rpc::PlacementGroupTableData::REMOVED);
}

}  // namespace gcs
}  // namespace ray
