// Copyright 2022 The Ray Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "ray/rpc/server_call.h"

#include <atomic>
#include <memory>

#include "ray/common/ray_config.h"

namespace ray {
namespace rpc {
namespace {

std::atomic<ServerCallThreadPoolMode> &ThreadPoolMode() {
  static std::atomic<ServerCallThreadPoolMode> mode(
      ServerCallThreadPoolMode::SYSTEM_COMPONENT);
  return mode;
}

std::unique_ptr<boost::asio::thread_pool> &_GetServerCallExecutor() {
  static auto thread_pool = std::make_unique<boost::asio::thread_pool>(
      ThreadPoolMode().load() == ServerCallThreadPoolMode::CORE_WORKER
          ? ::RayConfig::instance().core_worker_num_server_call_thread()
          : ::RayConfig::instance().num_server_call_thread());
  return thread_pool;
}

}  // namespace

boost::asio::thread_pool &GetServerCallExecutor() { return *_GetServerCallExecutor(); }

void DrainServerCallExecutor() { GetServerCallExecutor().join(); }

void ResetServerCallExecutor() {
  _GetServerCallExecutor() = std::make_unique<boost::asio::thread_pool>(
      ThreadPoolMode().load() == ServerCallThreadPoolMode::CORE_WORKER
          ? ::RayConfig::instance().core_worker_num_server_call_thread()
          : ::RayConfig::instance().num_server_call_thread());
}

void SetServerCallThreadPoolMode(ServerCallThreadPoolMode mode) {
  ThreadPoolMode().store(mode);
}

}  // namespace rpc
}  // namespace ray
