from SpiffWorkflow import TaskState
from SpiffWorkflow.bpmn import BpmnWorkflow

from .BpmnWorkflowTestCase import BpmnWorkflowTestCase


class ResetTokenOnBoundaryEventTest(BpmnWorkflowTestCase):
    """Assure that when we reset a token to a previous task, and that
    task has a boundary event, that the boundary event is reset to the
    correct state."""

    def setUp(self):
        spec, subprocesses = self.load_workflow_spec('reset_with_boundary_event.bpmn', 'token')
        self.workflow = BpmnWorkflow(spec, subprocesses)

    def testResetToOuterWorkflow(self):
        self.reset_to_outer_workflow(save_restore=False)

    def testResetToSubprocess(self):
        self.reset_to_subprocess(save_restore=False)

    def testSaveRestore(self):
        self.reset_to_outer_workflow(save_restore=True)

    def reset_to_outer_workflow(self, save_restore=False):

        # Advance insie the subworkflow
        self.advance_to_task('Last')
        sub = self.workflow.get_next_task(spec_name='subprocess')
        timer_event = self.workflow.get_next_task(spec_name='Event_My_Timer')
        self.assertEqual(TaskState.CANCELLED, timer_event.state)

        if save_restore:
            self.save_restore()

        # Here we reset back to the first task
        first = self.workflow.get_next_task(spec_name='First')
        self.workflow.reset_from_task_id(first.id)

        if save_restore:
            self.save_restore()

        # At which point, the timer event should return to a waiting state, the subprocess shoud have been removed
        task = self.workflow.get_next_task(spec_name='First')
        self.assertEqual(task.state, TaskState.READY)
        timer_event = self.workflow.get_next_task(spec_name='Event_My_Timer')
        self.assertEqual(timer_event.state, TaskState.WAITING)
        self.assertNotIn(sub.id, self.workflow.subprocesses)

        # Ensure the workflow can be completed without being stuck on stranded tasks
        self.complete_workflow()
        self.assertTrue(self.workflow.completed)

    def reset_to_subprocess(self, save_restore=False):

        # Advance past the subworkflow
        self.advance_to_task('Final')
        if save_restore:
            self.save_restore()

        # Reset to a task inside the subworkflow
        task = self.workflow.get_next_task(spec_name='Last')
        self.workflow.reset_from_task_id(task.id)

        if save_restore:
            self.save_restore()

        # The task we returned to should be ready, the subprocess should be waiting, the final task should be future
        sub = self.workflow.get_next_task(spec_name='subprocess')
        self.assertEqual(sub.state, TaskState.STARTED)
        self.assertEqual(task.state, TaskState.READY)
        final = self.workflow.get_next_task(spec_name='Final')
        self.assertEqual(final.state, TaskState.FUTURE)

        # Ensure the workflow can be completed without being stuck on stranded tasks        
        self.complete_workflow()
        self.assertTrue(self.workflow.completed)

    def advance_to_task(self, name):

        self.workflow.do_engine_steps()
        ready_tasks = self.workflow.get_tasks(state=TaskState.READY)
        while ready_tasks[0].task_spec.name != name:
            ready_tasks[0].run()
            self.workflow.do_engine_steps()
            self.workflow.refresh_waiting_tasks()
            ready_tasks = self.workflow.get_tasks(state=TaskState.READY)

    def complete_workflow(self):

        ready_tasks = self.workflow.get_tasks(state=TaskState.READY)
        while len(ready_tasks) > 0:
            ready_tasks[0].run()
            self.workflow.do_engine_steps()
            self.workflow.refresh_waiting_tasks()
            ready_tasks = self.workflow.get_tasks(state=TaskState.READY)
