# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from playwright.sync_api import Page, expect

from e2e_playwright.conftest import ImageCompareFunction, wait_for_app_run
from e2e_playwright.shared.app_utils import click_checkbox, goto_app


def test_desktop_top_nav(app: Page):
    """Test top navigation on desktop viewport."""
    # Set desktop viewport
    app.set_viewport_size({"width": 1280, "height": 800})

    # Default configuration shows top nav with 3 pages
    wait_for_app_run(app)

    # The top nav is rendered using rc-overflow component
    # Check that navigation links exist and are visible
    nav_links = app.get_by_test_id("stTopNavLink")
    expect(nav_links).to_have_count(3)  # 3 pages

    # Verify no sidebar is visible
    sidebar = app.get_by_test_id("stSidebar")
    expect(sidebar).not_to_be_visible()

    # Click second page
    nav_links.nth(1).click()
    wait_for_app_run(app)

    # Verify page content changed - be specific about which element
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 2")).to_be_visible()

    # Verify active state - check if the link has the active class or is selected
    # The active nav link should have some visual indication
    second_link = nav_links.nth(1)
    # Try different ways to check active state
    expect(second_link).to_be_visible()  # Just verify it's visible for now


def test_mobile_fallback_to_sidebar(app: Page):
    """Test that top nav falls back to sidebar on mobile."""
    # Set mobile viewport
    app.set_viewport_size({"width": 375, "height": 667})

    wait_for_app_run(app)

    # On mobile with AUTO state, sidebar should be collapsed by default
    sidebar = app.get_by_test_id("stSidebar")
    expect(sidebar).to_have_attribute("aria-expanded", "false")

    # Expand the sidebar to access navigation
    expand_button = app.get_by_test_id("stExpandSidebarButton")
    expand_button.click()

    # Wait for sidebar to expand and nav links to be visible
    expect(sidebar).to_have_attribute("aria-expanded", "true")
    nav_links = app.get_by_test_id("stSidebarNavLink")
    expect(nav_links.first).to_be_visible()

    # Test navigation functionality
    nav_links.nth(2).click()
    wait_for_app_run(app)

    # Verify content updated - be specific
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 3")).to_be_visible()


def test_overflow_behavior(app: Page):
    """Test overflow menu when too many pages for viewport."""
    # Enable overflow test mode
    click_checkbox(app, "Test Overflow (5 pages)")
    wait_for_app_run(app)

    # Set medium viewport that might trigger overflow
    app.set_viewport_size({"width": 800, "height": 600})

    # Verify we have 5 nav links total
    nav_links = app.get_by_test_id("stTopNavLink")
    expect(nav_links).to_have_count(5)

    # Note: Due to our mock of rc-overflow in the JS tests, all links will be visible
    # In real implementation, some would be hidden and overflow menu would appear
    # Check that the overflow container exists
    overflow_items = app.locator(".rc-overflow-item")
    expect(overflow_items.first).to_be_visible()


def test_top_nav_with_sections(app: Page):
    """Test top navigation with section headers."""
    app.set_viewport_size({"width": 1280, "height": 800})

    # Enable sections test mode
    click_checkbox(app, "Test Sections")
    wait_for_app_run(app)

    # When sections are used, section names become the top-level nav items
    # Verify sections are rendered as clickable items
    section_a_trigger = app.get_by_text("Section A").first
    section_b_trigger = app.get_by_text("Section B").first
    expect(section_a_trigger).to_be_visible()
    expect(section_b_trigger).to_be_visible()

    # Click section A to open dropdown/popover
    section_a_trigger.click()

    # Wait for pages to become visible after clicking section
    page1_in_popover = app.get_by_role("link", name="Page 1")
    page2_in_popover = app.get_by_role("link", name="Page 2")
    expect(page1_in_popover).to_be_visible()
    expect(page2_in_popover).to_be_visible()

    # Click a page in the popover
    page2_in_popover.click()
    wait_for_app_run(app)

    # Verify navigation worked - check the header
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 2")).to_be_visible()


def test_top_nav_with_single_section(app: Page):
    """Test top navigation with a single section containing 3 pages."""
    app.set_viewport_size({"width": 1280, "height": 800})

    # Enable single section test mode
    click_checkbox(app, "Test Single Section (3 pages)")
    wait_for_app_run(app)

    # When a single section is used, the section name should be in the top nav
    section_trigger = app.get_by_text("My Section").first
    expect(section_trigger).to_be_visible()

    # Click the section to open dropdown/popover
    section_trigger.click()

    # Wait for all 3 pages to become visible in the popover
    page1_in_popover = app.get_by_role("link", name="Page 1")
    page2_in_popover = app.get_by_role("link", name="Page 2")
    page3_in_popover = app.get_by_role("link", name="Page 3")

    expect(page1_in_popover).to_be_visible()
    expect(page2_in_popover).to_be_visible()
    expect(page3_in_popover).to_be_visible()

    # Navigate to page 3 to verify navigation works
    page3_in_popover.click()
    wait_for_app_run(app)

    # Verify navigation worked - check the header
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 3")).to_be_visible()

    # Click the section again to verify it still works after navigation
    section_trigger.click()

    # Verify pages are still accessible
    expect(page1_in_popover).to_be_visible()

    # Navigate back to page 1
    page1_in_popover.click()
    wait_for_app_run(app)

    # Verify we're back on page 1
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 1")).to_be_visible()


def test_hidden_navigation_mode(app: Page):
    """Test hidden navigation mode."""
    app.set_viewport_size({"width": 1280, "height": 800})

    # Enable hidden navigation mode
    click_checkbox(app, "Test Hidden Navigation")
    wait_for_app_run(app)

    # No sidebar should be visible
    expect(app.get_by_test_id("stSidebar")).not_to_be_visible()

    # No nav links should be visible
    expect(app.get_by_test_id("stTopNavLink")).not_to_be_visible()

    # Only first page content should be visible - check header
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 1")).to_be_visible()


def test_switching_navigation_modes(app: Page):
    """Test dynamically switching between navigation modes."""
    app.set_viewport_size({"width": 1280, "height": 800})

    # Enable navigation switching mode
    click_checkbox(app, "Test Navigation Switching")
    wait_for_app_run(app)

    # Initially should show sidebar (default)
    expect(app.get_by_test_id("stSidebar")).to_be_visible()

    # Navigate to page 2 first to test state persistence
    sidebar_links = app.get_by_test_id("stSidebarNavLink")
    sidebar_links.nth(1).click()
    wait_for_app_run(app)
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 2")).to_be_visible()

    # Click button to switch to top nav
    app.get_by_role("button", name="Switch to Top Nav").click()
    wait_for_app_run(app)

    # Verify switched to top nav - sidebar hidden, nav links visible at top
    expect(app.get_by_test_id("stSidebar")).not_to_be_visible()
    nav_links = app.get_by_test_id("stTopNavLink")
    expect(nav_links).to_have_count(3)
    expect(nav_links.first).to_be_visible()

    # Verify we're still on page 2 (state persistence)
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 2")).to_be_visible()

    # Navigation should still work in top nav mode
    nav_links.nth(2).click()
    wait_for_app_run(app)
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 3")).to_be_visible()


def test_top_nav_with_logo(app: Page, assert_snapshot: ImageCompareFunction):
    """Tests that the logo with a top navigation is shown in the correct size,
    even when the viewport is narrowed.
    """

    app.set_viewport_size({"width": 1280, "height": 800})

    # Enable features:
    click_checkbox(app, "Test Overflow (5 pages)")
    click_checkbox(app, "Test Sidebar Content")
    click_checkbox(app, "Test Logo")

    # Collapse sidebar
    app.get_by_test_id("stSidebar").hover()
    close_button = app.get_by_test_id("stSidebarCollapseButton")
    expect(close_button).to_be_visible()
    close_button.click()
    expect(app.get_by_test_id("stSidebar")).not_to_be_visible()

    # Wait for logo to be visible
    logo = app.get_by_test_id("stHeaderLogo")
    expect(logo).to_be_visible()

    # Take snapshot of the header with logo at full width
    header = app.locator("header").first
    assert_snapshot(header, name="st_navigation-top_nav_with_logo")

    # Test that logo size is preserved at a narrower viewport
    # This validates the flexShrink: 0 fix on StyledHeaderLeftSection
    app.set_viewport_size({"width": 800, "height": 600})

    # Logo should still be visible and maintain its size
    expect(logo).to_be_visible()
    assert_snapshot(header, name="st_navigation-top_nav_with_logo_narrow_viewport")


def test_top_nav_visual_regression(app: Page, assert_snapshot: ImageCompareFunction):
    """Visual regression test for top navigation."""
    app.set_viewport_size({"width": 1280, "height": 800})

    wait_for_app_run(app)

    # Wait for app to stabilize
    nav_links = app.get_by_test_id("stTopNavLink")
    expect(nav_links.first).to_be_visible()

    # Take screenshot of the navigation area
    # Since there's no specific top nav container, capture the header area
    nav_area = app.locator("header").first
    assert_snapshot(nav_area, name="st_navigation-top_nav_desktop")

    # Test hover state
    nav_links.first.hover()
    assert_snapshot(nav_area, name="st_navigation-top_nav_hover")

    # Test with sections
    click_checkbox(app, "Test Sections")
    wait_for_app_run(app)

    section_a = app.get_by_text("Section A").first
    expect(section_a).to_be_visible()
    section_a.click()

    # Wait for popover to appear using proper expect
    # The dropdown link should be visible after clicking the section
    dropdown_link = app.get_by_test_id("stTopNavDropdownLink").filter(has_text="Page 1")
    expect(dropdown_link).to_be_visible()

    # Get the visible popover container by its test ID
    # There are multiple popovers (one per section), so we need the visible one
    popover = app.get_by_test_id("stTopNavPopover").locator("visible=true")
    assert_snapshot(popover, name="st_navigation-top_nav_section_popover")

    # Test single section
    # First uncheck the Test Sections checkbox
    click_checkbox(app, "Test Sections")
    wait_for_app_run(app)

    # Enable single section test
    click_checkbox(app, "Test Single Section (3 pages)")
    wait_for_app_run(app)

    # Take screenshot of single section in nav
    nav_area = app.locator("header").first
    assert_snapshot(nav_area, name="st_navigation-top_nav_single_section")

    # Click the section to open popover
    single_section_trigger = app.get_by_text("My Section").first
    expect(single_section_trigger).to_be_visible()
    single_section_trigger.click()

    # Wait for popover with 3 pages
    page3_in_popover = app.get_by_role("link", name="Page 3")
    expect(page3_in_popover).to_be_visible()

    # Take screenshot of single section popover
    # Get one of the dropdown links to locate the popover container
    dropdown_link_in_single = app.get_by_test_id("stTopNavDropdownLink").filter(
        has_text="Page 1"
    )
    expect(dropdown_link_in_single).to_be_visible()

    # Get the popover container by its test ID
    # For single section there's only one popover but we use .first for consistency
    single_section_popover = app.get_by_test_id("stTopNavPopover").first
    assert_snapshot(
        single_section_popover, name="st_navigation-top_nav_single_section_popover"
    )


def test_mixed_empty_and_named_sections(app: Page):
    """Test top navigation with mixed empty and named sections.

    This tests the issue #12243 scenario where some sections have names
    and some sections are empty (""). The empty section pages should appear
    at the top level of navigation while named sections should have dropdowns.
    """
    app.set_viewport_size({"width": 1280, "height": 800})

    # Enable mixed sections test mode
    click_checkbox(app, "Test Mixed Empty/Named Sections")
    wait_for_app_run(app)

    # Verify that empty section pages appear as top-level nav items
    home_nav = app.get_by_test_id("stTopNavLink").filter(has_text="Home")
    dashboard_nav = app.get_by_test_id("stTopNavLink").filter(has_text="Dashboard")
    expect(home_nav).to_be_visible()
    expect(dashboard_nav).to_be_visible()

    # Verify that named sections appear as section triggers
    admin_trigger = app.get_by_text("Admin").first
    reports_trigger = app.get_by_text("Reports").first
    expect(admin_trigger).to_be_visible()
    expect(reports_trigger).to_be_visible()

    # Click on a top-level page (from empty section)
    dashboard_nav.click()
    wait_for_app_run(app)
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 2")).to_be_visible()

    # Click on Admin section to open dropdown
    admin_trigger.click()

    # Wait for Admin section pages to be visible
    settings_link = app.get_by_role("link", name="Settings")
    users_link = app.get_by_role("link", name="Users")
    expect(settings_link).to_be_visible()
    expect(users_link).to_be_visible()

    # Navigate to Settings page
    settings_link.click()
    wait_for_app_run(app)
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 3")).to_be_visible()

    # Click on Reports section
    reports_trigger.click()

    # Verify Reports section has only one page
    analytics_link = app.get_by_role("link", name="Analytics")
    expect(analytics_link).to_be_visible()

    # Navigate to Analytics
    analytics_link.click()
    wait_for_app_run(app)
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 5")).to_be_visible()

    # Navigate back to a top-level page to verify mixed navigation still works
    home_nav.click()
    wait_for_app_run(app)
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 1")).to_be_visible()


def test_mixed_sections_visual_regression(
    app: Page, assert_snapshot: ImageCompareFunction
):
    """Visual regression test for mixed empty and named sections navigation.

    Verifies the visual appearance of top navigation when configuration includes
    both empty sections (pages at top level) and named sections (with dropdowns).
    Tests navigation bar rendering, dropdown popovers, and hover states.
    """
    app.set_viewport_size({"width": 1280, "height": 800})

    # Enable mixed sections test mode
    click_checkbox(app, "Test Mixed Empty/Named Sections")
    wait_for_app_run(app)

    # Wait for navigation to stabilize
    home_nav = app.get_by_test_id("stTopNavLink").filter(has_text="Home")
    expect(home_nav).to_be_visible()

    # Take screenshot of mixed navigation bar
    nav_area = app.locator("header").first
    assert_snapshot(nav_area, name="st_navigation-mixed_sections_nav_bar")

    # Open Admin dropdown and capture popover
    admin_trigger = app.get_by_text("Admin").first
    admin_trigger.click()

    # Wait for dropdown to open
    settings_link = app.get_by_role("link", name="Settings")
    expect(settings_link).to_be_visible()

    # Capture Admin section popover
    admin_popover = app.get_by_test_id("stTopNavPopover").locator("visible=true")
    assert_snapshot(admin_popover, name="st_navigation-mixed_sections_admin_popover")

    # Close Admin dropdown by clicking elsewhere
    app.get_by_test_id("stMain").click()

    # Open Reports dropdown
    reports_trigger = app.get_by_text("Reports").first
    reports_trigger.click()

    # Wait for Reports dropdown
    analytics_link = app.get_by_role("link", name="Analytics")
    expect(analytics_link).to_be_visible()

    # Capture Reports section popover (single item)
    reports_popover = app.get_by_test_id("stTopNavPopover").locator("visible=true")
    assert_snapshot(
        reports_popover, name="st_navigation-mixed_sections_reports_popover"
    )

    # Test hover state on mixed navigation
    home_nav.hover()
    assert_snapshot(nav_area, name="st_navigation-mixed_sections_hover_home")

    # Hover on a section trigger
    admin_trigger.hover()
    assert_snapshot(nav_area, name="st_navigation-mixed_sections_hover_admin")


def test_mobile_sidebar_overlay_visual(
    app: Page, assert_snapshot: ImageCompareFunction
):
    """Visual regression test for mobile sidebar overlay behavior."""
    # Set mobile viewport
    app.set_viewport_size({"width": 375, "height": 667})

    wait_for_app_run(app)

    # On mobile with AUTO state, sidebar should be collapsed by default
    sidebar = app.get_by_test_id("stSidebar")
    expect(sidebar).to_have_attribute("aria-expanded", "false")

    # Take screenshot of initial collapsed state
    assert_snapshot(app, name="st_navigation-mobile_sidebar_overlay_collapsed")

    # Expand the sidebar to test overlay behavior
    expand_button = app.get_by_test_id("stExpandSidebarButton")
    expand_button.click()

    # Wait for sidebar to expand and verify navigation is visible
    expect(sidebar).to_have_attribute("aria-expanded", "true")
    nav_links = app.get_by_test_id("stSidebarNavLink")
    expect(nav_links).to_have_count(3)
    expect(nav_links.first).to_be_visible()

    # Take screenshot showing sidebar overlaying content
    # Capture the entire viewport to show overlay effect
    assert_snapshot(app, name="st_navigation-mobile_sidebar_overlay_expanded")

    # Test collapsed sidebar state
    # Click the close button to collapse sidebar
    close_button = app.get_by_test_id("stSidebarCollapseButton")
    close_button.click()

    # Wait for sidebar to collapse
    # The sidebar aria-expanded attribute should be false
    expect(sidebar).to_have_attribute("aria-expanded", "false")

    # Test navigation interaction
    # Expand sidebar again using the expand button in the header
    expand_button = app.get_by_test_id("stExpandSidebarButton")
    expand_button.click()

    # Wait for sidebar to expand
    expect(sidebar).to_have_attribute("aria-expanded", "true")
    expect(nav_links.first).to_be_visible()

    # Navigate to a different page
    nav_links.nth(1).click()
    wait_for_app_run(app)

    # Verify navigation worked and content updated behind sidebar
    expect(app.get_by_test_id("stHeading").filter(has_text="Page 2")).to_be_visible()

    # Take screenshot showing different page content with sidebar overlay
    assert_snapshot(app, name="st_navigation-mobile_sidebar_overlay_page2")

    # Test with sections enabled on mobile
    # First collapse sidebar to access the checkbox
    close_button = app.get_by_test_id("stSidebarCollapseButton")
    close_button.click()
    expect(sidebar).to_have_attribute("aria-expanded", "false")

    # Now click the Test Sections checkbox
    click_checkbox(app, "Test Sections")
    wait_for_app_run(app)

    # Expand sidebar to see sections
    expand_button = app.get_by_test_id("stExpandSidebarButton")
    expand_button.click()
    expect(sidebar).to_have_attribute("aria-expanded", "true")

    # Verify sections are rendered in sidebar on mobile
    section_a = app.get_by_text("Section A").first
    expect(section_a).to_be_visible()

    # Take screenshot of sections in mobile sidebar
    assert_snapshot(sidebar, name="st_navigation-mobile_sidebar_sections")


# ===== TOP PADDING VISUAL REGRESSION TESTS =====
# These tests validate the top padding logic in frontend/app/src/components/AppView/styled-components.ts
# which sets different top padding values based on embedded mode, toolbar visibility, and navigation presence:
# - 2.25rem: embedded minimal (no header, no toolbar)
# - 4.5rem: embedded with header but no toolbar
# - 6rem: non-embedded default OR embedded with show_toolbar
# - 8rem: non-embedded with top navigation present


def test_top_padding_visual_regression_embedded_modes(
    app: Page, assert_snapshot: ImageCompareFunction
):
    """Visual regression test for top padding in different embedded modes.

    Tests the top padding logic from styled-components.ts:
    - 2.25rem: embedded minimal (no header, no toolbar)
    - 4.5rem: embedded with header but no toolbar
    - 6rem: embedded with show_toolbar
    """
    app.set_viewport_size({"width": 1280, "height": 800})
    wait_for_app_run(app)

    # Get the current URL for embedding tests
    current_url = app.url
    base_url = current_url.split("?")[0]

    # Test 1: Embedded minimal mode (2.25rem) - enable hidden nav first to remove headers
    click_checkbox(app, "Test Hidden Navigation")
    wait_for_app_run(app)

    goto_app(app, f"{base_url}?embed=true")
    wait_for_app_run(app)

    # Should have minimal UI - this triggers the 2.25rem padding case
    main_content = app.get_by_test_id("stMain")
    expect(main_content).to_be_visible()
    assert_snapshot(main_content, name="st_app_top_padding-embedded_minimal_2_25rem")

    # Test 2: Embedded with toolbar (6rem)
    goto_app(app, f"{base_url}?embed=true&show_toolbar=true")
    wait_for_app_run(app)

    # Should show toolbar, triggering 6rem padding
    assert_snapshot(main_content, name="st_app_top_padding-embedded_with_toolbar_6rem")

    # Test 3: Go back to normal mode and test embedded with header (4.5rem)
    goto_app(app, base_url)
    wait_for_app_run(app)

    # Uncheck hidden navigation to enable normal page headers
    click_checkbox(app, "Test Hidden Navigation")  # This unchecks it
    wait_for_app_run(app)

    goto_app(app, f"{base_url}?embed=true")
    wait_for_app_run(app)

    # Should have headers but no toolbar, triggering 4.5rem padding
    assert_snapshot(main_content, name="st_app_top_padding-embedded_with_header_4_5rem")


def test_top_padding_visual_regression_non_embedded_modes(
    app: Page, assert_snapshot: ImageCompareFunction
):
    """Visual regression test for top padding in non-embedded modes.

    Tests the top padding logic from styled-components.ts:
    - 6rem: non-embedded default (no top nav)
    - 8rem: non-embedded with top nav (when navigation is present)
    """
    app.set_viewport_size({"width": 1280, "height": 800})
    wait_for_app_run(app)

    main_content = app.get_by_test_id("stMain")
    expect(main_content).to_be_visible()

    # Test 1: Non-embedded default (6rem) - enable hidden navigation to remove top nav
    click_checkbox(app, "Test Hidden Navigation")
    wait_for_app_run(app)

    # Should have 6rem padding when no top nav is present
    assert_snapshot(main_content, name="st_app_top_padding-non_embedded_default_6rem")

    # Test 2: Non-embedded with top nav (8rem) - disable hidden navigation
    click_checkbox(app, "Test Hidden Navigation")  # This unchecks it
    wait_for_app_run(app)

    # Should have 8rem padding when top nav is present (if navigation works)
    # Note: This test may capture the state regardless of whether nav links are visible
    assert_snapshot(
        main_content, name="st_app_top_padding-non_embedded_with_top_nav_8rem"
    )


def test_top_padding_mobile_responsive_behavior(
    app: Page, assert_snapshot: ImageCompareFunction
):
    """Test top padding behavior on mobile viewports."""
    # Mobile view
    app.set_viewport_size({"width": 375, "height": 667})
    wait_for_app_run(app)

    main_content = app.get_by_test_id("stMain")
    expect(main_content).to_be_visible()

    # Mobile should have different top padding since navigation behavior changes
    assert_snapshot(main_content, name="st_app_top_padding-mobile_responsive")

    # Test embedded mobile as well
    current_url = app.url
    base_url = current_url.split("?")[0]
    goto_app(app, f"{base_url}?embed=true")
    wait_for_app_run(app)

    assert_snapshot(main_content, name="st_app_top_padding-mobile_embedded")
