# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from playwright.sync_api import Page, expect

from e2e_playwright.conftest import ImageCompareFunction
from e2e_playwright.shared.app_utils import check_top_level_class


def test_alerts_rendering_themed(
    themed_app: Page, assert_snapshot: ImageCompareFunction
):
    """Test that alerts render correctly with theme-dependent styling."""
    alert_elements = themed_app.get_by_test_id("stAlert")
    expect(alert_elements).to_have_count(32)

    # The first 4 alerts are super basic, no need to screenshot test those
    expect(alert_elements.nth(0)).to_have_text("This is an error")
    expect(alert_elements.nth(1)).to_have_text("This is a warning")
    expect(alert_elements.nth(2)).to_have_text("This is an info message")
    expect(alert_elements.nth(3)).to_have_text("This is a success message")

    # Alert icons (colors differ by theme)
    assert_snapshot(alert_elements.nth(4), name="st_alert-error_icon")
    assert_snapshot(alert_elements.nth(5), name="st_alert-warning_icon")
    assert_snapshot(alert_elements.nth(6), name="st_alert-info_icon")
    assert_snapshot(alert_elements.nth(7), name="st_alert-success_icon")

    # Markdown alerts (markdown colors differ by theme)
    assert_snapshot(alert_elements.nth(10), name="st_alert-error_markdown")
    assert_snapshot(alert_elements.nth(11), name="st_alert-warning_markdown")
    assert_snapshot(alert_elements.nth(12), name="st_alert-info_markdown")
    assert_snapshot(alert_elements.nth(13), name="st_alert-success_markdown")

    # Custom icons (icon rendering may differ by theme)
    assert_snapshot(alert_elements.nth(16), name="st_alert-error_non_emoji_icon")
    assert_snapshot(alert_elements.nth(17), name="st_alert-warning_non_emoji_icon")
    assert_snapshot(alert_elements.nth(18), name="st_alert-info_non_emoji_icon")
    assert_snapshot(alert_elements.nth(19), name="st_alert-success_non_emoji_icon")

    # Alert with heading (heading colors differ by theme)
    assert_snapshot(alert_elements.nth(20), name="st_alert-error_with_heading")


def test_alerts_rendering_layout(app: Page, assert_snapshot: ImageCompareFunction):
    """Test that alerts layout variations render correctly (theme-independent)."""
    alert_elements = app.get_by_test_id("stAlert")
    expect(alert_elements).to_have_count(32)

    # Line wrapping (layout behavior)
    assert_snapshot(alert_elements.nth(8), name="st_alert-error_line_wrapping_1")
    assert_snapshot(alert_elements.nth(9), name="st_alert-error_line_wrapping_2")

    # Long code blocks (layout/overflow behavior)
    assert_snapshot(alert_elements.nth(14), name="st_alert-error_long_code")
    assert_snapshot(alert_elements.nth(15), name="st_alert-success_long_code")

    # Width="stretch" alerts (layout property)
    assert_snapshot(alert_elements.nth(22), name="st_alert-error_width_stretch")
    assert_snapshot(alert_elements.nth(23), name="st_alert-warning_width_stretch")
    assert_snapshot(alert_elements.nth(24), name="st_alert-info_width_stretch")
    assert_snapshot(alert_elements.nth(25), name="st_alert-success_width_stretch")

    # Width=200 alerts (layout property)
    assert_snapshot(alert_elements.nth(26), name="st_alert-error_width_200")
    assert_snapshot(alert_elements.nth(27), name="st_alert-warning_width_200")
    assert_snapshot(alert_elements.nth(28), name="st_alert-info_width_200")
    assert_snapshot(alert_elements.nth(29), name="st_alert-success_width_200")

    # Width with icon (layout property)
    assert_snapshot(alert_elements.nth(30), name="st_alert-error_width_stretch_icon")
    assert_snapshot(alert_elements.nth(31), name="st_alert-info_width_200_icon")


def test_check_top_level_class(app: Page):
    """Check that the top level class is correctly set."""
    check_top_level_class(app, "stAlert")


def test_material_symbol_from_latest_font_version_rendering(
    app: Page, assert_snapshot: ImageCompareFunction
):
    """Test that icon from latest version material symbols font renders correctly."""
    alert_elements = app.get_by_test_id("stAlert")
    expect(alert_elements).to_have_count(32)

    assert_snapshot(alert_elements.nth(21), name="st_alert-latest_material_symbol")
