# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import datetime
import random

import numpy as np
import pandas as pd

import streamlit as st

np.random.seed(0)
random.seed(0)

st.set_page_config(layout="wide")

# Generate a random dataframe
df = pd.DataFrame(
    np.random.randn(5, 5),
    columns=[f"col_{i}" for i in range(5)],
)


st.header(":material/visibility_off: Hide index parameter:")
st.dataframe(df, hide_index=True, width="content")
st.dataframe(df, hide_index=False, width="content")

st.header("Column order parameter:")
column_order = ["col_4", "col_3", "col_0"]
if st.button("Change column order"):
    column_order = ["col_0", "col_3", "col_4"]

st.dataframe(df, column_order=column_order, width="content")

st.header("Set column labels:")
st.dataframe(
    df,
    column_config={
        "_index": "Index column",
        "col_0": "Column 0",
        "col_2": st.column_config.Column("Column 1"),
    },
)

st.header("Hide columns:")
st.dataframe(
    df,
    column_config={"col_1": None, "col_3": {"hidden": True}},
    width="content",
)

st.header("Set column width:")
st.dataframe(
    df,
    column_config={
        "col_0": st.column_config.Column(width="small"),
        "col_1": st.column_config.Column(width="medium"),
        "col_4": {"width": "large"},
    },
    width="content",
)

st.header("Set help tooltips:")
st.caption("Hover over the column headers to see the tooltips.")
st.dataframe(
    pd.DataFrame(
        {
            "col_0": ["a", "b", "c", None],
        }
    ),
    column_config={
        "col_0": st.column_config.Column(help="This :red[is] a **tooltip** 🌟"),
        "_index": {"help": "Index tooltip!"},
    },
    width="content",
)


st.header("Ignore editing-only config options:")
st.dataframe(
    pd.DataFrame(
        {
            "col_0": ["a", "b", "c", None],
        }
    ),
    column_config={"col_0": st.column_config.Column(disabled=False, required=True)},
    width="content",
)


st.header("Text column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": ["Hello World", "{'foo': 'bar', 'baz': 123}", "", None],
            "col_1": [1, 2, 3, None],
        }
    ),
    column_config={
        "col_0": st.column_config.TextColumn(
            "Text column",
            width="medium",
            help="This is a text column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default="invalid",  # Should be ignored
            max_chars=5,  # Should be ignored
            validate="^[0-9]+$",  # Should be ignored
        ),
        "col_1": st.column_config.TextColumn(),
    },
    width="content",
    hide_index=True,
)

st.header("Number column:")
st.dataframe(
    pd.DataFrame(
        {
            "col_0": [1, 2, 3, None],
            "col_1": ["1", "2", "invalid", None],
        }
    ),
    column_config={
        "col_0": st.column_config.NumberColumn(
            "Number column",
            width=200,
            help="This is a number column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default=0,  # Should be ignored
            min_value=5,  # Should be ignored
            max_value=10,  # Should be ignored
            step=0.001,
        ),
        "col_1": st.column_config.NumberColumn(
            format="%.2f%%",
        ),
    },
    width="content",
    hide_index=True,
)

st.header("Checkbox column:")
st.dataframe(
    pd.DataFrame(
        {
            "col_0": [True, False, False, None],
            "col_1": ["yes", "no", "invalid", None],
        }
    ),
    column_config={
        "col_0": st.column_config.CheckboxColumn(
            "Checkbox column",
            width="medium",
            help="This is a checkbox column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default=True,  # Should be ignored
        ),
        "col_1": st.column_config.CheckboxColumn(),
    },
    width="content",
    hide_index=True,
)

st.header("Selectbox column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [1, 2, 3, None],
            "col_1": ["a", "b", "c", None],
        }
    ),
    column_config={
        "col_0": st.column_config.SelectboxColumn(
            "Selectbox column",
            width="medium",
            help="This is a selectbox column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default=True,  # Should be ignored
            options=[1, 2, 3, 4, 5],
            format_func=lambda x: f"Option {x}",
        ),
        "col_1": st.column_config.SelectboxColumn(options=["a", "b", "c", "d"]),
    },
    width="content",
    hide_index=True,
)

st.header("Link column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [
                "https://streamlit.io/",
                "https://docs.streamlit.io/",
                "https://streamlit.io/gallery",
                None,
            ],
            "col_1": ["/a", "/b", "", None],
            "col_2": [
                "https://roadmap.streamlit.app",
                "https://extras.streamlit.app",
                "",
                None,
            ],
            "col_3": [
                "https://roadmap.streamlit.app",
                "https://extras.streamlit.app",
                "",
                None,
            ],
            "col_4": [
                "https://roadmap.streamlit.app",
                "https://extras.streamlit.app",
                "",
                None,
            ],
        }
    ),
    column_config={
        "col_0": st.column_config.LinkColumn(
            "Link column",
            width="medium",
            help="This is a link column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default="https://streamlit.io/",  # Should be ignored
            max_chars=5,  # Should be ignored
            validate="^[0-9]+$",  # Should be ignored
        ),
        "col_1": st.column_config.LinkColumn(),
        "col_2": st.column_config.LinkColumn(
            "Display text via Regex",
            display_text=r"https://(.*?)\.streamlit\.app",
        ),
        "col_3": st.column_config.LinkColumn(
            "Static display text",
            display_text="Open link",
        ),
        "col_4": st.column_config.LinkColumn(
            "Static display icon",
            display_text=":material/open_in_new:",
        ),
    },
    width="content",
    hide_index=True,
)

st.header("Datetime column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [
                datetime.datetime(2021, 1, 1, 1, 0, 0, 123000),
                datetime.datetime(2022, 1, 2, 2, 0, 0, 234000),
                datetime.datetime(2023, 1, 3, 3, 0, 0, 345000),
                None,
            ],
            "col_1": [
                "2021-01-01T01:00:00.123",
                "2022-01-02T02:00:00.234",
                "invalid",
                None,
            ],
            "col_2": [
                datetime.date(2021, 1, 1),
                datetime.date(2022, 1, 2),
                datetime.date(2023, 1, 3),
                None,
            ],
        }
    ),
    column_config={
        "col_0": st.column_config.DatetimeColumn(
            "Datetime column",
            width="medium",
            help="This is a datetime column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default=datetime.datetime(2021, 1, 1, 1, 0, 0),  # Should be ignored
            min_value=datetime.datetime(2021, 1, 1, 1, 0, 0),  # Should be ignored
            max_value=datetime.datetime(2022, 1, 1, 1, 0, 0),  # Should be ignored
            step=0.01,
            format="YYYY-MM-DD HH:mm:ss.SSS",
        ),
        "col_1": st.column_config.DatetimeColumn(
            step=0.01,
        ),
        "col_2": st.column_config.DatetimeColumn(),
    },
    width="content",
    hide_index=True,
)

st.header("Date column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [
                datetime.date(2021, 1, 1),
                datetime.date(2022, 1, 2),
                datetime.date(2023, 1, 3),
                None,
            ],
            "col_1": [
                "2021-01-01T01:00:00",
                "2022-01-02T02:00:00",
                "invalid",
                None,
            ],
            "col_2": [
                datetime.datetime(2021, 1, 1, 1, 0, 0, 123000),
                datetime.datetime(2022, 1, 2, 2, 0, 0, 234000),
                datetime.datetime(2023, 1, 3, 3, 0, 0, 345000),
                None,
            ],
        }
    ),
    column_config={
        "col_0": st.column_config.DateColumn(
            "Date column",
            width="medium",
            help="This is a date column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default=datetime.date(2021, 1, 1),  # Should be ignored
            min_value=datetime.date(2021, 1, 1),  # Should be ignored
            max_value=datetime.date(2022, 1, 1),  # Should be ignored
            step=2,  # Should be ignored
        ),
        "col_1": st.column_config.DateColumn(),
        "col_2": st.column_config.DateColumn(),
    },
    width="content",
    hide_index=True,
)

st.header("Time column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [
                datetime.time(1, 2, 0, 123000),
                datetime.time(2, 3, 0, 234000),
                datetime.time(3, 4, 0, 345000),
                None,
            ],
            "col_1": [
                "2021-01-01T01:02:00",
                "2022-01-02T02:03:00",
                "invalid",
                None,
            ],
            "col_2": [
                datetime.datetime(2021, 1, 1, 1, 0, 0, 123000),
                datetime.datetime(2022, 1, 2, 2, 0, 0, 234000),
                datetime.datetime(2023, 1, 3, 3, 0, 0, 345000),
                None,
            ],
        }
    ),
    column_config={
        "col_0": st.column_config.TimeColumn(
            "Time column",
            width="medium",
            help="This is a time column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default=datetime.time(1, 2, 0),  # Should be ignored
            min_value=datetime.time(1, 2, 0),  # Should be ignored
            max_value=datetime.time(1, 3, 0),  # Should be ignored
            step=datetime.timedelta(milliseconds=1),
        ),
        "col_1": st.column_config.TimeColumn(
            format="HH:mm",
        ),
        "col_2": st.column_config.TimeColumn(),
    },
    width="content",
    hide_index=True,
)

st.header("Progress column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [0.1, 0.4872, 10.1, None],
            "col_1": ["200", "550", "1000", None],
            "col_2": [0.1, 0.4872, 1.1, None],
        }
    ),
    column_config={
        "col_0": st.column_config.ProgressColumn(
            "Progress column",
            width="medium",
            help="This is a progress column",
        ),
        "col_1": st.column_config.ProgressColumn(
            format="$%f", min_value=0, max_value=1000, color="blue"
        ),
        "col_2": st.column_config.ProgressColumn(step=0.0001, color="auto"),
    },
    width="content",
    hide_index=True,
)

st.header("List column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [[1, 2], [2, 3, 4], [], None],
            "col_1": [
                [
                    "Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo Foo",
                    "Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar Bar",
                ],
                [],
                [],
                None,
            ],
            "col_2": ["a,b", "c,d,e", "", None],
        }
    ),
    column_config={
        "col_0": st.column_config.ListColumn(
            "List column",
            width="medium",
            help="This is a list column",
        ),
        "col_1": st.column_config.ListColumn(width="medium"),
        "col_2": st.column_config.ListColumn(),
    },
    width="content",
    hide_index=True,
)

st.header("Bar chart column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [[1, 5, 2], [2, 3, 5, -4, -5], [], None],
            "col_1": ["1,2,3,4", "6, 5, 1, 10", "invalid", None],
        }
    ),
    column_config={
        "col_0": st.column_config.BarChartColumn(
            "Bar chart column",
            width="medium",
            help="This is a bar chart column",
            y_min=-5,
            y_max=5,
            color="auto",
        ),
        "col_1": st.column_config.BarChartColumn(),
    },
    width="content",
    hide_index=True,
)


st.header("Line chart column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [[1, 5, 2], [2, 3, 5, -4, -5], [], None],
            "col_1": ["1,2,3,4", "6, 5, 1, 10", "invalid", None],
        }
    ),
    column_config={
        "col_0": st.column_config.LineChartColumn(
            "Line chart column",
            width="medium",
            help="This is a line chart column",
            y_min=-5,
            y_max=5,
            color="auto",
        ),
        "col_1": st.column_config.LineChartColumn(),
    },
    width="content",
    hide_index=True,
)

st.header("Area chart column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [[1, 5, 2], [2, 3, 5, -4, -5], [], None],
            "col_1": ["1,2,3,4", "6, 5, 1, 10", "invalid", None],
        }
    ),
    column_config={
        "col_0": st.column_config.AreaChartColumn(
            "Area chart column",
            width="medium",
            help="This is an area chart column",
            y_min=-5,
            y_max=5,
            color="auto",
        ),
        "col_1": st.column_config.AreaChartColumn(),
    },
    width="content",
    hide_index=True,
)

st.header("Chart column colors:")

st.dataframe(
    pd.DataFrame(
        {
            "red": [[1, 5, 2, 6], [2, 3, 5, 1]],
            "blue": [[1, 5, 2, 6], [2, 3, 5, 1]],
            "orange": [[1, 5, 2, 6], [2, 3, 5, 1]],
            "violet": [[1, 5, 2, 6], [2, 3, 5, 1]],
            "gray": [[1, 5, 2, 6], [2, 3, 5, 1]],
            "auto": [[1, 5, 2, 6], [6, 2, 5, 1]],  # up and down trend
            "auto-inverse": [[1, 5, 2, 6], [6, 2, 5, 1]],  # up and down trend
        }
    ),
    column_config={
        "red": st.column_config.BarChartColumn("Red", color="red"),
        "blue": st.column_config.AreaChartColumn("Blue", color="blue"),
        "orange": st.column_config.BarChartColumn("Orange", color="orange"),
        "violet": st.column_config.LineChartColumn("Violet", color="violet"),
        "gray": st.column_config.AreaChartColumn("Gray", color="gray"),
        "auto": st.column_config.AreaChartColumn("Auto", color="auto"),
        "auto-inverse": st.column_config.AreaChartColumn(
            "Auto-inverse", color="auto-inverse"
        ),
    },
    width="content",
    hide_index=True,
)


st.header("Image column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [
                "https://streamlit.io/images/brand/streamlit-mark-color.png",
                "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAJYAAABSCAMAAACBpt1yAAAAwFBMVEVHcEyAhJWAhJUzNT97f4+AhJWAhJVtcH9WWWaAhJWAhJWAhJVCRFGAhJWAhJWAhJVVWGeAhJWAhJWAhJVVWGcmJzCAhJWAhJWAhJUmJzAmJzBVWGcmJzAmJzAmJzAmJzAmJzCAhJUmJzBVWGcmJzAmJzAmJzBVWGeAhJVVWGdVWGdVWGeAhJVVWGdVWGdVWGdvcoJVJ2WAhJVVWGcmJzBTVmVaXWx8f5B1eYlhZHNoa3tDRVI3OUQtLzhucoJMT13kXsyQAAAAMXRSTlMA+SYWE/ExBQvmfME1P7Uc9lhpjSdsTZ7cnETlsyP6xvHNg7XXV+bYq6NjTtR1yJLh8/IzCQAABHhJREFUaN7Nmut2mkAYRZGLIMQbRMFbvMQYkmjS9YFJU23z/m9VwEpgGJgZYLo4/2ratXbdhyNMFIS68jzzXm+EpuVl5nne623DqKRXL8yb2iyst4jKG7w0S+HgguU93jZPYZin5mhU32Iqb/DcPIWN0phQGGnUGqewQRpTChuj8fbVQ1ODRnVsDeUaFVbXuBhtpsJYgZYxLP3fUzMKK2qURlvf3wkWBGn1xiVX8PbRw6WsRmm09MOE71YYxe2qNSmMUkqj9rC8i6hWgmzAJYrdVqtfhRU0ag/zC5TvzwVBh2s6drsehWHuGTVq0xjKXz0IgtiBbzBLZFL45OWHSaM6naz8OEspeMWGRBydAex5UIA1Y7iDvklC+f4ofG3cSnLBXu9XV8ik8WayS0L5u0X4qryGdNYm1b5qT4VU3uBAt56bNJTvTy6Xng5o1jTDX6iQVuNis0Wg/Lvp5UftToaLYvgJCqk0RpOOJix8dBm4kA1p+EkKoxyoJh3J5vrzYQvDRRh+okKSRuk66Uh28b/JlD4e/m4FhZFGmTzp+MLjS08YfiqF+RqTk44W/uH7r2FKXzz8VArzNKrT+crPy7XwuaWPwbLDfzPzaJPRqCKTnlf4/NJ/fyKlh1+79+hzKJ703MKH6e+hMPvU8B8YqNIas5OOZJ6eOgsISXwiMShMa8RMekHhw3QVElc8/EwKExqxk45kKyFF7AEx/4b/wEjlzV4ooZDChzGBIsHwsyqMNEpazqQjWU1RrL4DVGD2vXdk55rTQGUKT1X6a/58MmN9namo7kbZ7SWX/pqPP+/MXL9psLYLzOeUAdT5+MUIdjzRvFkT3OenCcAP7HguU/gwosPCBT9/vTN0/0jWeLfE38zaABzByBpH+Fsz+tLHYJ/Ua3Ekatwt8FiyAcARjKRxkneLbgJwBCNcjfjClyh9PGOfdVyNSyn3acQGKAf2Xl3jKP8hadwCrmAn2tvS6qVnASvQOCl6WNahfGiGP1cjeluKlL5TgYtmX0/shRfQMzgOYHkaN8VnKKVLTztjeI07wplT3nFEfWBYjXPSIbcOwBnszFp4wnFETcOP0biViEfRLgBnsBNr4cnHEbXsK6pxRXHIWkPpSfuKapzTHJXrALzBToyFD8vVVoA72Jn0HMat9IXDn9K4EahSV+mLZuyL5rYUOY7YA/AGS3w2zml/N2QBcAc7/qa4La36ZEa3r0fc1UhX+Og4ogfABwxzNU7of+1oAvABe89opC08wxlcxRn7It+W8i89BuzMUnhepU+AHWONuwULlmYAx8TD/1X8HPbfSo/M2PE0ZaIqeRzBDMb6JQ7BbAF/sBnzt41VXeHN9fGjxJdn1bHBF0xx20KZyF3d7hn7jlKvz5bi7A3X1rsVvqqnarLYHg9N3XIDQqejlGJsKUrHWRs929LNYbctylpt3+VVVVnui+1uwBhAWrbb6xnGer13HKeTSvCCs1+vDaPnurYVgJjDcYDSr5GlEDN4J+V+EFEU23GCP4jBa7IcUKilOf4CgQRuuzC9EJcAAAAASUVORK5CYII=",
                "data:image/svg+xml,%3Csvg width='301' height='165' viewBox='0 0 301 165' fill='none' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M150.731 101.547L98.1387 73.7471L6.84674 25.4969C6.7634 25.4136 6.59674 25.4136 6.51341 25.4136C3.18007 23.8303 -0.236608 27.1636 1.0134 30.497L47.5302 149.139L47.5385 149.164C47.5885 149.281 47.6302 149.397 47.6802 149.514C49.5885 153.939 53.7552 156.672 58.2886 157.747C58.6719 157.831 58.9461 157.906 59.4064 157.998C59.8645 158.1 60.5052 158.239 61.0552 158.281C61.1469 158.289 61.2302 158.289 61.3219 158.297H61.3886C61.4552 158.306 61.5219 158.306 61.5886 158.314H61.6802C61.7386 158.322 61.8052 158.322 61.8636 158.322H61.9719C62.0386 158.331 62.1052 158.331 62.1719 158.331V158.331C121.084 164.754 180.519 164.754 239.431 158.331V158.331C240.139 158.331 240.831 158.297 241.497 158.231C241.714 158.206 241.922 158.181 242.131 158.156C242.156 158.147 242.189 158.147 242.214 158.139C242.356 158.122 242.497 158.097 242.639 158.072C242.847 158.047 243.056 158.006 243.264 157.964C243.681 157.872 243.87 157.806 244.436 157.611C245.001 157.417 245.94 157.077 246.527 156.794C247.115 156.511 247.522 156.239 248.014 155.931C248.622 155.547 249.201 155.155 249.788 154.715C250.041 154.521 250.214 154.397 250.397 154.222L250.297 154.164L150.731 101.547Z' fill='%23FF4B4B'/%3E%3Cpath d='M294.766 25.4981H294.683L203.357 73.7483L254.124 149.357L300.524 30.4981V30.3315C301.691 26.8314 298.108 23.6648 294.766 25.4981' fill='%237D353B'/%3E%3Cpath d='M155.598 2.55572C153.264 -0.852624 148.181 -0.852624 145.931 2.55572L98.1389 73.7477L150.731 101.548L250.398 154.222C251.024 153.609 251.526 153.012 252.056 152.381C252.806 151.456 253.506 150.465 254.123 149.356L203.356 73.7477L155.598 2.55572Z' fill='%23BD4043'/%3E%3C/svg%3E%0A",  # noqa: E501
                "",
                None,
            ],
        }
    ),
    column_config={
        "col_0": st.column_config.ImageColumn(
            "Image column",
            width="medium",
            help="This is a image column",
        ),
    },
    width="content",
    hide_index=True,
)

st.subheader("Long column header")
st.dataframe(
    pd.DataFrame(
        np.random.randn(100, 15),
        columns=[
            "this is a very long column header name",
            "A",
            "Short header",
            "B",
            "this is another very very column long header name",
            "C",
            (
                "this is another very very very very very very very very very very very"
                " very very very very long header name"
            ),
            "D",
            "E",
            "F",
            "G",
            "H",
            "I",
            "J",
            "K",
        ],
    ),
    width="content",
)

st.subheader("Hierarchical headers")

st.dataframe(
    pd.DataFrame(
        np.random.randn(3, 6),
        index=["A", "B", "C"],
        columns=pd.MultiIndex.from_tuples(
            [
                ("a", "b", "c"),
                ("a", "b", "d"),
                ("e", "f", "c"),
                ("g", "h", "d (Test)"),
                ("", "h", "i"),
                ("j", "", ""),
            ],
            names=["first", "second", "third"],
        ),
    ),
    width="content",
)

df = pd.DataFrame(
    np.random.randn(15, 25),
    columns=[f"col_{i}" for i in range(25)],
)

st.header("Pinned columns:")
st.dataframe(
    df,
    column_config={
        "_index": st.column_config.Column(pinned=False),
        "col_2": st.column_config.Column(pinned=True),
        "col_4": st.column_config.Column(pinned=True),
        "col_16": st.column_config.Column(pinned=True),
    },
    # Use reversed column order to test that pinned columns
    # use the column order as well.
    column_order=reversed(df.columns.tolist()),
    width="content",
)

st.header("Configurable row height:")

streamlit_logo = "data:image/svg+xml,%3Csvg width='301' height='165' viewBox='0 0 301 165' fill='none' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M150.731 101.547L98.1387 73.7471L6.84674 25.4969C6.7634 25.4136 6.59674 25.4136 6.51341 25.4136C3.18007 23.8303 -0.236608 27.1636 1.0134 30.497L47.5302 149.139L47.5385 149.164C47.5885 149.281 47.6302 149.397 47.6802 149.514C49.5885 153.939 53.7552 156.672 58.2886 157.747C58.6719 157.831 58.9461 157.906 59.4064 157.998C59.8645 158.1 60.5052 158.239 61.0552 158.281C61.1469 158.289 61.2302 158.289 61.3219 158.297H61.3886C61.4552 158.306 61.5219 158.306 61.5886 158.314H61.6802C61.7386 158.322 61.8052 158.322 61.8636 158.322H61.9719C62.0386 158.331 62.1052 158.331 62.1719 158.331V158.331C121.084 164.754 180.519 164.754 239.431 158.331V158.331C240.139 158.331 240.831 158.297 241.497 158.231C241.714 158.206 241.922 158.181 242.131 158.156C242.156 158.147 242.189 158.147 242.214 158.139C242.356 158.122 242.497 158.097 242.639 158.072C242.847 158.047 243.056 158.006 243.264 157.964C243.681 157.872 243.87 157.806 244.436 157.611C245.001 157.417 245.94 157.077 246.527 156.794C247.115 156.511 247.522 156.239 248.014 155.931C248.622 155.547 249.201 155.155 249.788 154.715C250.041 154.521 250.214 154.397 250.397 154.222L250.297 154.164L150.731 101.547Z' fill='%23FF4B4B'/%3E%3Cpath d='M294.766 25.4981H294.683L203.357 73.7483L254.124 149.357L300.524 30.4981V30.3315C301.691 26.8314 298.108 23.6648 294.766 25.4981' fill='%237D353B'/%3E%3Cpath d='M155.598 2.55572C153.264 -0.852624 148.181 -0.852624 145.931 2.55572L98.1389 73.7477L150.731 101.548L250.398 154.222C251.024 153.609 251.526 153.012 252.056 152.381C252.806 151.456 253.506 150.465 254.123 149.356L203.356 73.7477L155.598 2.55572Z' fill='%23BD4043'/%3E%3C/svg%3E%0A"  # noqa: E501
st.dataframe(
    pd.DataFrame(
        {
            "col_0": [
                (
                    "this is a very long sentence that does not contain any reasonable content.this is a "
                    "very long sentence that does not contain any reasonable content."
                ),
                "Hello World",
            ],
            "col_1": [
                streamlit_logo,
                streamlit_logo,
            ],
        },
    ),
    column_config={
        "col_0": st.column_config.TextColumn("Text", width=400),
        "col_1": st.column_config.ImageColumn("Logo", width="medium"),
    },
    row_height=100,
    width="content",
)

st.header("NumberColumn Formatting:")

st.dataframe(
    pd.DataFrame(
        {
            "default": [0.0123123, -1234.567, 12, 0],
            "percent": [0.0123123, -1234.567, 12, 0],
            "compact": [0.0123123, -1234.567, 12, 0],
            "scientific": [0.0123123, -1234.567, 12, 0],
            "engineering": [0.0123123, -1234.567, 12, 0],
            "plain": [0.0123123, -1234.567, 12, 0],
            "dollar": [0.0123123, -1234.567, 12, 0],
            "euro": [0.0123123, -1234.567, 12, 0],
            "yen": [0.0123123, -1234.567, 12, 0],
            "localized": [0.0123123, -1234.567, 12, 0],
            "accounting": [0.0123123, -1234.567, 12, 0],
            "bytes": [0.0123123, -1234.567, 12, 0],
            "custom format": [0.0123123, -1234.567, 12, 0],
        }
    ),
    column_config={
        "percent": st.column_config.NumberColumn(format="percent"),
        "compact": st.column_config.NumberColumn(format="compact"),
        "scientific": st.column_config.NumberColumn(format="scientific"),
        "engineering": st.column_config.NumberColumn(format="engineering"),
        "plain": st.column_config.NumberColumn(format="plain"),
        "dollar": st.column_config.NumberColumn(format="dollar", step=0.1),
        "euro": st.column_config.NumberColumn(format="euro"),
        "yen": st.column_config.NumberColumn(format="yen"),
        "localized": st.column_config.NumberColumn(format="localized"),
        "accounting": st.column_config.NumberColumn(format="accounting"),
        "bytes": st.column_config.NumberColumn(format="bytes"),
        "custom format": st.column_config.NumberColumn(format="%.2f"),
    },
    hide_index=True,
    width="content",
)

st.header("Date Time Formatting:")

st.dataframe(
    pd.DataFrame(
        {
            "default": [
                datetime.datetime(2022, 4, 6, 9, 30, 0),
                datetime.datetime(2024, 1, 1, 15, 45, 30),
                datetime.datetime(2019, 8, 9, 12, 0, 0),
            ],
            "localized": [
                datetime.datetime(2022, 4, 6, 9, 30, 0),
                datetime.datetime(2024, 1, 1, 15, 45, 30),
                datetime.datetime(2019, 8, 9, 12, 0, 0),
            ],
            "calendar": [
                datetime.datetime(2022, 4, 6, 9, 30, 0),
                datetime.datetime(2024, 1, 1, 15, 45, 30),
                datetime.datetime(2019, 8, 9, 12, 0, 0),
            ],
            "iso8601": [
                datetime.datetime(2022, 4, 6, 9, 30, 0),
                datetime.datetime(2024, 1, 1, 15, 45, 30),
                datetime.datetime(2019, 8, 9, 12, 0, 0),
            ],
            "localized date": [
                datetime.date(2022, 4, 6),
                datetime.date(2024, 1, 1),
                datetime.date(2019, 8, 9),
            ],
            "localized time": [
                datetime.time(9, 30, 0),
                datetime.time(15, 45, 30),
                datetime.time(12, 0, 0),
            ],
            "custom format": [
                datetime.datetime(2022, 4, 6, 9, 30, 0),
                datetime.datetime(2024, 1, 1, 15, 45, 30),
                datetime.datetime(2019, 8, 9, 12, 0, 0),
            ],
        }
    ),
    column_config={
        "localized": st.column_config.DatetimeColumn(format="localized"),
        "calendar": st.column_config.DatetimeColumn(format="calendar"),
        "custom format": st.column_config.DatetimeColumn(
            format="MMM DD, YYYY - hh:mm A"
        ),
        "localized date": st.column_config.DateColumn(format="localized"),
        "localized time": st.column_config.TimeColumn(format="localized"),
        "iso8601": st.column_config.DatetimeColumn(format="iso8601"),
        # We cannot reliably test distance via e2e tests because it wouldn't
        # stay stable:
        # "distance": st.column_config.DatetimeColumn(format="distance"),  # noqa: ERA001
    },
    hide_index=True,
    width="content",
)

st.header("Json column:")

st.dataframe(
    pd.DataFrame(
        {
            "dict": [
                {"name": "test", "value": 123},
                {"name": "test2", "value": 456},
                {},
                None,
            ],
            "string json": [
                '{"name": {"foo": "bar"}, "value": 456}',
                '{"name": "test", "value": 123}',
                "",
                None,
            ],
            "list": [
                ["Foo", "Bar", "Baz"],
                ["Hello", "World"],
                [],
                None,
            ],
            "string list": [
                "[1, 2, 3]",
                "[4, 5]",
                "[]",
                None,
            ],
            "incompatible values": [
                "{hello world}",
                "foo",
                "{ this is no JSON!",
                None,
            ],
        }
    ),
    column_config={
        "dict": st.column_config.JsonColumn(width="medium"),
        # We explicitly don't set the string json column to json
        # to test the behavior that text based columns should auto activate
        # the json renderer.
        "list": st.column_config.JsonColumn(width="medium"),
        "string list": st.column_config.JsonColumn(width="medium"),
        "incompatible values": st.column_config.JsonColumn(width="medium"),
    },
    hide_index=True,
    width="content",
)

st.header("Localized Date/Number Formatting:")

st.dataframe(
    data=pd.DataFrame(
        {
            "Name": ["John", "Jane", "Jim", "Jill"],
            "Percent": [0.5, 0.6, 0.7, 0.8],
            "Salary": [50000, 54000, 58000, 62000],
            "appointment": [
                datetime.datetime(2024, 2, 5, 12, 30),
                datetime.datetime(2023, 11, 10, 18, 0),
                datetime.datetime(2024, 3, 11, 20, 10),
                datetime.datetime(2023, 9, 12, 3, 0),
            ],
        }
    ),
    column_config={
        "Name": st.column_config.TextColumn("Name"),
        "Percent": st.column_config.NumberColumn("Percent", format="localized"),
        "Salary": st.column_config.NumberColumn("Salary", format="localized"),
        "appointment": st.column_config.DatetimeColumn(
            "Appointment",
            min_value=datetime.datetime(2023, 6, 1),
            max_value=datetime.datetime(2025, 1, 1),
            format="localized",
            step=60,
        ),
    },
)

st.header("Multiselect column:")

st.dataframe(
    pd.DataFrame(
        {
            "col_0": [["a", "b"], ["b", "c", "d"], [], None],
            "col_1": ["a,b", "b,c,d", "", None],
            "col_2": [
                ["orange", "banana", "apple"],
                ["pineapple", "mango", "orange", "strawberry"],
                [],
                None,
            ],
        }
    ),
    column_config={
        "col_0": st.column_config.MultiselectColumn(
            "Multiselect column",
            width="medium",
            help="This is a multi-select column",
            required=True,  # Should be ignored
            disabled=False,  # Should be ignored
            default=["a", "b"],  # Should be ignored
            accept_new_options=False,  # Should be ignored
            options=["a", "b", "c", "d", "e"],
            color=[
                "green",
                "blue",
                "red",
                "#19747E",
            ],
        ),
        "col_1": st.column_config.MultiselectColumn(
            width="medium",
            options=["a", "b", "c", "d", "e"],
            color="primary",
            format_func=lambda x: f"Option {x}",
        ),
        "col_2": st.column_config.MultiselectColumn(
            width="medium",
            options=["orange", "banana", "apple", "pineapple", "mango", "strawberry"],
            color="auto",
        ),
    },
    width="content",
    hide_index=True,
)

st.header("Missing placeholder:")
st.dataframe(
    pd.DataFrame(
        {
            "with_none": [1, None, 3],
            "all_missing": [None, None, None],
            "nan": [None, np.nan, 3],
        }
    ),
    placeholder="-",
    width="content",
)
