# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import graphviz

import streamlit as st

# basic graph
hello = graphviz.Digraph("Hello World")
hello.edge("Hello", "World")

# styled graph
styled = graphviz.Graph("G", filename="g_c_n.gv")
styled.attr(bgcolor="purple:pink", label="agraph", fontcolor="white")

with styled.subgraph(name="cluster1") as c:
    c.attr(
        fillcolor="blue:cyan",
        label="acluster",
        fontcolor="white",
        style="filled",
        gradientangle="270",
    )
    c.attr(
        "node", shape="box", fillcolor="red:yellow", style="filled", gradientangle="90"
    )
    c.node("anode")

# complex graph
engine = st.sidebar.radio(
    "Select engine",
    ["dot", "neato", "twopi", "circo", "fdp", "osage", "patchwork"],
)
st.sidebar.write(engine)
finite = graphviz.Digraph("finite_state_machine", filename="fsm.gv", engine=engine)
finite.attr(rankdir="LR", size="8,5")

finite.attr("node", shape="doublecircle")
finite.node("LR_0")
finite.node("LR_3")
finite.node("LR_4")
finite.node("LR_8")

finite.attr("node", shape="circle")
finite.edge("LR_0", "LR_2", label="SS(B)")
finite.edge("LR_0", "LR_1", label="SS(S)")
finite.edge("LR_1", "LR_3", label="S($end)")
finite.edge("LR_2", "LR_6", label="SS(b)")
finite.edge("LR_2", "LR_5", label="SS(a)")
finite.edge("LR_2", "LR_4", label="S(A)")
finite.edge("LR_5", "LR_7", label="S(b)")
finite.edge("LR_5", "LR_5", label="S(a)")
finite.edge("LR_6", "LR_6", label="S(b)")
finite.edge("LR_6", "LR_5", label="S(a)")
finite.edge("LR_7", "LR_8", label="S(b)")
finite.edge("LR_7", "LR_5", label="S(a)")
finite.edge("LR_8", "LR_6", label="S(b)")
finite.edge("LR_8", "LR_5", label="S(a)")

# draw graphs
st.graphviz_chart(hello)

st.graphviz_chart(styled)

st.graphviz_chart(finite)

# draw graphs in columns

left_graph = graphviz.Digraph("Left")
left_graph.edge("Left", "Graph")

right_graph = graphviz.Digraph("Right")
right_graph.edge("Right", "Graph")

col1, col2 = st.columns([1, 1])

with col1:
    st.graphviz_chart(left_graph)

with col2:
    st.graphviz_chart(right_graph)


dot_code = """
digraph Dot {
  A -> {B, C, D} -> {F}
}
"""
st.graphviz_chart(dot_code)


horizontal_flow_chart = """
digraph Dot {
  rankdir=LR;
  A -> B -> C -> D -> {F}
}
"""
st.graphviz_chart(horizontal_flow_chart, use_container_width=True)

st.subheader("Width Parameters")

st.graphviz_chart(dot_code, width="content")
st.graphviz_chart(dot_code, width="stretch")
st.graphviz_chart(dot_code, width=100)

st.subheader("Height Parameters")

st.write("height=content (default)")
st.graphviz_chart(dot_code, height="content")

st.write("height=stretch")
with st.container(border=True, height=400):
    st.graphviz_chart(dot_code, height="stretch")

st.write("height=200")
st.graphviz_chart(dot_code, height=200)

st.subheader("Combined Width and Height")

st.write("width=300, height=150")
st.graphviz_chart(dot_code, width=300, height=150)
